package com.voxeet.sdk.core;

import android.content.Context;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.util.Log;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.voxeet.authent.token.RefreshTokenCallback;
import com.voxeet.log.endpoints.LogEndpoints;
import com.voxeet.log.factory.LogFactory;
import com.voxeet.push.center.NotificationCenterFactory;
import com.voxeet.push.center.management.Constants;
import com.voxeet.sdk.core.preferences.VoxeetPreferences;
import com.voxeet.sdk.core.services.AudioService;
import com.voxeet.sdk.core.services.ChatService;
import com.voxeet.sdk.core.services.CommandService;
import com.voxeet.sdk.core.services.ConferenceService;
import com.voxeet.sdk.core.services.FilePresentationService;
import com.voxeet.sdk.core.services.LocalStatsService;
import com.voxeet.sdk.core.services.MediaDeviceService;
import com.voxeet.sdk.core.services.NotificationService;
import com.voxeet.sdk.core.services.RecordingService;
import com.voxeet.sdk.core.services.ScreenShareService;
import com.voxeet.sdk.core.services.SessionService;
import com.voxeet.sdk.core.services.VideoPresentationService;
import com.voxeet.sdk.core.services.authenticate.WebSocketState;
import com.voxeet.sdk.events.sdk.SocketStateChangeEvent;
import com.voxeet.sdk.exceptions.ExceptionManager;
import com.voxeet.sdk.exceptions.VoxeetSentry;
import com.voxeet.sdk.factories.VoxeetIntentFactory;
import com.voxeet.sdk.json.ConferenceDestroyedPush;
import com.voxeet.sdk.utils.AbstractVoxeetEnvironmentHolder;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.NoDocumentation;
import com.voxeet.sdk.utils.Validate;
import com.voxeet.sdk.utils.converter.JacksonConverterFactory;

import org.greenrobot.eventbus.EventBus;
import org.greenrobot.eventbus.EventBusException;

import java.net.UnknownHostException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import eu.codlab.simplepromise.Promise;
import okhttp3.OkHttpClient;
import retrofit2.Retrofit;

@Annotate
public final class VoxeetSdk {

    private final static String TAG = VoxeetSdk.class.getSimpleName();
    private final static long DEFAULT_TIMEOUT_MS = 60 * 1000;

    private static Context CurrentApplication = null;
    private static VoxeetSdk CurrentInstance;

    private VoxeetHttp _voxeet_http;
    private Context _application_context;
    private EventBus _event_bus;
    private AbstractVoxeetEnvironmentHolder _voxeet_environment_holder;
    private HashMap<Class<? extends AbstractVoxeetService>, AbstractVoxeetService> _services;
    private VoxeetSentry voxeetSentry;

    @NoDocumentation
    protected static void setInstance(@NonNull VoxeetSdk sdk) {
        CurrentInstance = sdk;

        Validate.hasInternetPermissions(sdk.getApplicationContext(), true);
        VoxeetPreferences.init(sdk.getApplicationContext(),
                sdk.getVoxeetEnvironmentHolder());
    }

    @Nullable
    public static VoxeetSdk instance() {
        return CurrentInstance;
    }

    public static void setApplication(@NonNull Context application) {
        CurrentApplication = application;
    }

    /**
     * This function initializes the Voxeet SDK, the behavior of Voxeet SDK functions are
     * undetermined if this function is not called. It should be called as early as possible.
     * <p>
     * Using a null value for the UserInfo? Prefer the signature without it then
     *
     * @param appId    the app id
     * @param password the password
     */
    public static synchronized void initialize(@NonNull String appId,
                                               @NonNull String password) {
        if (null == CurrentInstance) {
            VoxeetSdk sdk = new VoxeetSdk(CurrentApplication,
                    appId,
                    password,
                    null,
                    null,
                    new VoxeetEnvironmentHolder(CurrentApplication),
                    false);

            VoxeetSdk.setInstance(sdk);
        } else {
            Log.d(TAG, "initialize: Instance already started !");
        }
    }

    /**
     * This function initializes the Voxeet SDK, the behavior of Voxeet SDK functions are
     * undetermined if this function is not called. It should be called as early as possible.
     * <p>
     * This method will invoke the SDK with the thirdparty authentication feature
     * <p>
     * Using a null value for the UserInfo? Prefer the signature without it then
     *
     * @param accessToken  the default accessToken to use
     * @param refreshToken the refreshToken callback
     */
    public static synchronized void initialize(@NonNull String accessToken,
                                               @NonNull RefreshTokenCallback refreshToken) {
        if (null == CurrentInstance) {
            VoxeetSdk sdk = new VoxeetSdk(CurrentApplication,
                    null,
                    null,
                    accessToken,
                    refreshToken,
                    new VoxeetEnvironmentHolder(CurrentApplication),
                    false);
            VoxeetSdk.setInstance(sdk);
        } else {
            Log.d(TAG, "initialize: Instance already started !");
        }
    }

    private VoxeetSdk(@NonNull Context application_context,
                      @Nullable String appId,
                      @Nullable String password,
                      @Nullable String tokenAccess,
                      @Nullable RefreshTokenCallback tokenRefresh,
                      @NonNull AbstractVoxeetEnvironmentHolder holder,
                      boolean debug) {
        initExceptionIfOk(application_context);

        //set log factory init
        Retrofit retrofit = new Retrofit.Builder()
                //.addCallAdapterFactory(RxJavaCallAdapterFactory.create())
                .addConverterFactory(JacksonConverterFactory.create(new ObjectMapper().disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES)))
                .baseUrl(LogFactory.ENDPOINT) //TODO refacto
                .build();
        LogEndpoints endpoint = retrofit.create(LogEndpoints.class);
        LogFactory.instance.init(endpoint);

        _application_context = application_context;
        _event_bus = EventBus.getDefault();

        //make sure the looper is the main looper for promise execution
        Promise.setHandler(new Handler(Looper.getMainLooper()));

        Log.d("VoxeetSDK", "VoxeetSdk: version := " + holder.getVersionName());

        _voxeet_environment_holder = holder;

        VoxeetHttp.Builder builder = new VoxeetHttp.Builder()
                .setApplication(application_context)
                .setVoxeetSDK(this)
                .setServerUrl(_voxeet_environment_holder.getServerUrl())
                .setServerPort(_voxeet_environment_holder.getServerPort())
                .setAppId(appId)
                .setPassword(password)
                .setTokenAccess(tokenAccess)
                .setTokenRefresh(tokenRefresh)
                .setVoxeetServiceListener(new VoxeetHttp.VoxeetServiceListener() {

                    @Override
                    public void onIdentifySuccess(VoxeetHttp provider) {
                        session().connectSocket(provider);
                    }

                    @Override
                    public void onIdentifyError(String error) {
                        if (null == error) error = "identfify error";
                        _event_bus.post(new SocketStateChangeEvent(WebSocketState.CLOSED));
                    }

                    @Override
                    public void onNetworkError(UnknownHostException error) {
                        _event_bus.post(new SocketStateChangeEvent(WebSocketState.CLOSED));
                    }
                })
                .setDebug(debug);

        _voxeet_http = builder.build();


        _services = new HashMap<>();

        VoxeetPreferences.init(getApplicationContext(), getVoxeetEnvironmentHolder());
        initServices();
    }

    /**
     * Retrieve the unique instance of the ChatService
     *
     * @return With an initialized SDK, the instance of the ChatService
     */
    @NoDocumentation
    @Nullable
    public static ChatService chat() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(ChatService.class);
    }


    /**
     * Retrieve the unique instance of the ConferenceService.
     * <p>
     * This Service holds the various calls to interact with Conferences in the SDK :
     * - create conferences
     * - join conferences
     * - leave conferences
     * etc...
     *
     * @return With an initialized SDK, the instance of the ConferenceService
     */
    @Nullable
    public static ConferenceService conference() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(ConferenceService.class);
    }

    /**
     * Retrieve the unique instance of the MediaDeviceService
     * <p>
     * This service helps to interact with Devices available through the system :
     * - start video
     * - stop video
     * - get the instance of the Camera Context
     * - etc...
     *
     * @return With an initialized SDK, the instance of the MediaDeviceService
     */
    @Nullable
    public static MediaDeviceService mediaDevice() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(MediaDeviceService.class);
    }

    /**
     * Retrieve the unique instance of the ScreenShareService
     * <p>
     * In conjunction with the UXKit and its VoxeetAppCompatActivity, the ScreenShareService can help
     * developers to integrate, allow, enable, start and stop ScreenShare from the device it's started on
     *
     * @return With an initialized SDK, the instance of the ScreenShareService
     */
    @Nullable
    public static ScreenShareService screenShare() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(ScreenShareService.class);
    }

    /**
     * Retrieve the unique instance of the SessionService
     * <p>
     * The service manages every call to :
     * - open a session
     * - close a session
     * - retrieve hi·er status
     *
     * @return With an initialized SDK, the instance of the SessionService
     */
    @Nullable
    public static SessionService session() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(SessionService.class);
    }

    /**
     * Retrieve the unique instance of the FilePresentationService
     * <p>
     * The service manages every call to :
     * - start a file presentation
     * - stop the started presentation
     * - update the started presentation
     * - get events from the presentation or the current state
     *
     * @return With an initialized SDK, the instance of the FilePresentationService
     */
    @Nullable
    public static FilePresentationService filePresentation() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(FilePresentationService.class);
    }

    /**
     * Retrieve the unique instance of the VideoPresentationService
     * <p>
     * The service manages every call to :
     * - start a VideoPresentation
     * - stop a VideoPresentation
     * - get or update the VideoPresentation state
     * <p>
     * The UXKit also provides components to help developers extends there apps and this service with auto injection of various plugin
     * - Youtube
     * - MP4
     * - or even custom implementations
     *
     * @return With an initialized SDK, the instance of the VideoPresentationService
     */
    @Nullable
    public static VideoPresentationService videoPresentation() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(VideoPresentationService.class);
    }

    /**
     * Retrieve the unique instance of the AudioService
     * <p>
     * This service wraps the various System call to manage Audio Routes, Bluetooth etc...
     *
     * @return With an initialized SDK, the instance of the AudioService
     */
    @Nullable
    public static AudioService audio() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(AudioService.class);
    }

    /**
     * Retrieve the unique instance of the RecordingService
     * <p>
     * This service wraps the recording related calls
     *
     * @return With an initialized SDK, the instance of the RecordingService
     */
    @Nullable
    public static RecordingService recording() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(RecordingService.class);
    }

    /**
     * Retrieve the unique instance of the CommandService
     * <p>
     * This service is used to send messages in a string formatted shape into specified conferences
     *
     * @return With an initialized SDK, the instance of the CommandService
     */
    @Nullable
    public static CommandService command() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(CommandService.class);
    }

    /**
     * Retrieve the unique instance of the LocalStatsService
     * <p>
     * This service is responsible for call to the Stats inside conferences
     *
     * @return With an initialized SDK, the instance of the LocalStatsService
     */
    @Nullable
    public static LocalStatsService localStats() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(LocalStatsService.class);
    }

    /**
     * Retrieve the unique instance of the NotificationService
     * <p>
     * This service is responsible for forwarding notification from developer to the proper registered manager
     *
     * @return With an initialized SDK, the instance of the NotificationService
     */
    @Nullable
    public static NotificationService notification() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(NotificationService.class);
    }

    @NoDocumentation
    public Retrofit getRetrofit() {
        return _voxeet_http.getRetrofit();
    }

    @NoDocumentation
    public OkHttpClient getClient() {
        return _voxeet_http.getClient();
    }

    @NoDocumentation
    public Context getApplicationContext() {
        return _application_context;
    }

    @NoDocumentation
    public AbstractVoxeetEnvironmentHolder getVoxeetEnvironmentHolder() {
        return _voxeet_environment_holder;
    }

    /**
     * Retrieve the instance of EventBus that is used by the SDK
     *
     * @return The instance of the EventBus which is internally used
     */
    @NonNull
    public EventBus getEventBus() {
        return _event_bus;
    }

    /**
     * Register object onto the SDK's EventBus. To be called before creating/joining any conference to trigger events
     *
     * @param subscriber the subscriber
     */
    public boolean register(@Nullable Object subscriber) {
        try {
            //MediaSDK.setContext(context);
            EventBus eventBus = getEventBus();

            if (null != subscriber && !eventBus.isRegistered(subscriber))
                eventBus.register(subscriber);
        } catch (EventBusException error) {
            error.printStackTrace();
            return false;
        }

        return true;
    }

    /**
     * Unregister from the SDk's EventBus. To be called in the on destroy of your activty/fragment
     *
     * @param subscriber the subscriber
     */
    public void unregister(@NonNull Object subscriber) {
        try {
            if (getEventBus().isRegistered(subscriber))
                getEventBus().unregister(subscriber);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @NoDocumentation
    protected HashMap<Class<? extends AbstractVoxeetService>, AbstractVoxeetService> getServices() {
        return _services;
    }

    @NoDocumentation
    protected VoxeetHttp getVoxeetHttp() {
        return _voxeet_http;
    }

    private void initExceptionIfOk(@NonNull Context application_context) {
        Log.d(TAG, "initExceptionIfOk: checking for exception to log internally");
        voxeetSentry = new VoxeetSentry(application_context);
        ExceptionManager.register(voxeetSentry);
        Log.d(TAG, "initExceptionIfOk: finished to try implementing error management");
    }

    private <T extends AbstractVoxeetService> T getServiceForKlass(Class<T> klass) {
        T service = null;
        Iterator<Class<? extends AbstractVoxeetService>> iterator = _services.keySet().iterator();
        while (iterator.hasNext()) {
            Class<? extends AbstractVoxeetService> next = iterator.next();
            AbstractVoxeetService item = _services.get(next);


            if (klass.isInstance(item)) {
                service = (T) item;
            }
        }

        if (service == null) {
            Log.d(TAG, klass.getSimpleName() + " not found in the list of services");
        }
        return service;
    }

    private void initServices() {
        getServices().put(ConferenceService.class, new ConferenceService(this, DEFAULT_TIMEOUT_MS));
        getServices().put(SessionService.class, new SessionService(this));
        getServices().put(FilePresentationService.class, new FilePresentationService(this));
        getServices().put(VideoPresentationService.class, new VideoPresentationService(this));
        getServices().put(AudioService.class, new AudioService(this));
        getServices().put(ScreenShareService.class, new ScreenShareService(this));
        getServices().put(MediaDeviceService.class, new MediaDeviceService(this));
        getServices().put(LocalStatsService.class, new LocalStatsService(this));
        getServices().put(ChatService.class, new ChatService(this));
        getServices().put(CommandService.class, new CommandService(this));
        getServices().put(RecordingService.class, new RecordingService(this));
        getServices().put(NotificationService.class, new NotificationService(this));
    }
}
