package com.voxeet.sdk.core.services.conference.promises;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.sdk.core.services.AbstractPromiseable;
import com.voxeet.sdk.core.services.ConferenceSdkObservableProvider;
import com.voxeet.sdk.core.services.ConferenceService;
import com.voxeet.sdk.core.services.MediaDeviceService;
import com.voxeet.sdk.core.services.conference.information.ConferenceInformation;
import com.voxeet.sdk.core.services.conference.information.ConferenceUserType;
import com.voxeet.sdk.core.services.conference.information.LocalConferenceType;
import com.voxeet.sdk.events.error.HttpException;
import com.voxeet.sdk.events.error.ReplayConferenceErrorEvent;
import com.voxeet.sdk.events.restapi.ReplayConferenceResult;
import com.voxeet.sdk.json.SdkConferenceReplayBody;
import com.voxeet.sdk.utils.HttpHelper;

import org.greenrobot.eventbus.EventBus;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.ErrorPromise;
import eu.codlab.simplepromise.solve.PromiseExec;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;
import retrofit2.Call;
import retrofit2.Response;

public class ReplayPromise extends AbstractPromiseable<Boolean> {
    private final static String TAG = ReplayPromise.class.getSimpleName();
    private final String conferenceId;
    private final long offset;

    public ReplayPromise(@NonNull ConferenceService parent,
                         @NonNull MediaDeviceService mediaDeviceService,
                         @NonNull ConferenceSdkObservableProvider provider,
                         @Nullable ConferenceInformation information,
                         @NonNull EventBus eventBus,
                         @NonNull String conferenceId,
                         long offset) {
        super(parent, mediaDeviceService, provider, information, eventBus);

        this.conferenceId = conferenceId;
        this.offset = offset;
    }

    @NonNull
    @Override
    public Promise<Boolean> createPromise() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                final ConferenceInformation information = getConferenceInformation(conferenceId);

                information.setConferenceUserType(ConferenceUserType.LISTENER);

                final Call<ReplayConferenceResult> user = getProvider().getReplayObservable(conferenceId, new SdkConferenceReplayBody(offset));
                HttpHelper.enqueue(user, new HttpHelper.HttpCallback<ReplayConferenceResult>() {
                    @Override
                    public void onSuccess(@NonNull ReplayConferenceResult object, @NonNull Response<ReplayConferenceResult> response) {
                        Log.d(TAG, "Success server answer for replaying mConference with id: " +
                                conferenceId + "at offset: " + offset + " given := " + object.conferenceId + " " + object.conferenceAlias
                                + " " + information);

                        moveConference(object.conferenceId, information);

                        information.getConference().setConferenceId(object.conferenceId);
                        information.getConference().setConferenceAlias(object.conferenceAlias);

                        information.setConferenceType(LocalConferenceType.REPLAY);

                        joinConference(information)
                                .then(new PromiseExec<Boolean, Object>() {
                                    @Override
                                    public void onCall(@Nullable Boolean result, @NonNull Solver<Object> internal_solver) {
                                        solver.resolve(result);
                                    }
                                })
                                .error(new ErrorPromise() {
                                    @Override
                                    public void onError(Throwable error) {
                                        error.printStackTrace();
                                        solver.resolve(false);
                                    }
                                });
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<ReplayConferenceResult> response) {
                        HttpException.dumpErrorResponse(response);

                        Log.d(TAG, "Failed to replay mConference");

                        getEventBus().post(new ReplayConferenceErrorEvent(getParent().handleError(e)));
                        e.printStackTrace();
                        solver.resolve(false);
                    }
                });

            }
        });
    }
}
