package com.voxeet.sdk.core.services.conference.promises;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.sdk.core.services.AbstractPromiseable;
import com.voxeet.sdk.core.services.ConferenceSdkObservableProvider;
import com.voxeet.sdk.core.services.ConferenceService;
import com.voxeet.sdk.core.services.MediaDeviceService;
import com.voxeet.sdk.core.services.conference.information.ConferenceInformation;
import com.voxeet.sdk.events.error.GetConferenceStatusErrorEvent;
import com.voxeet.sdk.events.error.HttpException;
import com.voxeet.sdk.events.promises.PromiseGetConferenceStatusErrorEventException;
import com.voxeet.sdk.events.restapi.ConferenceStatusResult;
import com.voxeet.sdk.models.v1.ConferenceUser;
import com.voxeet.sdk.utils.HttpHelper;

import org.greenrobot.eventbus.EventBus;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;
import retrofit2.Call;
import retrofit2.Response;

public class GetConferenceStatusPromiseable  extends AbstractPromiseable<ConferenceStatusResult> {

    public GetConferenceStatusPromiseable(@NonNull ConferenceService parent,
                                          @NonNull MediaDeviceService mediaDeviceService,
                                          @NonNull ConferenceSdkObservableProvider provider,
                                          @Nullable ConferenceInformation information,
                                          @NonNull EventBus eventBus) {
        super(parent, mediaDeviceService, provider, information, eventBus);
    }

    @NonNull
    @Override
    public Promise<ConferenceStatusResult> createPromise() {
        return new Promise<>(new PromiseSolver<ConferenceStatusResult>() {
            @Override
            public void onCall(@NonNull final Solver<ConferenceStatusResult> solver) {
                final Call<ConferenceStatusResult> user = getProvider().getConferenceStatusObservable(getInformation().getConference().getId());
                HttpHelper.enqueue(user, new HttpHelper.HttpCallback<ConferenceStatusResult>() {
                    @Override
                    public void onSuccess(@NonNull ConferenceStatusResult object, @NonNull Response<ConferenceStatusResult> response) {
                        try {
                            if (null != response && null != object.getConferenceUsers()) {
                                for (ConferenceUser conferenceUser : object.getConferenceUsers()) {
                                    log(conferenceUser.getUserId() + " / " + conferenceUser.getStatus());
                                }
                                log("Dispatching mConference status result");
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                        }

                        getEventBus().post(object);
                        solver.resolve(object);
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<ConferenceStatusResult> response) {
                        HttpException.dumpErrorResponse(response);

                        log("Error while getting mConference history");
                        e.printStackTrace();

                        try {
                            GetConferenceStatusErrorEvent event = new GetConferenceStatusErrorEvent(getParent().handleError(e));
                            getEventBus().post(event);
                            throw new PromiseGetConferenceStatusErrorEventException(event, e);
                        } catch (PromiseGetConferenceStatusErrorEventException exception) {
                            solver.reject(exception);
                        }
                    }
                });
            }
        });
    }
}
