package com.voxeet.sdk.core.services;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.google.gson.Gson;
import com.voxeet.sdk.core.VoxeetSdk;
import com.voxeet.sdk.core.network.websocket.VoxeetWebSocket;
import com.voxeet.sdk.core.services.authenticate.WebSocketState;
import com.voxeet.sdk.events.BaseEvent;
import com.voxeet.sdk.events.error.SocketErrorEvent;
import com.voxeet.sdk.events.sdk.MessageReceived;
import com.voxeet.sdk.events.sdk.SocketConnectEvent;
import com.voxeet.sdk.events.sdk.SocketDisconnectEvent;
import com.voxeet.sdk.events.sdk.SocketStateChangeEvent;
import com.voxeet.sdk.factories.EventsFactory;
import com.voxeet.sdk.json.BroadcastEvent;
import com.voxeet.sdk.json.ConferenceMessageReceived;
import com.voxeet.sdk.json.ConferenceStats;
import com.voxeet.sdk.json.Event;
import com.voxeet.sdk.json.EventNames;
import com.voxeet.sdk.json.FileConverted;
import com.voxeet.sdk.json.FilePresentationStarted;
import com.voxeet.sdk.json.FilePresentationStopped;
import com.voxeet.sdk.json.FilePresentationUpdated;
import com.voxeet.sdk.json.InterfaceEvent;
import com.voxeet.sdk.json.VideoPresentationPaused;
import com.voxeet.sdk.json.VideoPresentationPlay;
import com.voxeet.sdk.json.VideoPresentationSeek;
import com.voxeet.sdk.json.VideoPresentationStarted;
import com.voxeet.sdk.json.VideoPresentationStopped;

import org.greenrobot.eventbus.EventBus;

public class VoxeetDispatcher {

    private static final String TAG = VoxeetDispatcher.class.getSimpleName();

    private static EventBus eventBus = EventBus.getDefault();

    public static void dispatch(WebSocketState state) {
        eventBus.post(new SocketStateChangeEvent(state));
    }
    /**
     * Map and dispatch events through the eventbus.
     *
     * @param eventName the event name
     * @param message   the message
     */
    public static void dispatch(String eventName, String message) {
        Log.d(TAG, "dispatch: " + eventName + " := " + message);
        try {
            InterfaceEvent event;
            switch (eventName) {
                case VoxeetWebSocket.SOCKET_DISCONNECTED:
                    event = handleDisconnect(message);
                    break;
                case VoxeetWebSocket.SOCKET_ERROR:
                    event = handleError(message);
                    break;
                case VoxeetWebSocket.SOCKET_MESSAGE:
                    event = handleMessage(message);
                    break;
                case VoxeetWebSocket.SOCKET_CONNECTED:
                    event = handleSocketConnect(message);
                    break;
                default:
                    throw new IllegalStateException("Unknown event type");
            }

            if (null != event) {
                eventBus.post(event);
            } else {
                Log.d(TAG, "dispatch: event internally managed");
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * Dispatch a socket error event.
     */
    private static BaseEvent handleError(String message) {
        return new SocketErrorEvent(message);
    }

    /**
     * Dispatch a socket disconnected event.
     */
    private static BaseEvent handleDisconnect(String message) {
        return new SocketDisconnectEvent(message);
    }

    /**
     * Dispatch a socket connected event.
     */
    private static BaseEvent handleSocketConnect(String message) {
        return new SocketConnectEvent(message);
    }

    /**
     * Dispatch all other events.
     * The null values implies it has been managed so nothing to do here
     */
    @Nullable
    private static InterfaceEvent handleMessage(String message) {
        InterfaceEvent busEvent;

        Event event = EventsFactory.decode(message);
        switch (event.getType()) {
            case EventNames.OFFER_CREATED:
                busEvent = event;
                break;
            case EventNames.CONFERENCE_MESSAGE_RECEIVED:
                busEvent = handleConferenceMessage((ConferenceMessageReceived) event);
                break;
            case EventNames.BROADCAST_EVENT:
                busEvent = handleBroadcast((BroadcastEvent) event);
                break;
            case EventNames.CONFERENCE_DESTROYED:
            case EventNames.CONTACT_PROFILE_UPDATED:
            case EventNames.PARTICIPANT_UPDATED:
            case EventNames.OWN_CONFERENCE_CREATED:
            case EventNames.OWN_EXTERNAL_INVITATION:
            case EventNames.OWN_CONTACT_REMOVED:
            case EventNames.RENEGOCIATION_ENDED:
            case EventNames.RECORDING_STATUS_UPDATE:
            case EventNames.INVITATION_RECEIVED:
            case EventNames.FILE_DELETED:
            case EventNames.FILE_ADDED:
            case EventNames.QUALITY_UPDATED:
            case EventNames.WHISPER_INVITATION_RECEIVED:
            case EventNames.WHISPER_INVITATION_ACCEPTED:
            case EventNames.WHISPER_INVITATION_DECLINED:
            case EventNames.WHISPER_LEFT:
            case EventNames.OWN_USER_INVITED:
            case EventNames.USER_INVITED:
            case EventNames.PARTICIPANT_SWITCH:
            case EventNames.PARTICIPANT_ADDED:
            case EventNames.OWN_USER_SWITCH:
            case EventNames.CONFERENCE_ENDED:
            case EventNames.CONFERENCE_UPDATED:
            case EventNames.CONTACT_ADDED:
            case EventNames.PEER_CONNECTION_UPDATED:
                busEvent = event;
                break;
            case EventNames.FILE_CONVERTED:
            case EventNames.FILE_PRESENTATION_STARTED:
            case EventNames.FILE_PRESENTATION_STOPPED:
            case EventNames.FILE_PRESENTATION_UPDATED:
                handleFilePresentation(event);
                return null;
            case EventNames.VIDEO_PRESENTATION_STARTED:
            case EventNames.VIDEO_PRESENTATION_STOPPED:
            case EventNames.VIDEO_PRESENTATION_PLAY:
            case EventNames.VIDEO_PRESENTATION_PAUSED:
            case EventNames.VIDEO_PRESENTATION_SEEK:
                handleVideoPresentation(event);
                return null;
            case EventNames.CONFERENCE_STATS:
                busEvent = new Gson().fromJson(message, ConferenceStats.class);
                break;
            default:
                throw new IllegalStateException("Unknown type of event");
        }
        return busEvent;
    }

    /**
     * Dispatch conference related events
     */
    private static InterfaceEvent handleConferenceMessage(ConferenceMessageReceived event) {
        Event event1 = EventsFactory.decode(event.message);
        if (event1 == null)
            return new MessageReceived(event.userId, event.conferenceId, event.message);
        else {
            switch (event1.getType()) {
                case EventNames.VIDEO_PRESENTATION_STARTED:
                case EventNames.VIDEO_PRESENTATION_STOPPED:
                case EventNames.VIDEO_PRESENTATION_PLAY:
                case EventNames.VIDEO_PRESENTATION_PAUSED:
                case EventNames.VIDEO_PRESENTATION_SEEK:
                    handleVideoPresentation(event1);
                    return null;
                default:
                    return new MessageReceived(event.userId, event.conferenceId, event.message);
            }
        }
    }

    private static void handleFilePresentation(@NonNull Event event) {
        FilePresentationService service = VoxeetSdk.filePresentation();

        if(null == service) {
            Log.d(TAG, "handleFilePresentation: invalid service state");
            return;
        }

        switch(event.getType()) {
            case EventNames.FILE_CONVERTED:
                service.onInternalServiceEvent((FileConverted) event);
                break;
            case EventNames.FILE_PRESENTATION_STARTED:
                service.onInternalServiceEvent((FilePresentationStarted) event);
                break;
            case EventNames.FILE_PRESENTATION_STOPPED:
                service.onInternalServiceEvent((FilePresentationStopped) event);
                break;
            case EventNames.FILE_PRESENTATION_UPDATED:
                service.onInternalServiceEvent((FilePresentationUpdated) event);
                break;
        }
    }

    private static void handleVideoPresentation(@NonNull Event event) {
        VideoPresentationService service = VoxeetSdk.videoPresentation();
        if (null == service) {
            Log.d(TAG, "handleVideoPresentation: invalid service state");
            return;
        }
        switch (event.getType()) {
            case EventNames.VIDEO_PRESENTATION_STARTED:
                service.onInternalServiceEvent((VideoPresentationStarted) event);
                return;
            case EventNames.VIDEO_PRESENTATION_STOPPED:
                service.onInternalServiceEvent((VideoPresentationStopped) event);
                return;
            case EventNames.VIDEO_PRESENTATION_PLAY:
                service.onInternalServiceEvent((VideoPresentationPlay) event);
                return;
            case EventNames.VIDEO_PRESENTATION_PAUSED:
                service.onInternalServiceEvent((VideoPresentationPaused) event);
                return;
            case EventNames.VIDEO_PRESENTATION_SEEK:
                service.onInternalServiceEvent((VideoPresentationSeek) event);
                return;
            default:
        }
    }

    /**
     * Dispatch a conference broadcast based event.
     */
    private static InterfaceEvent handleBroadcast(BroadcastEvent event) {
        Event event1 = EventsFactory.decode(event.message);
        switch (event1.getType()) {
            case EventNames.TYPING_DETECTION:
                return event1;
            default:
                throw new IllegalStateException("Unknown broadcast type event");
        }
    }
}
