package com.voxeet.sdk.views;

import android.content.Context;
import android.content.res.Resources;
import android.graphics.Point;
import android.graphics.SurfaceTexture;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.Nullable;
import android.util.AttributeSet;
import android.view.Display;
import android.view.TextureView;
import android.view.WindowManager;

import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.NoDocumentation;

import org.webrtc.EglBaseMethods;
import org.webrtc.GlRectDrawer;
import org.webrtc.Logging;
import org.webrtc.RendererCommon;
import org.webrtc.SafeRenderFrameEglRenderer;
import org.webrtc.ThreadUtils;
import org.webrtc.VideoFrame;
import org.webrtc.VideoSink;

import java.util.concurrent.CountDownLatch;

/**
 * Implements org.webrtc.VideoRenderer.Callbacks by displaying the video stream on a SurfaceView.
 * renderFrame() is asynchronous to avoid blocking the calling thread.
 * This class is thread safe and handles access from potentially four different threads:
 * Interaction from the main app in init, release, setMirror, and setScalingtype.
 * Interaction from C++ rtc::VideoSinkInterface in renderFrame.
 * Interaction from the Activity lifecycle in surfaceCreated, surfaceChanged, and surfaceDestroyed.
 * Interaction with the layout framework in onMeasure and onSizeChanged.
 */
@Annotate
public class VoxeetRenderer extends TextureView
        implements TextureView.SurfaceTextureListener, VideoSink {
    private static final String TAG = "VoxeetRenderer";

    private Point size = new Point();

    // Cached resource name.
    private final String resourceName;
    private final RendererCommon.VideoLayoutMeasure videoLayoutMeasure =
            new RendererCommon.VideoLayoutMeasure();
    private final SafeRenderFrameEglRenderer eglRenderer;
    private Handler mHandler;
    private boolean pendingLayout = false;

    // Callback for reporting renderer events. Read-only after initilization so no lock required.
    private RendererCommon.RendererEvents rendererEvents;

    private final Object layoutLock = new Object();
    private boolean isRenderingPaused = false;
    private boolean isFirstFrameRendered;
    private int rotatedFrameWidth;
    private int rotatedFrameHeight;
    private int frameRotation;

    // Accessed only on the main thread.
    private boolean enableFixedSize;
    private int surfaceWidth;
    private int surfaceHeight;
    private boolean isEglRendererInitialized;

    private RendererCommon.ScalingType setScalingType;
    private int delayedPostTimeout = 1000;
    private float translateX = 0;
    private float translateY = 0;
    private boolean attached;

    /**
     * Standard View constructor. In order to render something, you must first call init().
     */
    public VoxeetRenderer(Context context) {
        super(context);
        isEglRendererInitialized = false;
        this.resourceName = getResourceName();
        eglRenderer = new SafeRenderFrameEglRenderer(new SafeRenderFrameEglRenderer.SurfaceTextureProvider() {
            @Nullable
            @Override
            public SurfaceTexture getSurfaceTexture() {
                return VoxeetRenderer.this.getSurfaceTexture();
            }
        }, resourceName);
        setSurfaceTextureListener(this);

        mHandler = new Handler(Looper.getMainLooper());
    }

    /**
     * Standard View constructor. In order to render something, you must first call init().
     */
    public VoxeetRenderer(Context context, AttributeSet attrs) {
        super(context, attrs);
        isEglRendererInitialized = false;
        this.resourceName = getResourceName();
        eglRenderer = new SafeRenderFrameEglRenderer(new SafeRenderFrameEglRenderer.SurfaceTextureProvider() {
            @Nullable
            @Override
            public SurfaceTexture getSurfaceTexture() {
                return VoxeetRenderer.this.getSurfaceTexture();
            }
        }, resourceName);
        setSurfaceTextureListener(this);

        mHandler = new Handler(Looper.getMainLooper());
    }


    /**
     * Initialize this class, sharing resources with |sharedContext|. It is allowed to call init() to
     * reinitialize the renderer after a previous init()/release() cycle.
     */
    public void init(EglBaseMethods.Context sharedContext, RendererCommon.RendererEvents rendererEvents) {
        init(sharedContext, rendererEvents, EglBaseMethods.CONFIG_RGBA, new GlRectDrawer());
    }

    /**
     * Initialize this class, sharing resources with |sharedContext|. The custom |drawer| will be used
     * for drawing frames on the EGLSurface. This class is responsible for calling release() on
     * |drawer|. It is allowed to call init() to reinitialize the renderer after a previous
     * init()/release() cycle.
     */
    public void init(final EglBaseMethods.Context sharedContext,
                     RendererCommon.RendererEvents rendererEvents, final int[] configAttributes,
                     RendererCommon.GlDrawer drawer) {
        synchronized (eglRenderer) {
            if (isEglRendererInitialized) return;

            ThreadUtils.checkIsOnMainThread();
            this.rendererEvents = rendererEvents;
            synchronized (layoutLock) {
                isFirstFrameRendered = false;
                rotatedFrameWidth = 0;
                rotatedFrameHeight = 0;
                frameRotation = 0;
            }
            eglRenderer.init(sharedContext, configAttributes, drawer);
            isEglRendererInitialized = true;
        }
    }

    @NoDocumentation
    @Override
    protected void onAttachedToWindow() {
        super.onAttachedToWindow();
        setSurfaceTextureListener(this);

        if (null != eglRenderer) eglRenderer.setAttached(true);

        WindowManager wm = (WindowManager) getContext().getSystemService(Context.WINDOW_SERVICE);
        if (wm != null) {
            Display display = wm.getDefaultDisplay();

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
                display.getRealSize(size);
            } else {
                display.getSize(size);
            }
        }
    }

    @NoDocumentation
    @Override
    protected void onDetachedFromWindow() {
        if (null != eglRenderer) eglRenderer.setAttached(false);

        super.onDetachedFromWindow();
    }

    /**
     * Block until any pending frame is returned and all GL resources released, even if an interrupt
     * occurs. If an interrupt occurs during release(), the interrupt flag will be set. This function
     * should be called before the Activity is destroyed and the EGLContext is still valid. If you
     * don't call this function, the GL resources might leak.
     */
    public void release() {
        synchronized (eglRenderer) {
            if (!isEglRendererInitialized) return;

            isEglRendererInitialized = false;
            setSurfaceTextureListener(null);
            eglRenderer.releaseEglSurface(new Runnable() {
                @Override
                public void run() {
                    //nothing to do
                }
            });
            eglRenderer.release();

            mHandler = null;
        }
    }

    /**
     * Register a callback to be invoked when a new video frame has been received.
     *
     * @param listener    The callback to be invoked. The callback will be invoked on the render thread.
     *                    It should be lightweight and must not call removeFrameListener.
     * @param scale       The scale of the Bitmap passed to the callback, or 0 if no Bitmap is
     *                    required.
     * @param drawerParam Custom drawer to use for this frame listener.
     */
    public void addFrameListener(
            SafeRenderFrameEglRenderer.FrameListener listener, float scale, RendererCommon.GlDrawer drawerParam) {
        eglRenderer.addFrameListener(listener, scale, drawerParam);
    }

    /**
     * Register a callback to be invoked when a new video frame has been received. This version uses
     * the drawer of the EglRenderer that was passed in init.
     *
     * @param listener The callback to be invoked. The callback will be invoked on the render thread.
     *                 It should be lightweight and must not call removeFrameListener.
     * @param scale    The scale of the Bitmap passed to the callback, or 0 if no Bitmap is
     *                 required.
     */
    public void addFrameListener(SafeRenderFrameEglRenderer.FrameListener listener, float scale) {
        eglRenderer.addFrameListener(listener, scale);
    }

    /**
     * Unregister a callback to be invoked when a new video frame has been received.
     *
     * @param listener The specific instance of the listener to remove
     */
    public void removeFrameListener(SafeRenderFrameEglRenderer.FrameListener listener) {
        eglRenderer.removeFrameListener(listener);
    }

    /**
     * Enables fixed size for the surface. This provides better performance but might be buggy on some
     * devices. By default this is turned off.
     */
    public void setEnableHardwareScaler(boolean enabled) {
        ThreadUtils.checkIsOnMainThread();
        enableFixedSize = enabled;
        updateSurfaceSize(false);
    }

    /**
     * Set if the video stream should be mirrored or not.
     */
    public void setMirror(final boolean mirror) {
        eglRenderer.setMirror(mirror);
    }

    /**
     * Check for the current mirror state of the renderer
     *
     * @return the indicator of the mirror activation
     */
    public boolean isMirror() {
        return eglRenderer.isMirror();
    }

    /**
     * Check for any first frame already rendered. Usefull to prevent blink when a renderer was rendering a video, stopped, disappeared and starts a new one showing the previous last frame(s)
     *
     * @return the first frame indicator
     */
    public boolean isFirstFrameRendered() {
        return isFirstFrameRendered;
    }

    /**
     * Set how the video will fill the allowed layout area.
     */
    public void setScalingType(RendererCommon.ScalingType scalingType) {
        ThreadUtils.checkIsOnMainThread();
        surfaceHeight = 0;
        surfaceWidth = 0;

        setScalingType = scalingType;
        setScalingType(scalingType, scalingType);
    }

    /**
     * Set a scaling type on the 2 various orientation : match and mismatch
     * @param scalingTypeMatchOrientation
     * @param scalingTypeMismatchOrientation
     */
    public void setScalingType(RendererCommon.ScalingType scalingTypeMatchOrientation,
                               RendererCommon.ScalingType scalingTypeMismatchOrientation) {
        setNextDelayedPostTimeout(0);
        ThreadUtils.checkIsOnMainThread();
        videoLayoutMeasure.setScalingType(scalingTypeMatchOrientation, scalingTypeMismatchOrientation);

        eglRenderer.setScalingType(scalingTypeMatchOrientation);
        updateSurfaceSize();
    }

    /**
     * TODO implement a way to get both types if set
     *
     * @return
     */
    @Nullable
    public RendererCommon.ScalingType getScalingType() {
        return setScalingType;
    }

    /**
     * Limit render framerate.
     *
     * @param fps Limit render framerate to this value, or use Float.POSITIVE_INFINITY to disable fps
     *            reduction.
     */
    public void setFpsReduction(float fps) {
        synchronized (layoutLock) {
            isRenderingPaused = fps == 0f;
        }
        eglRenderer.setFpsReduction(fps);
    }

    /**
     * Disable the FpsReduction. Usefull to prevent testing environments or when high quality is possible and needed
     */
    public void disableFpsReduction() {
        synchronized (layoutLock) {
            isRenderingPaused = false;
        }
        eglRenderer.disableFpsReduction();
    }

    public void pauseVideo() {
        synchronized (layoutLock) {
            isRenderingPaused = true;
        }
        eglRenderer.pauseVideo();
    }

    @NoDocumentation
    @Override
    public void onFrame(VideoFrame frame) {
        try {
            updateFrameDimensionsAndReportEvents(frame);
            eglRenderer.onFrame(frame);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    // View layout interface.
    @NoDocumentation
    @Override
    protected void onMeasure(int widthSpec, int heightSpec) {
        ThreadUtils.checkIsOnMainThread();
        final Point size;
        synchronized (layoutLock) {
            size = videoLayoutMeasure.measure(widthSpec, heightSpec, rotatedFrameWidth, rotatedFrameHeight);
        }

        if (size.y > getScreenHeight()) size.y = getScreenHeight();
        if (size.x > getScreenWidth()) size.x = getScreenWidth();

        setMeasuredDimension(size.x, size.y);

        pendingLayout = false;
        posting = false;
    }

    @NoDocumentation
    @Override
    protected void onLayout(boolean changed, int left, int top, int right, int bottom) {
        ThreadUtils.checkIsOnMainThread();

        eglRenderer.setLayoutAspectRatio((right - left) / (float) (bottom - top));
        updateSurfaceSize(false);
    }

    private void updateSurfaceSize() {
        updateSurfaceSize(true);
    }

    public void forceRecalculateWidthHeight() {
        surfaceWidth = surfaceHeight = 0;
    }

    private void updateSurfaceSize(boolean sendLayout) {
        ThreadUtils.checkIsOnMainThread();
        synchronized (layoutLock) {

            if (enableFixedSize && rotatedFrameWidth != 0 && rotatedFrameHeight != 0 && getWidth() != 0
                    && getHeight() != 0) {
                final float layoutAspectRatio = getWidth() / (float) getHeight();
                final float frameAspectRatio = rotatedFrameWidth / (float) rotatedFrameHeight;
                final int drawnFrameWidth;
                final int drawnFrameHeight;
                if (frameAspectRatio > layoutAspectRatio) {
                    drawnFrameWidth = (int) (rotatedFrameHeight * layoutAspectRatio);
                    drawnFrameHeight = rotatedFrameHeight;
                } else {
                    drawnFrameWidth = rotatedFrameWidth;
                    drawnFrameHeight = (int) (rotatedFrameWidth / layoutAspectRatio);
                }
                // Aspect ratio of the drawn frame and the view is the same.
                final int width = Math.min(getWidth(), drawnFrameWidth);
                final int height = Math.min(getHeight(), drawnFrameHeight);
                if (width != surfaceWidth || height != surfaceHeight) {
                    surfaceWidth = width;
                    surfaceHeight = height;
                    if (sendLayout && surfaceWidth != 0 && surfaceHeight != 0)
                        requestLayoutIfNotPending();
                }
            } else {
                surfaceWidth = surfaceHeight = 0;
            }
        }

    }

    @NoDocumentation
    @Override
    public void onSurfaceTextureAvailable(SurfaceTexture surface, int width, int height) {
        ThreadUtils.checkIsOnMainThread();
        eglRenderer.createEglSurface(surface);
        surfaceWidth = surfaceHeight = 0;

        updateSurfaceSize();
    }

    @NoDocumentation
    @Override
    public void onSurfaceTextureSizeChanged(SurfaceTexture surface, int width, int height) {
        ThreadUtils.checkIsOnMainThread();
    }

    @NoDocumentation
    @Override
    public boolean onSurfaceTextureDestroyed(SurfaceTexture surface) {
        ThreadUtils.checkIsOnMainThread();
        final CountDownLatch completionLatch = new CountDownLatch(1);
        eglRenderer.releaseEglSurface(new Runnable() {
            @Override
            public void run() {
                completionLatch.countDown();
            }
        });
        ThreadUtils.awaitUninterruptibly(completionLatch);

        return true;
    }

    @NoDocumentation
    @Override
    public void onSurfaceTextureUpdated(SurfaceTexture surface) {
        ThreadUtils.checkIsOnMainThread();
        surfaceWidth = surfaceHeight = 0;
        updateSurfaceSize(false);
    }

    private String getResourceName() {
        try {
            return getResources().getResourceEntryName(getId()) + ": ";
        } catch (Resources.NotFoundException e) {
            return "";
        }
    }

    /**
     * Post a task to clear the SurfaceView to a transparent uniform color.
     */
    public void clearImage() {
        eglRenderer.clearImage();
    }

    private Runnable update = new Runnable() {
        @Override
        public void run() {
            updateSurfaceSize();
            requestLayoutIfNotPending();
        }
    };

    // Update frame dimensions and report any changes to |rendererEvents|.
    private void updateFrameDimensionsAndReportEvents(VideoFrame frame) {
        synchronized (layoutLock) {
            if (isRenderingPaused) {
                return;
            }
            if (!isFirstFrameRendered) {
                isFirstFrameRendered = true;
                logD("Reporting first rendered frame.");
                if (rendererEvents != null) {
                    rendererEvents.onFirstFrameRendered();
                }
            }
            if (rotatedFrameWidth != frame.getRotatedWidth()
                    || rotatedFrameHeight != frame.getRotatedHeight()
                    || frameRotation != frame.getRotation()) {
                logD("Reporting frame resolution changed to " + frame.getBuffer().getWidth() + "x"
                        + frame.getBuffer().getHeight() + " with rotation " + frame.getRotation());
                if (rendererEvents != null) {
                    rendererEvents.onFrameResolutionChanged(
                            frame.getBuffer().getWidth(), frame.getBuffer().getHeight(), frame.getRotation());
                }
                rotatedFrameWidth = frame.getRotatedWidth();
                rotatedFrameHeight = frame.getRotatedHeight();
                frameRotation = frame.getRotation();
                post(update);
            }
        }
    }

    private void logD(String string) {
        Logging.d(TAG, resourceName + string);
    }

    private boolean posting = false;
    private boolean lock = false;

    private final Runnable post = new Runnable() {
        @Override
        public void run() {
            lock = false;
            if (null != mHandler) mHandler.removeCallbacks(this);
            requestLayout();
        }
    };

    private void setNextDelayedPostTimeout(int delayedPostTimeout) {
        this.delayedPostTimeout = delayedPostTimeout;
    }

    private void requestLayoutIfNotPending() {

        if (posting || (null == mHandler)) return;
        if (lock) return;

        lock = true;
        posting = true;
        int delay = delayedPostTimeout;
        delayedPostTimeout = 1000;
        mHandler.postDelayed(post, delay);
    }

    @NoDocumentation
    public int getScreenWidth() {
        return size.x;
    }

    @NoDocumentation
    public int getScreenHeight() {
        return size.y;
    }

    /**
     * Method to be called by any parent view to force (or when known) the view to fire once again the first frame
     * @param isFirstFrameRendered the new state to force inject
     */
    public void setFirstFrameRendered(boolean isFirstFrameRendered) {
        this.isFirstFrameRendered = isFirstFrameRendered;
    }

    @NoDocumentation
    public void setDeltaX(float dx) {
        translateX = dx;
        applyRawTranslation();
    }

    @NoDocumentation
    public void setDeltaY(float dy) {
        translateY = dy;
        applyRawTranslation();
    }

    @NoDocumentation
    public float getTranslationY() {
        return translateY;
    }

    @NoDocumentation
    public float getTranslationX() {
        return translateX;
    }

    @NoDocumentation
    public float getRendererScaleX() {
        return rotatedFrameWidth * 1.f / getWidth();
    }

    @NoDocumentation
    public float getRendererScaleY() {
        return rotatedFrameHeight * 1.f / getHeight();
    }

    @NoDocumentation
    public float getRotatedFrameWidth() {
        return rotatedFrameWidth;
    }

    @NoDocumentation
    public float getRotatedFrameHeight() {
        return rotatedFrameHeight;
    }

    private void applyRawTranslation() {
        float scaleX = getRendererScaleX();
        float scaleY = getRendererScaleY();

        if (scaleY < scaleX) {
            //un/-stretch to fit height of the screen
            scaleX = scaleY;
            scaleY = 1;
        } else if (scaleY > scaleX) {
            //un/stretch to fit the width of the screen
            scaleX = 1;
            scaleY = scaleX;
        }

        float dx = scaleX * translateX;
        float dy = scaleY * translateY;

        eglRenderer.setTranslation(dx, dy);
    }
}
