package com.voxeet.sdk.models;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.android.media.MediaStream;
import com.voxeet.sdk.json.UserInfo;
import com.voxeet.sdk.models.v1.ConferenceUser;
import com.voxeet.sdk.models.v1.ConferenceUserStatus;
import com.voxeet.sdk.models.v1.Participant;
import com.voxeet.sdk.models.v1.SdkParticipant;
import com.voxeet.sdk.models.v2.UserMediaStreamHandler;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.NoDocumentation;

import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

@Annotate
/**
 * The User class represents information about a given user. It is responsible to hold the various information about hier
 * - id
 * - user info
 *   - external id
 *   - name
 *   - avatar url
 *
 * Developers are responsible about the UserInfo element. The id is provided and managed by the SDK
 */
public class User {

    @NonNull
    private UserMediaStreamHandler streamsHandler;

    @NonNull
    private CopyOnWriteArrayList<MediaStream> streams;

    @Nullable
    private String id;

    @Nullable
    private UserInfo userInfo;

    @NonNull
    private ConferenceUserStatus status;
    private String quality; //needed ?

    private User() {
        streams = new CopyOnWriteArrayList<>();
        streamsHandler = new UserMediaStreamHandler(this);
    }

    @NoDocumentation
    public User(@Nullable String id, @Nullable UserInfo userInfo) {
        this();
        this.id = id;
        this.userInfo = userInfo;
    }

    /**
     * Create a new ConferenceUser
     *
     * @param conferenceUser the conference user to use as creator
     */
    @NoDocumentation
    public User(@NonNull ConferenceUser conferenceUser) {
        this(conferenceUser.getUserId(), conferenceUser.getUserInfo());
        updateStatus(conferenceUser.getConferenceStatus());
    }

    /**
     * Create a new ConferenceUser
     *
     * @param userInfo the developer's management of the information
     */
    @NoDocumentation
    public User(@Nullable UserInfo userInfo) {
        this(null, userInfo);
    }

    /**
     * Create a new ConferenceUser
     *
     * @param participant from a given participant obtained from a server event
     */
    @NoDocumentation
    public User(@NonNull Participant participant) {
        this(participant.getParticipantId(), new UserInfo(participant.getName(), participant.getExternalId(), participant.getAvatarUrl()));
    }

    @NoDocumentation
    public User(@NonNull SdkParticipant participant) {
        this(participant.getUserId(),
                new com.voxeet.sdk.json.UserInfo(
                        participant.getMetadata().getExternalName(),
                        participant.getMetadata().getExternalId(),
                        participant.getMetadata().getExternalPhotoUrl()
                )
        );
        updateStatus(ConferenceUserStatus.valueOf(participant.getStatus()));
    }

    public boolean isLocallyActive() {
        return ConferenceUserStatus.ON_AIR.equals(getStatus())
                ||
                (ConferenceUserStatus.CONNECTING.equals(getStatus()) && streams().size() > 0);
    }

    /**
     * Get the Voxeet's identifier of the given user
     *
     * @return a possible null value
     */
    @Nullable
    public String getId() {
        return id;
    }

    /**
     * Get the Developer's information about the current user
     *
     * @return the displayable and manageable information
     */
    @Nullable
    public UserInfo getUserInfo() {
        return userInfo;
    }

    @NonNull
    public List<MediaStream> streams() {
        return streams;
    }

    @NonNull
    public UserMediaStreamHandler streamsHandler() {
        return streamsHandler;
    }


    /**
     * Set the status of the user from a server point of view. TBU by the SDK only
     * <p>
     * Warning : the status is not checking any lifecycle, setting an improper information and not managing it can lead to developer's side issue
     *
     * @param status a valid status to set
     * @return the current instance of the user
     */
    @NonNull
    public User updateStatus(@NonNull ConferenceUserStatus status) {
        this.status = status;
        return this;
    }

    /**
     * Get the current status
     *
     * @return a valid instance of Status
     */
    @NonNull
    public ConferenceUserStatus getStatus() {
        return status;
    }

    /**
     * Check for the equality of another object
     * - ConferenceUser
     * - Participant
     * - User
     * <p>
     * In order to proceed to an update of the instance if needed internally in the SDK
     *
     * @param obj a nullable object to check against
     * @return true if both are equals or "compatible" for update
     */
    @NoDocumentation
    @Override
    public boolean equals(@Nullable Object obj) {
        if (null == id && null == userInfo) return false;
        boolean same = false;

        if (this == obj) return true;

        if (obj instanceof ConferenceUser) {
            ConferenceUser conferenceUser = (ConferenceUser) obj;
            UserInfo conferenceUserInfo = conferenceUser.getUserInfo();
            String externalId = null != userInfo ? userInfo.getExternalId() : null;
            if (null != id) same |= id.equals(conferenceUser.getUserId());
            if (null != externalId && null != conferenceUserInfo)
                same |= externalId.equals(conferenceUserInfo.getExternalId());
        }

        if (obj instanceof Participant) {
            Participant participant = (Participant) obj;
            String externalId = null != userInfo ? userInfo.getExternalId() : null;

            if (null != id) same |= id.equals(participant.getParticipantId());
            if (null != id && null != externalId)
                same |= externalId.equals(participant.getExternalId());
        }

        if (obj instanceof User) {
            User user = (User) obj;
            String externalId = null != userInfo ? userInfo.getExternalId() : null;
            UserInfo other = user.getUserInfo();
            if (null != id) same |= id.equals(user.getId());
            if (null != id && null != externalId && null != other)
                same |= externalId.equals(other.getExternalId());
        }

        return same;
    }

    /**
     * Set the current user info
     * <p>
     * Note : it does not lead to an update server side, this must be done afterward
     * Every field can be null
     *
     * @param userInfo a valid instance of information
     * @return the current user instance
     */
    @NoDocumentation
    public User setUserInfo(@NonNull UserInfo userInfo) {
        this.userInfo = userInfo;
        return this;
    }

    /**
     * Set the quality for this specific user
     *
     * @param quality
     * @deprecated
     */
    @NoDocumentation
    @Deprecated
    public void setQuality(@NonNull String quality) {
        this.quality = quality;
    }

    /**
     * Check for a required local update of the field of this current instance
     *
     * @param name      the name to set
     * @param avatarUrl the avatar url - can be null
     */
    @NoDocumentation
    public void updateIfNeeded(@Nullable String name, @Nullable String avatarUrl) {
        if (null == userInfo) {
            userInfo = new UserInfo();
        }

        if (null != name && (null == userInfo.getName() || !name.equals(userInfo.getName())))
            userInfo.setName(name);
        if (null != avatarUrl && (null == userInfo.getAvatarUrl() || avatarUrl.equals(userInfo.getAvatarUrl())))
            userInfo.setAvatarUrl(avatarUrl);
    }

    /**
     * Get the string representation of the given instance
     *
     * @return a valid string
     */
    @NoDocumentation
    @NonNull
    @Override
    public String toString() {
        return "User{" +
                "id='" + id + '\'' +
                ", userInfo=" + userInfo +
                ", status=" + status +
                ", quality='" + quality + '\'' +
                '}';
    }
}
