package com.voxeet.sdk.core.services.conference.promises;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.sdk.core.preferences.VoxeetPreferences;
import com.voxeet.sdk.core.services.AbstractPromiseable;
import com.voxeet.sdk.core.services.ConferenceSdkObservableProvider;
import com.voxeet.sdk.core.services.ConferenceService;
import com.voxeet.sdk.core.services.MediaDeviceService;
import com.voxeet.sdk.core.services.conference.information.ConferenceInformation;
import com.voxeet.sdk.events.error.HttpException;
import com.voxeet.sdk.events.success.not_to_fire.AddConferenceParticipantResultEvent;
import com.voxeet.sdk.json.SdkConferenceInvitation;
import com.voxeet.sdk.models.User;
import com.voxeet.sdk.models.v1.ConferenceUserStatus;
import com.voxeet.sdk.utils.HttpHelper;

import org.greenrobot.eventbus.EventBus;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;
import okhttp3.ResponseBody;
import retrofit2.Call;
import retrofit2.Response;

public class InvitePromise extends AbstractPromiseable<List<User>> {

    private final String TAG = InvitePromise.class.getSimpleName();

    private final List<String> ids;
    private final String conferenceId;

    public InvitePromise(@NonNull ConferenceService parent,
                         @NonNull MediaDeviceService mediaDeviceService,
                         @NonNull ConferenceSdkObservableProvider provider,
                         @Nullable ConferenceInformation information,
                         @NonNull EventBus eventBus,
                         @NonNull String conferenceId,
                         @NonNull List<String> ids) {
        super(parent, mediaDeviceService, provider, information, eventBus);

        this.ids = ids;
        this.conferenceId = conferenceId;
    }

    @NonNull
    @Override
    public Promise<List<User>> createPromise() {
        return new Promise<>(new PromiseSolver<List<User>>() {
            @Override
            public void onCall(@NonNull final Solver<List<User>> solver) {
                //remove the timeout
                removeTimeoutCallbacks();

                //a new one is now sent
                sendTimeoutCallbacks();

                boolean sdk = isSDK();
                List<String> voxeetIds = sdk ? null : new ArrayList<String>();
                final List<String> externalIds = new ArrayList<>();

                externalIds.addAll(ids);
                if (!sdk) voxeetIds.addAll(ids);

                //send an error if you're not in a conference
                if (null == conferenceId) {
                    try {
                        throw new IllegalStateException("You're not in a conference");
                    } catch (Exception e) {
                        getEventBus().post(new AddConferenceParticipantResultEvent(false));
                        solver.reject(e);
                        return;
                    }
                }

                final Call<ResponseBody> user = getProvider().getInviteObservable(conferenceId, new SdkConferenceInvitation(voxeetIds, externalIds));
                HttpHelper.enqueue(user, new HttpHelper.HttpCallback<ResponseBody>() {
                    @Override
                    public void onSuccess(@NonNull ResponseBody object, @NonNull Response<ResponseBody> response) {
                        List<User> list = new ArrayList<>();
                        getEventBus().post(new AddConferenceParticipantResultEvent(response.code() == 200));

                        try {
                            String body = response.body().string();
                            Log.d(TAG, "onNext: " + body);
                        } catch (IOException e) {
                            e.printStackTrace();
                        }

                        if (response.code() == 200) {
                            for (String userId : externalIds) {
                                if (!userId.equals(VoxeetPreferences.id())) {
                                    Log.d(TAG, "Conference participant with id: " + userId + " invited");
                                    list.add(updateConferenceParticipants(userId, ConferenceUserStatus.IN_PROGRESS));
                                }
                            }
                        }
                        solver.resolve(list);
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<ResponseBody> response) {
                        HttpException.dumpErrorResponse(response);

                        e.printStackTrace();

                        getEventBus().post(new AddConferenceParticipantResultEvent(false));
                        solver.reject(e);
                    }
                });
            }
        });
    }
}
