package com.voxeet.sdk.core.services.builders;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.sdk.core.services.conference.information.ConferenceUserType;
import com.voxeet.sdk.json.internal.MetadataHolder;
import com.voxeet.sdk.json.internal.ParamsHolder;
import com.voxeet.sdk.utils.Annotate;

/**
 * Hold and manage information to join a conference
 */
@Annotate
public class ConferenceJoinInformation {
    private String conferenceId;
    private MetadataHolder metadataHolder;
    private ParamsHolder paramsHolder;
    private ConferenceUserType conferenceUserType;

    private ConferenceJoinInformation() {

    }


    /**
     * Get the alias requested for the conference to create
     * @return the alias if set to a valid value or null
     */
    @Nullable
    public String getConferenceId() {
        return conferenceId;
    }

    /**
     * Get the various Metadata for the conference
     * @return the holder instance or null
     */
    @Nullable
    public MetadataHolder getMetadataHolder() {
        return metadataHolder;
    }

    /**
     * Get the various Parameters for the conference
     * @return the holder instance or null
     */
    @Nullable
    public ParamsHolder getParamsHolder() {
        return paramsHolder;
    }

    /**
     * Get the specified user type
     *
     * it will be used locally to set the proper information
     * @return the type requested in the builder
     */
    public ConferenceUserType getConferenceUserType() {
        return conferenceUserType;
    }

    /**
     * Build a given conference
     */
    @Annotate
    public static class Builder {
        private ConferenceJoinInformation conferenceJoinInformation;

        private Builder() {
            conferenceJoinInformation = new ConferenceJoinInformation();
        }

        /**
         * Any Builder instance needs to know the minimum information about the conference to join which is the conferenceId
         * @param conferenceId a valid conferenceId to join, e.g. after a create call
         */
        public Builder(@NonNull String conferenceId) {
            this();

            conferenceJoinInformation.conferenceId = conferenceId;
        }

        /**
         * Add metadata into the conference to create
         * @param metadataHolder the reference to the metadata
         * @return this builder instance
         */
        public Builder setMetadataHolder(@Nullable MetadataHolder metadataHolder) {
            conferenceJoinInformation.metadataHolder = metadataHolder;
            return this;
        }

        /**
         * Add parameters into the conference to create
         *
         * @param paramsHolder the reference to the parameters
         * @return this builder instance
         */
        public Builder setParamsHolder(@Nullable ParamsHolder paramsHolder) {
            conferenceJoinInformation.paramsHolder = paramsHolder;
            return this;
        }

        /**
         * Set the user type that will join the conference from a local point of view.
         *
         * Valid types can be NORMAL (by default), Broadcaster (for streaming/broadcasting modes) or Listener (no up stream, only receiving)
         *
         * @param conferenceUserType the specified type to use
         * @return this builder instance
         */
        public Builder setConferenceUserType(@NonNull ConferenceUserType conferenceUserType) {
            conferenceJoinInformation.conferenceUserType = conferenceUserType;
            return this;
        }

        /**
         * Constructs the corresponding ConferenceJoinInformation from the data set
         * @return the new valid instance of the conference join representation
         */
        @NonNull
        public ConferenceJoinInformation build() {
            return conferenceJoinInformation;
        }

    }
}
