package com.voxeet.sdk.core.services.localstats;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.sdk.core.services.localstats.events.LocalStatsUserChangeEvent;
import com.voxeet.stats.LocalStats;
import com.voxeet.stats.RTCInboundRTPStreamStats;

import org.greenrobot.eventbus.EventBus;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class LocalStatsUserInfo {
    private final static String TYPE_AUDIO = "audio";
    private final static String TYPE_VIDEO = "video";

    private final static long MAXIMUM_DIFFERENCE_MS = 50000;
    private static final String TAG = LocalStatsUserInfo.class.getSimpleName();
    private String mUserId;
    private HashMap<String, List<TimePoint>> mPacketReceived;
    private HashMap<String, List<TimePoint>> mBytesReceived;
    private List<TimePoint> mPacketOverallReceived;
    private LocalStatsUserTypes currentType;

    public LocalStatsUserInfo(String userId) {
        this.mUserId = userId;
        mPacketReceived = new HashMap<>();
        mBytesReceived = new HashMap<>();
        mPacketOverallReceived = new ArrayList<>();
        currentType = LocalStatsUserTypes.DEFAULT;
    }

    public void addFromLocalStats(@Nullable LocalStats stats) {
        long added = 0;

        if (null != stats) {
            List<RTCInboundRTPStreamStats> inbounds = stats.getStatsForClass(RTCInboundRTPStreamStats.class);

            for (RTCInboundRTPStreamStats inbound : inbounds) {
                String type = inbound.mediaType;

                if (null != type) {
                    getPacketReceived(type).add(new TimePoint(System.currentTimeMillis(), inbound.packetsReceived));
                    getBytesReceived(type).add(new TimePoint(System.currentTimeMillis(), inbound.bytesReceived));

                    added += inbound.packetsReceived;
                }
            }
        }

        mPacketOverallReceived.add(new TimePoint(System.currentTimeMillis(), added));

        purge();
        checkUserState();
    }

    public LocalStatsUserTypes getCurrentType() {
        return currentType;
    }

    public boolean isDisconnected() {
        List<String> keys = new ArrayList<>();
        keys.addAll(mPacketReceived.keySet());

        if (keys.size() == 0) return true;
        for (String key : keys) {
            if (!isDisconnected(mPacketReceived.get(key))) {
                return false;
            }
        }

        return true;
    }

    public boolean isFluctuating() {
        List<String> keys = new ArrayList<>();
        keys.addAll(mPacketReceived.keySet());

        if (keys.size() == 0) return true;
        for (String key : keys) {
            if (!isFluctuating(mPacketReceived.get(key))) {
                return false;
            }
        }

        return true;
    }

    private boolean isFluctuating(@Nullable List<TimePoint> list) {
        if (null == list) return false;
        return isListDisconnectedForTime(list, 1000);
    }

    private boolean isDisconnected(@Nullable List<TimePoint> list) {
        if (null == list) return false;
        return isListDisconnectedForTime(list, 3000);
    }

    private boolean isListDisconnectedForTime(@NonNull List<TimePoint> list, long time) {
        long past = System.currentTimeMillis() - time;
        long minPacketCount = -1;
        long maxPacketCount = -1;
        boolean treshold = false;

        if (list.size() > 1) {
            int index = list.size() - 1;
            TimePoint point;
            while (index >= 0 && !treshold) {
                point = list.get(index);

                if (point.getTime() > past) {
                    if (minPacketCount == -1 || minPacketCount > point.getPackets())
                        minPacketCount = point.getPackets();
                    if (maxPacketCount < point.getPackets()) maxPacketCount = point.getPackets();
                } else {
                    treshold = true;
                }
                index--;
            }

            return !(minPacketCount >= 0 && maxPacketCount > minPacketCount);
        }

        return true;
    }

    @NonNull
    private List<TimePoint> getPacketReceived(@NonNull String type) {
        return getTimeList(type, mPacketReceived);
    }

    @NonNull
    private List<TimePoint> getBytesReceived(@NonNull String type) {
        return getTimeList(type, mBytesReceived);
    }

    @NonNull
    private List<TimePoint> getTimeList(@NonNull String type, @NonNull HashMap<String, List<TimePoint>> map) {
        synchronized (map) {
            List<TimePoint> holder = map.get(type);
            if (null == holder) {
                holder = new ArrayList<>();
                map.put(type, holder);
            }
            return holder;
        }
    }

    private void purge() {
        removeInMap(mBytesReceived);
        removeInMap(mPacketReceived);
        removeInList(mPacketOverallReceived);
    }

    private void removeInMap(@NonNull Map<String, List<TimePoint>> map) {
        synchronized (map) {
            Set<String> keys = map.keySet();

            for (String key : keys) {
                List<TimePoint> list = map.get(key);
                removeInList(list);
            }
        }
    }

    private void removeInList(@NonNull List<TimePoint> list) {
        int i = 0;
        boolean finished = false;
        long maximum_in_past = System.currentTimeMillis() - MAXIMUM_DIFFERENCE_MS;
        while (i < list.size() && !finished) {
            if (list.get(i).getTime() < maximum_in_past) {
                list.remove(i);
            } else {
                finished = true;
                i++;
            }
        }
    }

    private void checkUserState() {
        boolean isDisconnected = isDisconnected();
        boolean isFluctuating = isFluctuating();
        LocalStatsUserTypes new_type = currentType.next(isDisconnected, isFluctuating);

        if (null != new_type) {
            LocalStatsUserTypes current = currentType;
            currentType = new_type;

            EventBus.getDefault().post(
                    new LocalStatsUserChangeEvent(mUserId,
                            this,
                            current,
                            new_type)
            );
        }
    }
}
