package com.voxeet.sdk.core.services;

import android.content.Context;
import android.content.Intent;
import android.os.Build;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.util.Log;

import com.voxeet.android.media.MediaEngine;
import com.voxeet.android.media.MediaEngineException;
import com.voxeet.android.media.MediaStream;
import com.voxeet.android.media.MediaStreamType;
import com.voxeet.android.media.SdpCandidate;
import com.voxeet.audio.AudioRoute;
import com.voxeet.authent.models.DeviceType;
import com.voxeet.push.center.NotificationCenterFactory;
import com.voxeet.push.center.management.Constants;
import com.voxeet.sdk.core.VoxeetSdk;
import com.voxeet.sdk.core.network.endpoints.ISdkConferenceRService;
import com.voxeet.sdk.core.preferences.VoxeetPreferences;
import com.voxeet.sdk.core.services.builders.ConferenceCreateInformation;
import com.voxeet.sdk.core.services.builders.ConferenceJoinInformation;
import com.voxeet.sdk.core.services.conference.information.ConferenceInformation;
import com.voxeet.sdk.core.services.conference.information.ConferenceInformationHolder;
import com.voxeet.sdk.core.services.conference.information.ConferenceState;
import com.voxeet.sdk.core.services.conference.information.ConferenceUserType;
import com.voxeet.sdk.core.services.conference.information.LocalConferenceType;
import com.voxeet.sdk.core.services.conference.promises.CreateConferencePromiseable;
import com.voxeet.sdk.core.services.conference.promises.DeclinePromise;
import com.voxeet.sdk.core.services.conference.promises.GetConferenceHistoryPromiseable;
import com.voxeet.sdk.core.services.conference.promises.GetConferenceStatus;
import com.voxeet.sdk.core.services.conference.promises.GetConferenceStatusPromiseable;
import com.voxeet.sdk.core.services.conference.promises.InvitePromise;
import com.voxeet.sdk.core.services.conference.promises.JoinPromise;
import com.voxeet.sdk.core.services.conference.promises.LeavePromise;
import com.voxeet.sdk.core.services.conference.promises.ReplayPromise;
import com.voxeet.sdk.core.services.conference.promises.StartScreensharePromise;
import com.voxeet.sdk.core.services.conference.promises.StartVideoPromise;
import com.voxeet.sdk.core.services.conference.promises.StopScreenSharePromise;
import com.voxeet.sdk.core.services.conference.promises.StopVideoPromise;
import com.voxeet.sdk.core.services.conference.promises.SubscribeConferenceEventPromiseable;
import com.voxeet.sdk.core.services.conference.promises.SubscribeForCallStartPromiseable;
import com.voxeet.sdk.core.services.conference.promises.UnsubscribeConferenceEventPromiseable;
import com.voxeet.sdk.core.services.conference.promises.UnsubscribeForCallStartPromiseable;
import com.voxeet.sdk.core.services.holder.ServiceProviderHolder;
import com.voxeet.sdk.events.error.HttpException;
import com.voxeet.sdk.events.error.ParticipantAddedErrorEvent;
import com.voxeet.sdk.events.error.PermissionRefusedEvent;
import com.voxeet.sdk.events.promises.InConferenceException;
import com.voxeet.sdk.events.promises.NotInConferenceException;
import com.voxeet.sdk.events.promises.PromiseConferenceJoinedErrorException;
import com.voxeet.sdk.events.promises.PromiseParticipantAddedErrorEventException;
import com.voxeet.sdk.events.restapi.ConferenceStatusResult;
import com.voxeet.sdk.events.restapi.ResumeConference;
import com.voxeet.sdk.events.sdk.ConferenceStateEvent;
import com.voxeet.sdk.events.sdk.ConferenceUserCallDeclinedEvent;
import com.voxeet.sdk.events.sdk.ConferenceUserQualityUpdatedEvent;
import com.voxeet.sdk.events.sdk.ConferenceUsersInvitedResult;
import com.voxeet.sdk.events.sdk.DeclineConferenceResultEvent;
import com.voxeet.sdk.events.sdk.GetConferenceHistoryResult;
import com.voxeet.sdk.events.sdk.IncomingCallEvent;
import com.voxeet.sdk.events.sdk.SocketConnectEvent;
import com.voxeet.sdk.events.v2.StreamAddedEvent;
import com.voxeet.sdk.events.v2.StreamRemovedEvent;
import com.voxeet.sdk.events.v2.StreamUpdatedEvent;
import com.voxeet.sdk.events.v2.UserAddedEvent;
import com.voxeet.sdk.events.v2.UserUpdatedEvent;
import com.voxeet.sdk.events.websocket.RenegociationEndedEvent;
import com.voxeet.sdk.exceptions.ExceptionManager;
import com.voxeet.sdk.factories.VoxeetIntentFactory;
import com.voxeet.sdk.json.ConferenceDestroyedPush;
import com.voxeet.sdk.json.ConferenceEnded;
import com.voxeet.sdk.json.InvitationReceivedEvent;
import com.voxeet.sdk.json.JoinParameters;
import com.voxeet.sdk.json.OfferCreated;
import com.voxeet.sdk.json.OwnConferenceCreated;
import com.voxeet.sdk.json.OwnConferenceUserSwitch;
import com.voxeet.sdk.json.ParticipantAdded;
import com.voxeet.sdk.json.ParticipantUpdated;
import com.voxeet.sdk.json.QualityUpdated;
import com.voxeet.sdk.json.UserInfo;
import com.voxeet.sdk.json.internal.MetadataHolder;
import com.voxeet.sdk.json.internal.ParamsHolder;
import com.voxeet.sdk.media.MediaSDK;
import com.voxeet.sdk.media.audio.UserPosition;
import com.voxeet.sdk.media.camera.CameraContext;
import com.voxeet.sdk.media.peer.PendingPeerCallback;
import com.voxeet.sdk.media.peer.SdpDescription;
import com.voxeet.sdk.media.peer.SdpMessage;
import com.voxeet.sdk.models.Conference;
import com.voxeet.sdk.models.User;
import com.voxeet.sdk.models.v1.CandidatesPush;
import com.voxeet.sdk.models.v1.ConferenceType;
import com.voxeet.sdk.models.v1.ConferenceUser;
import com.voxeet.sdk.models.v1.ConferenceUserStatus;
import com.voxeet.sdk.models.v1.CreateConferenceResult;
import com.voxeet.sdk.models.v1.Invitation;
import com.voxeet.sdk.models.v1.OfferCandidate;
import com.voxeet.sdk.models.v1.OfferDescription;
import com.voxeet.sdk.models.v1.Participant;
import com.voxeet.sdk.models.v1.SdkParticipant;
import com.voxeet.sdk.utils.AndroidManifest;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.ConferenceUtils;
import com.voxeet.sdk.utils.HttpHelper;
import com.voxeet.sdk.utils.NoDocumentation;
import com.voxeet.sdk.utils.PeerInformation;
import com.voxeet.sdk.utils.Validate;

import org.greenrobot.eventbus.EventBus;
import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;
import org.webrtc.CameraVideoCapturer;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.locks.ReentrantLock;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.ErrorPromise;
import eu.codlab.simplepromise.solve.PromiseExec;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;
import okhttp3.ResponseBody;
import retrofit2.Call;
import retrofit2.Callback;
import retrofit2.Response;

/**
 * Conference SDK Service
 * <p>
 * Warning : always implements the error for each promises you may call to start resolve them
 * You can also use the execute method but in case of Exception, you will trigger errors...
 */
@Annotate
public class ConferenceService extends com.voxeet.sdk.core.AbstractVoxeetService<ISdkConferenceRService> {

    private final static String TAG = ConferenceService.class.getSimpleName();
    private final VoxeetSdk mInstance;
    private VoxeetSdk mSDK;
    private ConferenceSdkObservableProvider mConferenceObservableProvider;

    private EventBus mEventBus;
    private boolean mInConference = false;
    private long mTimeOutTimer = -1;
    private ReentrantLock joinLock = new ReentrantLock();

    private boolean isDefaultOnSpeaker;

    private String mConferenceId = null;
    private ConferenceInformationHolder mConferenceInformationHolder = new ConferenceInformationHolder();


    private ExecutorService executorService = Executors.newCachedThreadPool();

    private Context mContext;

    private @Nullable
    TimeoutRunnable timeoutRunnable = null;

    /**
     * Can be used using the metadata in your app manifest :
     * <meta-data android:name="voxeet_automatic_telecom_conference_alias_prefix"
     * android:value="someprefix" />
     */
    private String automaticTelecomModePrefix = null;

    private boolean telecomMode = false;

    private MediaEngine.StreamListener mediaStreamListener = new MediaEngine.StreamListener() {
        @Override
        public void onStreamAdded(@NonNull final String peer, @NonNull final MediaStream stream) {
            PeerInformation peerInformation = new PeerInformation(peer, MediaStreamType.Camera);
            onStreamAddedFromInformation(peerInformation, stream);
        }

        @Override
        public void onStreamUpdated(@NonNull final String peer, @NonNull final MediaStream stream) {
            PeerInformation peerInformation = new PeerInformation(peer, MediaStreamType.Camera);
            onStreamUpdatedFromInformation(peerInformation, stream);
        }

        @Override
        public void onStreamRemoved(@NonNull final String peer) {
            PeerInformation peerInformation = new PeerInformation(peer, MediaStreamType.Camera);
            onStreamRemovedFromInformation(peerInformation);
        }

        @Override
        public void onScreenStreamAdded(@NonNull final String peer, @NonNull final MediaStream stream) {
            PeerInformation peerInformation = new PeerInformation(peer, MediaStreamType.ScreenShare);
            onStreamAddedFromInformation(peerInformation, stream);
        }

        @Override
        public void onScreenStreamRemoved(@NonNull String peer) {
            PeerInformation peerInformation = new PeerInformation(peer, MediaStreamType.ScreenShare);
            onStreamRemovedFromInformation(peerInformation);
        }

        private void onStreamAddedFromInformation(@NonNull final PeerInformation peerInformation, @NonNull MediaStream stream) {
            String peer = peerInformation.peerId;
            Log.d("DUMP_INFORMATION", "onStreamAdded: stream for peer " + peer + " " + stream.getType());
            postOnMainThread(new Runnable() {
                @Override
                public void run() {
                    ConferenceInformation conferenceInformation = getCurrentConferenceInformation();

                    User user = findUserById(peer);
                    Log.d("DUMP_INFORMATION", "New stream for user " + peer + " user := " + user);

                    if (null != conferenceInformation && null == user) {
                        Log.d(TAG, "run: WARNING obtained stream for user which did not existed " + peer);
                        user = new User(peer, null);
                        conferenceInformation.getConference().updateUser(user);
                    }

                    if (user != null) {
                        if (!peer.equalsIgnoreCase(VoxeetSdk.session().getUserId()) && mTimeOutTimer != -1) {
                            removeTimeoutCallbacks();
                        }

                        //insert or update -> but in the context, add
                        user.streamsHandler().insertOrUpdate(stream);

                        //update the current conference state specificly from the users point of view
                        updateConferenceFromUsers();
                        mEventBus.post(new StreamAddedEvent(conferenceInformation.getConference(),
                                user, stream));
                    } else {
                        Log.d(TAG, "run: unknown user in stream added");
                    }
                }
            });
        }

        private void onStreamUpdatedFromInformation(@NonNull final PeerInformation peerInformation, @NonNull final MediaStream stream) {
            String peer = peerInformation.peerId;
            Log.d("DUMP_INFORMATION", "onStreamUpdated: screen for peer " + peer);
            postOnMainThread(new Runnable() {
                @Override
                public void run() {
                    ConferenceInformation conferenceInformation = getCurrentConferenceInformation();
                    User user = findUserById(peer);

                    if (null != conferenceInformation && null == user) {
                        Log.d("DUMP_INFORMATION", "run: WARNING obtained stream for user which did not existed " + peer);
                        user = new User(peer, null);
                        conferenceInformation.getConference().updateUser(user);
                    }
                    Log.d(TAG, "Updated stream for user " + user);

                    if (user != null) {
                        //insert or update -> but in the context, add
                        user.streamsHandler().insertOrUpdate(stream);

                        //update the current conference state specificly from the users point of view
                        updateConferenceFromUsers();
                        mEventBus.post(new StreamUpdatedEvent(conferenceInformation.getConference(),
                                user, stream));
                    } else {
                        Log.d(TAG, "run: unknown user in stream updated");
                    }
                }
            });
        }

        public void onStreamRemovedFromInformation(@NonNull final PeerInformation peerInformation) {
            String peer = peerInformation.peerId;
            Log.d("DUMP_INFORMATION", "onStreamRemoved: OnStreamRemoved");
            postOnMainThread(new Runnable() {
                @Override
                public void run() {
                    removeStreamForUser(peer, peerInformation.calculatedMediaStreamType);
                }
            });
        }

        @Override
        public void onShutdown() {

        }

        @Override
        public void onIceCandidateDiscovered(String peer, SdpCandidate[] candidates) {

            if (null == getConferenceId()) return;

            Call<ResponseBody> call = mConferenceObservableProvider.candidates(getConferenceId(),
                    peer,
                    new CandidatesPush(candidates));

            call.enqueue(new Callback<ResponseBody>() {
                @Override
                public void onResponse(Call<ResponseBody> call, Response<ResponseBody> response) {
                    Log.d(TAG, "onResponse: Candidates sent ");
                }

                @Override
                public void onFailure(Call<ResponseBody> call, Throwable t) {
                    t.printStackTrace();
                }
            });
        }
    };

    private void removeStreamForUser(@NonNull String peer, MediaStreamType mediaStreamType) {
        ConferenceInformation conferenceInformation = getCurrentConferenceInformation();
        User user = findUserById(peer);

        if (null != conferenceInformation && null == user) {
            Log.d(TAG, "run: WARNING obtained stream for user which did not existed " + peer);
            user = new User(peer, null);
            conferenceInformation.getConference().updateUser(user);
        }
        Log.d(TAG, "Remove stream " + mediaStreamType + " for user " + user);

        if (user != null) {
            //insert or update -> but in the context, add
            MediaStream stream = user.streamsHandler().getFirst(mediaStreamType);
            user.streamsHandler().remove(mediaStreamType);

            //update the current conference state specificly from the users point of view
            updateConferenceFromUsers();

            mEventBus.post(new StreamRemovedEvent(conferenceInformation.getConference(), user, stream));
        } else {
            Log.d(TAG, "run: unknown user in stream updated" + Arrays.toString(getConferenceUsers().toArray()));
        }
    }

    private CameraVideoCapturer.CameraEventsHandler cameraEventsHandler = new CameraVideoCapturer.CameraEventsHandler() {
        @Override
        public void onCameraError(String s) {
            Log.d(TAG, "onCameraError: error...");

            stopVideo().then(new PromiseExec<Boolean, Object>() {
                @Override
                public void onCall(@Nullable Boolean result, @NonNull Solver<Object> solver) {
                    Log.d(TAG, "onCall: stopped video after camera issue " + result);
                }
            }).error(new ErrorPromise() {
                @Override
                public void onError(@NonNull Throwable error) {
                    error.printStackTrace();
                }
            });
        }

        @Override
        public void onCameraDisconnected() {
            Log.d(TAG, "onCameraDisconnected");
        }

        @Override
        public void onCameraFreezed(String s) {
            Log.d(TAG, "onCameraFreezed: " + s);
        }

        @Override
        public void onCameraOpening(String s) {
            Log.d(TAG, "onCameraOpening: " + s);
        }

        @Override
        public void onFirstFrameAvailable() {
            Log.d(TAG, "onFirstFrameAvailable");
        }

        @Override
        public void onCameraClosed() {
            Log.d(TAG, "onCameraClosed");
        }
    };

    private boolean isDefaultVideo = false;
    private boolean isDefaultMute = false;
    private boolean isICERestartEnabled = false;

    @NoDocumentation
    public ConferenceService(VoxeetSdk instance, long timeout) {
        super(instance, new ServiceProviderHolder.Builder<ISdkConferenceRService>()
                .setRetrofit(instance.getRetrofit())
                .setService(ISdkConferenceRService.class)
                .setEventBus(instance.getEventBus())
                .setClient(instance.getClient())
                .build());

        mInstance = instance;

        mConferenceObservableProvider = new ConferenceSdkObservableProvider();
        mConferenceObservableProvider.setRetrofitInstantiatedProvider(getService());
        mSDK = instance;
        mTimeOutTimer = timeout;
        mEventBus = EventBus.getDefault();
        mContext = instance.getApplicationContext();
        setDefaultBuiltInSpeaker(true);


        /**
         * Can be used using the metadata in your app manifest :
         * <meta-data android:name="voxeet_automatic_telecom_conference_alias_prefix"
         * android:value="someprefix" />
         */
        automaticTelecomModePrefix = AndroidManifest.readMetadata(mContext, "voxeet_automatic_telecom_conference_alias_prefix", null);


        registerEventBus();
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Public management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    /**
     * Mute or unmute the current user
     *
     * @param mute the new unmute/mute state
     * @return if the state can be changed
     */
    public boolean mute(boolean mute) {
        MediaDeviceService service = VoxeetSdk.mediaDevice();
        MediaEngine media = null != service ? service.getMedia() : null;
        if (null != media) {
            if (!mute && media.isMuted()) {
                if (!Validate.hasMicrophonePermissions(context)) {
                    getEventBus().post(new PermissionRefusedEvent(PermissionRefusedEvent.Permission.MICROPHONE));
                    return false;
                } else {
                    media.unMute();
                }
            } else if (mute) {
                media.mute();
            }
        }
        return true;
    }

    /**
     * Get the current muted state
     *
     * @return the muted state, false can indicate an invalid current conference
     */
    public boolean isMuted() {
        MediaEngine media = VoxeetSdk.mediaDevice().getMedia();
        return null == media || media.isMuted();
    }

    @NoDocumentation
    @Deprecated
    public boolean isUserMuted(String userId) {
        //TODO muted management
        //User user = findUserById(userId);
        //return user != null && user.isMuted();

        return false;
    }

    /**
     * If a current conference is started, automatically manage the toggle state.
     *
     * @deprecated Deprecated method, please handle it on developer side
     */
    @Deprecated
    public void toggleVideo() {
        Promise<Boolean> promise = isVideoOn() ? stopVideo() : startVideo();
        promise.then(new PromiseExec<Boolean, Object>() {
            @Override
            public void onCall(@Nullable Boolean result, @NonNull Solver<Object> solver) {
                Log.d(TAG, "onSuccess: toggleVideo " + result);
            }
        }).error(manageError());
    }

    /**
     * If a screenshare has been started by the current user, automatically stops it
     *
     * @deprecated Deprecated method, please handle it on developer side
     */
    @Deprecated
    public void toggleScreenShare() {
        if (isScreenShareOn()) {
            stopScreenShare().then(new PromiseExec<Boolean, Object>() {
                @Override
                public void onCall(@Nullable Boolean result, @NonNull Solver<Object> solver) {
                    Log.d(TAG, "onSuccess: toggleScreenShare " + result);
                }
            }).error(manageError());
        } else {
            VoxeetSdk.screenShare().sendRequestStartScreenShare();
        }
    }

    /**
     * Get the current active speaker
     *
     * @return the current active speaker's id or null if none
     */
    @Nullable
    public String currentSpeaker() {
        MediaEngine media = VoxeetSdk.mediaDevice().getMedia();
        Conference conference = getConference();
        if (null == media || conference == null) {
            return VoxeetSdk.session().getUserId();
        } else {
            String currentSpeaker = null;
            for (User user : conference.getUsers()) {
                if (user.getId() != null
                        && !user.getId().equals(VoxeetPreferences.id())
                        && user.isLocallyActive()) {
                    double peerVuMeter = media.getPeerVuMeter(user.getId());
                    if (Double.isNaN(peerVuMeter)) peerVuMeter = 0;

                    if (currentSpeaker == null || (peerVuMeter > 0.001 && media.getPeerVuMeter(currentSpeaker) < peerVuMeter))
                        currentSpeaker = user.getId();
                }
            }
            return currentSpeaker;
        }
    }

    @NonNull
    private String currentUserOrEmpty() {
        String currentUserId = VoxeetPreferences.id();
        return null != currentUserId ? currentUserId : "";
    }

    /**
     * Given any userId get the corresponding in conference user or null if :
     * - the conference does not exist for the current time session
     * - the user in this session does not exists
     *
     * @param userId the user id to check upon
     * @return the instance of the User or null if none
     */
    @Nullable
    public User findUserById(final String userId) {
        Conference conference = getConference();
        return null != conference ? conference.findUserById(userId) : null;
    }

    /**
     * Set the current mode to telecom. If any conference has already been started, also update its state to be the same
     * A telecom mode will automatically stop any conference locally (and send a leave to the server) as soon as a user disconnects
     *
     * @param telecomMode the new telecommunication mode
     */
    public void setTelecomMode(boolean telecomMode) {
        this.telecomMode = telecomMode;

        ConferenceInformation currentConference = getCurrentConferenceInformation();
        if (null != currentConference) {
            currentConference.setTelecomMode(telecomMode);
        }
    }

    /**
     * Check the internal telecom mode for the conferences
     * It does not reflect the telecom mode for a specific live conference. It must be checked when needed directly from
     * the ConferenceInformation or the Conference
     *
     * @return the telecom mode that will be set to conferences
     */
    public boolean isTelecomMode() {
        return telecomMode;
    }

    /**
     * Can be used using the metadata in your app manifest :
     * <meta-data android:name="voxeet_automatic_telecom_conference_alias_prefix"
     * android:value="someprefix" />
     */
    @Nullable
    public String getAutomaticTelecomModePrefix() {
        return automaticTelecomModePrefix;
    }

    @Nullable
    public String getConferenceId() {
        return mConferenceId;
    }

    public long getTimeout() {
        return mTimeOutTimer;
    }

    @Deprecated
    public boolean muteUser(@NonNull String userId, boolean shouldMute) {
        /*Validate.runningOnUiThread();

        User user = findUserById(userId);
        if (user != null) {
            user.setMuted(shouldMute);
            getMedia().changePeerGain(userId, shouldMute ? ConferenceUtils.MUTE_FACTOR : ConferenceUtils.UNMUTE_FACTOR);
        }
        return user != null;*/

        return false;
    }

    @Deprecated
    public boolean isVideoOn() {
        ConferenceInformation information = getCurrentConferenceInformation();
        return null != information && information.isOwnVideoStarted();
    }

    @Deprecated
    public boolean isScreenShareOn() {
        ConferenceInformation information = getCurrentConferenceInformation();
        return null != information && information.isScreenShareOn();
    }

    @Deprecated
    public boolean setDefaultVideo(boolean default_state) {
        //isDefaultVideo = default_state;
        //return true;
        return false;
    }

    public boolean setDefaultMute(boolean default_state) {
        isDefaultMute = default_state;
        return true;
    }

    public boolean setDefaultBuiltInSpeaker(boolean default_state) {
        isDefaultOnSpeaker = default_state;
        VoxeetPreferences.setDefaultBuiltInSpeakerOn(default_state);
        return true;
    }

    public boolean setTimeOut(long timeout) {
        mTimeOutTimer = timeout;
        return true;
    }

    @Deprecated
    public boolean setUserPosition(final String userId, final double angle, final double distance) {
        MediaEngine media = VoxeetSdk.mediaDevice().getMedia();
        if (null != media) {
            //TODO possible improvement here in case of recovery
            ConferenceInformation conference = getCurrentConferenceInformation();
            if (null != conference) {
                UserPosition position = conference.getPosition(userId);
                position.angle = angle;
                position.distance = distance;
                conference.setPosition(userId, position);
            }
            media.changePeerPosition(userId, angle, distance);
            return true;
        }
        return false;
    }

    /**
     * Get the current conference type
     *
     * @return a ConferenceType, ConferenceType.NONE is the default value
     */
    @NonNull
    public LocalConferenceType getConferenceType() {
        ConferenceInformation information = getCurrentConferenceInformation();
        if (null == information || null == mConferenceId) return LocalConferenceType.NONE;
        return information.getConferenceType();
    }

    /**
     * Get the current mConference object
     *
     * @return a nullable object
     */
    @Nullable
    public Conference getConference() {
        ConferenceInformation information = getCurrentConferenceInformation();
        if (null == information) return null;

        return information.getConference();
    }

    /**
     * Get the current state of joining
     * in conjunction of the getConference() it is easy to check the current state :
     * <p>
     * mInConference() + getConference() non null = mConference joined and on air
     * mInConference() + getConference() null = joining mConference with the answer from server for now
     * !mInConference() = not currently joining, joined mConference (or left)
     *
     * @return if in mConference or "attempting" to join
     */
    public boolean isInConference() {
        return mInConference;
    }

    void setIsInConference(boolean status) {
        mInConference = status;
    }

    public double getPeerVuMeter(@Nullable String peerId) {
        Validate.runningOnUiThread();
        MediaEngine media = VoxeetSdk.mediaDevice().getMedia();

        if (null != media && null != peerId) return media.getPeerVuMeter(peerId);
        return 0.;
    }

    @NonNull
    public List<User> getConferenceUsers() {
        Conference conference = getConference();
        if (null == conference) {
            Log.d(TAG, "getConferenceUsers: returning a new instance :: NOT IN A CONFERENCE");
            return new ArrayList<>();
        }

        return conference.getUsers();
    }

    @NonNull
    public List<User> getLastInvitationUsers() {
        ConferenceInformation information = getCurrentConferenceInformation();
        if (null == information) return new ArrayList<>();
        return information.getLastInvitationReceived();
    }

    public boolean isLive() {
        return null != getConference();
    }

    /**
     * Give the ability to cancel the timeout in some cases
     */
    @NonNull
    public ConferenceService cancelTimeout() {
        removeTimeoutCallbacks();

        return this;
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Public Promises management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    /**
     * resolve -> the result event
     * reject -> network error
     *
     * @param conferenceId the conference id to decline
     * @return a promise to resolve
     */
    @NonNull
    public Promise<DeclineConferenceResultEvent> decline(final String conferenceId) {
        return new DeclinePromise(this,
                VoxeetSdk.mediaDevice(),
                mConferenceObservableProvider,
                getConferenceInformation(conferenceId),
                getEventBus(),
                conferenceId).createPromise();
    }

    /**
     * Call te
     */
    @Deprecated
    public void sendRequestStartScreenShare() {
        VoxeetSdk.screenShare().sendRequestStartScreenShare();
    }

    public void onUserCanceledScreenShare() {
        //call before any specific code so stopScreenCapturer() should do nothing /!\
        MediaEngine media = VoxeetSdk.mediaDevice().getMedia();
        if (null != media) {
            media.stopScreenCapturer();
        }
        ConferenceInformation information = getCurrentConferenceInformation();
        if (null != information) {
            information.setScreenShareOn(false);
        }
    }

    @NonNull
    public Promise<Boolean> startScreenShare(@NonNull final Intent intent,
                                             final int width,
                                             final int height) {
        return new StartScreensharePromise(this,
                VoxeetSdk.mediaDevice(),
                mConferenceObservableProvider,
                getCurrentConferenceInformation(),
                getEventBus(),
                intent,
                width,
                height).createPromise();
    }

    /**
     * resolve -> StopVideoAnswerEvent
     * reject -> MediaException
     * reject -> NotInConferenceException
     */
    @NonNull
    public Promise<Boolean> stopScreenShare() {
        return new StopScreenSharePromise(this,
                VoxeetSdk.mediaDevice(),
                mConferenceObservableProvider,
                getCurrentConferenceInformation(),
                getEventBus()).createPromise();
    }

    /**
     * Start the video
     * reject -> PromisePermissionRefusedEventException
     * reject -> MediaException
     * <p>
     * If the app does not have the permission, will fire a PermissionRefusedEvent
     */
    @NonNull
    public Promise<Boolean> startVideo() {
        CameraContext provider = VoxeetSdk.mediaDevice().getCameraContext();
        return startVideo(provider.isDefaultFrontFacing());
    }

    /**
     * Start the video
     * reject -> PromisePermissionRefusedEventException
     * reject -> MediaException
     * <p>
     * If the app does not have the permission, will fire a PermissionRefusedEvent
     */
    @NonNull
    public Promise<Boolean> startVideo(boolean isDefaultFrontFacing) {
        return new StartVideoPromise(this,
                VoxeetSdk.mediaDevice(),
                mConferenceObservableProvider,
                getCurrentConferenceInformation(),
                getEventBus(),
                isDefaultFrontFacing).createPromise();
    }

    /**
     * resolve -> StopVideoAnswerEvent
     * reject -> MediaException
     * reject -> NotInConferenceException
     */
    public Promise<Boolean> stopVideo() {
        return new StopVideoPromise(this,
                VoxeetSdk.mediaDevice(),
                mConferenceObservableProvider,
                getCurrentConferenceInformation(),
                getEventBus()).createPromise();
    }

    /**
     * Create a video answer and post it
     * <p>
     * resolve -> true -> everything went fine
     * resolve -> false -> on error occured
     * reject -> media exception
     *
     * @param userId
     * @param offerDescription
     * @param offerCandidates
     * @return
     */
    Promise<Boolean> createVideoAnswer(@NonNull final String userId, @Nullable final OfferDescription offerDescription, final List<OfferCandidate> offerCandidates) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                MediaSDK media = VoxeetSdk.mediaDevice().getMedia();
                String sdp = "";
                String type = "";
                if (null != offerDescription) {
                    sdp = offerDescription.sdp;
                    type = offerDescription.type;
                }

                try {
                    if (null == media) {
                        throw new MediaEngineException("media is null");
                    }
                } catch (MediaEngineException exception) {
                    solver.reject(exception);
                }

                if (null == offerDescription) {
                    stopVideo().execute();
                    Log.d(TAG, "onCall: unable to start video, are you a listener?");
                    solver.resolve(false);
                    return;
                }


                SdpDescription description = new SdpDescription(type, sdp);

                List<SdpCandidate> candidates = new ArrayList<>();
                if (offerCandidates != null) {
                    for (OfferCandidate candidate : offerCandidates) {
                        candidates.add(new SdpCandidate(candidate.getMid(), Integer.parseInt(candidate.getmLine()), candidate.getSdp()));
                    }
                }

                //TODO use the promise.resolve()
                createAnswerForPeerThreaded(media, userId, description.ssrc, description, candidates, userId.equals(VoxeetSdk.session().getUserId()))
                        .then(new PromiseExec<SdpMessage, Object>() {
                            @Override
                            public void onCall(@Nullable SdpMessage message, @NonNull Solver<Object> internal_solver) {
                                Log.d(TAG, "onMessage: having a message " + message);
                                answer(userId, message)
                                        .then(new PromiseExec<Integer, Object>() {
                                            @Override
                                            public void onCall(@Nullable Integer result, @NonNull Solver<Object> internal_solver) {
                                                Log.d(TAG, "onSuccess: " + result);

                                                //resolve true value : all clear
                                                solver.resolve(true);
                                            }
                                        })
                                        .error(new ErrorPromise() {
                                            @Override
                                            public void onError(Throwable error) {
                                                if (error instanceof PromiseParticipantAddedErrorEventException)
                                                    mEventBus.post(((PromiseParticipantAddedErrorEventException) error).getEvent());
                                                else
                                                    error.printStackTrace();

                                                //resolve false value : something happened
                                                solver.resolve(false);
                                            }
                                        });
                            }
                        })
                        .error(new ErrorPromise() {
                            @Override
                            public void onError(@NonNull Throwable error) {
                                error.printStackTrace();
                                solver.reject(error);
                            }
                        });
            }
        });
    }

    /**
     * resolve -> true
     * resolve -> false
     * reject -> exception
     *
     * @param conferenceAlias create a given conference then join it
     * @return a promise to resolve
     */
    public Promise<CreateConferenceResult> create(@Nullable final String conferenceAlias) {
        return create(new ConferenceCreateInformation.Builder()
                .setConferenceAlias(conferenceAlias).build()
        );
    }

    /**
     * resolve -> true
     * resolve -> false
     * reject -> exception
     *
     * @param conferenceAlias create a given conference then join it
     * @param metadata
     * @param paramsholder
     * @return a promise to resolve
     * @deprecated use {@link #create(ConferenceCreateInformation)} instead.
     */
    @Deprecated
    public Promise<CreateConferenceResult> create(@Nullable String conferenceAlias,
                                                  @Nullable MetadataHolder metadata,
                                                  @Nullable ParamsHolder paramsholder) {
        return create(new ConferenceCreateInformation.Builder()
                .setConferenceAlias(conferenceAlias)
                .setMetadataHolder(metadata)
                .setParamsHolder(paramsholder)
                .build()
        );
    }

    /**
     * Create a conference based on information from the built holder
     *
     * @param conferenceCreateInformation the information holder where id, params and metadata can be passed into
     * @return a conference creation information holder
     */
    public Promise<CreateConferenceResult> create(@NonNull ConferenceCreateInformation conferenceCreateInformation) {
        return new CreateConferencePromiseable(this,
                VoxeetSdk.mediaDevice(),
                mConferenceObservableProvider,
                getCurrentConferenceInformation(),
                getEventBus(),
                conferenceCreateInformation.getConferenceAlias(),
                conferenceCreateInformation.getMetadataHolder(),
                conferenceCreateInformation.getParamsHolder()
        ).createPromise();
    }

    /**
     * Join a conference with a conference
     * <p>
     * Note that the conferenceId needs to be a Voxeet conferenceId,
     *
     * @param conferenceId id of the conference to join as a listener
     * @return a promise to resolve
     */
    public Promise<Boolean> listen(@NonNull String conferenceId) {
        return join(new ConferenceJoinInformation.Builder(conferenceId)
                .setConferenceUserType(ConferenceUserType.LISTENER)
                .build()
        );
    }

    /**
     * Join a conference with a conference
     * <p>
     * Note that the conferenceId needs to be a Voxeet conferenceId,
     *
     * @param conferenceId id of the conference to join as a broadcaster
     * @return a promise to resolve
     */
    public Promise<Boolean> broadcast(@NonNull String conferenceId) {
        return join(new ConferenceJoinInformation.Builder(conferenceId)
                .setConferenceUserType(ConferenceUserType.BROADCASTER)
                .build()
        );
    }

    /**
     * resolve -> true
     * resolve -> false
     * reject -> exception
     *
     * @param conferenceId the existing conference to join
     * @return a promise to resolve
     */
    public Promise<Boolean> join(@NonNull final String conferenceId) {
        return join(new ConferenceJoinInformation.Builder(conferenceId)
                .setConferenceUserType(ConferenceUserType.NORMAL)
                .build()
        );
    }

    /**
     * resolve -> true
     * resolve -> false
     * reject -> exception
     *
     * @param conferenceJoinInformation the holder of the information to join
     * @return a promise to resolve
     */
    private Promise<Boolean> join(@NonNull ConferenceJoinInformation conferenceJoinInformation) {
        return new JoinPromise(this,
                VoxeetSdk.audio(),
                VoxeetSdk.mediaDevice(),
                mConferenceObservableProvider,
                getConferenceInformation(conferenceJoinInformation.getConferenceId()),
                getEventBus(),
                mInstance,
                conferenceJoinInformation.getConferenceId(),
                conferenceJoinInformation.getConferenceUserType()).createPromise();
    }

    /**
     * resole -> true
     * reject -> network error or on created error
     *
     * @return a promise with a boolean indicating the demo conference is (or not) successfully joined
     */
    public Promise<Boolean> demo() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                final Call<CreateConferenceResult> user = mConferenceObservableProvider.getCreateDemoObservable();
                HttpHelper.enqueue(user, new HttpHelper.HttpCallback<CreateConferenceResult>() {
                    @Override
                    public void onSuccess(@NonNull final CreateConferenceResult object, @NonNull Response<CreateConferenceResult> response) {
                        solver.resolve(listen(response.body().conferenceId));
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<CreateConferenceResult> response) {
                        HttpException.dumpErrorResponse(response);
                        solver.reject(e);
                    }
                });
            }
        });
    }

    /**
     * resolve -> list of ConferenceRefreshedUsers
     * reject -> network error
     *
     * @param userInfos a non null list of non null strings
     * @return a non null promise
     */
    @NonNull
    public Promise<List<User>> invite(final String conferenceId, final List<UserInfo> userInfos) {
        return new Promise<>(new PromiseSolver<List<User>>() {
            @Override
            public void onCall(@NonNull Solver<List<User>> solver) {
                //Warning = getConferenceUsers is returning a new non retained if not in a conference
                List<User> users = getConferenceUsers();
                List<String> strings = new ArrayList<>();

                if (null != userInfos) {
                    for (UserInfo info : userInfos) {
                        if (null != info && info.getExternalId() != null) {
                            int i = 0;
                            while (i < users.size()) {
                                User user = users.get(i);
                                if (null != user && null != user.getUserInfo()
                                        && info.getExternalId().equalsIgnoreCase(user.getUserInfo().getExternalId())) {
                                    user.updateIfNeeded(info.getName(), info.getAvatarUrl());
                                }
                                i++;
                            }
                            strings.add(info.getExternalId());
                        }
                    }
                }

                solver.resolve(
                        new InvitePromise(ConferenceService.this,
                                VoxeetSdk.mediaDevice(),
                                mConferenceObservableProvider,
                                getCurrentConferenceInformation(),
                                getEventBus(),
                                conferenceId,
                                strings
                        ).createPromise()
                );
            }
        });
    }

    /**
     * resolve -> true
     * resolve -> false in case of error
     * reject -> dangerous issue
     *
     * @param conferenceId the mConference id
     * @param offset       the offset to start with
     * @return a promise to resolve indicating the result of the request
     */
    public Promise<Boolean> replay(final String conferenceId, final long offset) {
        return new ReplayPromise(this,
                VoxeetSdk.mediaDevice(),
                mConferenceObservableProvider,
                getConferenceInformation(conferenceId),
                getEventBus(),
                conferenceId,
                offset).createPromise();
    }

    /**
     * Get the current mConference status
     * send events but also managed by the promise
     * <p>
     * resolve -> ConferenceStatusResult
     * reject -> error
     *
     * @param conferenceId the mConference id
     * @return
     */
    public Promise<ConferenceStatusResult> getConferenceStatus(final String conferenceId) {
        return new GetConferenceStatusPromiseable(this,
                VoxeetSdk.mediaDevice(),
                mConferenceObservableProvider,
                getConferenceInformation(conferenceId),
                getEventBus()).createPromise();
    }

    /**
     * resolve -> event
     * reject -> network error
     *
     * @param conferenceId the mConference id
     * @return
     */
    public Promise<GetConferenceHistoryResult> conferenceHistory(final String conferenceId) {
        return new GetConferenceHistoryPromiseable(this,
                VoxeetSdk.mediaDevice(),
                mConferenceObservableProvider,
                getCurrentConferenceInformation(),
                getEventBus()).createPromise();
    }

    /**
     * resolve -> true
     * reject -> network error
     *
     * @param conferenceId the mConference id
     * @return
     */
    public Promise<Boolean> subscribe(@NonNull final String conferenceId) {
        return new SubscribeConferenceEventPromiseable(this,
                VoxeetSdk.mediaDevice(),
                mConferenceObservableProvider,
                getConferenceInformation(conferenceId),
                getEventBus()).createPromise();
    }

    /**
     * resolve -> true
     * reject -> network error
     *
     * @return a valid promise
     */
    public Promise<Boolean> unSubscribe(@NonNull final String conferenceId) {
        return new UnsubscribeConferenceEventPromiseable(this,
                VoxeetSdk.mediaDevice(),
                mConferenceObservableProvider,
                getConferenceInformation(conferenceId),
                getEventBus()).createPromise();
    }

    /**
     * resolve -> true
     * reject -> network error
     *
     * @param conferenceId
     * @return a promise to resolve indicating the result of the request
     */
    public Promise<Boolean> subscribeForCall(final String conferenceId) {
        return new SubscribeForCallStartPromiseable(this,
                VoxeetSdk.mediaDevice(),
                mConferenceObservableProvider,
                getConferenceInformation(conferenceId),
                getEventBus()).createPromise();
    }

    /**
     * resolve -> true
     * reject -> network error
     *
     * @param conferenceId
     * @return a promise to resolve indicating the result of the request
     */
    public Promise<Boolean> unSubscribeFromCall(final String conferenceId) {
        return new UnsubscribeForCallStartPromiseable(this,
                VoxeetSdk.mediaDevice(),
                mConferenceObservableProvider,
                getConferenceInformation(conferenceId),
                getEventBus()).createPromise();
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(@NonNull OwnConferenceUserSwitch switchEvent) {
        Log.d(TAG, "onEvent: OwnConferenceUserSwitch : type ? " + switchEvent.getType());


        if (isLive()) {
            leave().then(new PromiseExec<Boolean, Object>() {
                @Override
                public void onCall(@Nullable Boolean result, @NonNull Solver<Object> solver) {
                    Log.d(TAG, "onCall: on leave for switch user");
                }
            }).error(new ErrorPromise() {
                @Override
                public void onError(@NonNull Throwable error) {
                    Log.d(TAG, "onError: on leave for switch user");
                }
            });
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(@NonNull SocketConnectEvent event) {
        String conferenceId = getConferenceId();
        ConferenceInformation information = getCurrentConferenceInformation();

        if (null != conferenceId && null != information && ConferenceState.JOINED.equals(information.getConferenceState())) {
            if (isICERestartEnabled()) {
                Log.d(TAG, "onEvent: SocketConnectEvent Joined <3");
                final Call<ResponseBody> user = mConferenceObservableProvider.iceRestart(conferenceId);
                HttpHelper.enqueue(user, new HttpHelper.HttpCallback<ResponseBody>() {
                    @Override
                    public void onSuccess(@NonNull ResponseBody object, @NonNull Response<ResponseBody> response) {
                        Log.d(TAG, "onEvent: SocketConnectEvent Joined responded <3");
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<ResponseBody> response) {
                        Log.d(TAG, "onEvent: SocketConnectEvent Joined error </3");
                        HttpException.dumpErrorResponse(response);
                    }
                });
            } else {
                Log.d(TAG, "onEvent: socket state opened while in conference but no isICERestartEnabled() = true. A reconnect may be longer");
            }
        } else {
            ConferenceState state = ConferenceState.DEFAULT;
            if (null != information) {
                state = information.getConferenceState();
            }
            Log.d(TAG, "onEvent: SocketConnectEvent not rejoined </3 " + state + " " + conferenceId);
        }
    }

    /**
     * resolve -> true
     * reject -> InConferenceException must be considered false
     * reject -> PromiseConferenceJoinedErrorException must be considered false
     *
     * @param conferenceId the mConference id
     * @return a promise to catch
     */
    @NonNull
    private Promise<Boolean> joinVoxeetConference(@NonNull final String conferenceId) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                String existingConferenceId = getConferenceId();

                Log.d(TAG, "onCall: " + mInConference + " " + existingConferenceId + " " + conferenceId);
                if (mInConference && (null == existingConferenceId || !existingConferenceId.equals(conferenceId))) {
                    try {
                        throw new InConferenceException();
                    } catch (InConferenceException exception) {
                        solver.reject(exception);
                        return;
                    }
                }

                //remove the timeout, a new one should be started when user will be invited
                removeTimeoutCallbacks();

                Log.d(TAG, "joining " + conferenceId);

                setIsInConference(true);
                mConferenceId = conferenceId;
                final ConferenceInformation information = getCurrentConferenceInformation();

                if (null != information) {
                    solver.resolve(joinConferenceInternalPackage(information));
                } else {
                    Log.d(TAG, "onCall: WARNING :: IMPROPER RESULT !!");
                    solver.resolve(false);
                }

            }
        });
    }


    void setCurrentConferenceIfNotInPreviousConference(@NonNull ConferenceInformation conference) {
        String conferenceId = conference.getConference().getId();

        boolean invalidId = null == conferenceId;
        boolean inConference = isInConference();
        boolean sameConference = null == mConferenceId || mConferenceId.equals(conferenceId);
        boolean canOverride = !inConference || sameConference;

        if (!invalidId && canOverride) {
            mConferenceId = conferenceId;
        }

        Log.d(TAG, "setCurrentConferenceIfNotInPreviousConference: "
                + " invalidId = " + invalidId
                + " inConference = " + inConference
                + " sameConference = " + sameConference + " (" + mConferenceId + " " + conferenceId + ") "
                + " canOverride = " + canOverride
                + " mConferenceId = " + mConferenceId);
    }

    /**
     * resolve -> true
     * reject -> InConferenceException must be considered false
     * reject -> PromiseConferenceJoinedErrorException must be considered false
     * <p>
     * Warning : if multiple attempt to join the same conference are made, it is not managed...
     * May be fix in a future version but currently not in the flow
     *
     * @param conference the conference to join
     * @return a promise to catch
     */
    @NonNull
    Promise<Boolean> joinConferenceInternalPackage(@NonNull final ConferenceInformation conference) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                final String conferenceId = conference.getConference().getId();

                boolean invalidId = null == conferenceId;
                boolean inConference = isInConference();
                boolean sameConference = null == mConferenceId || mConferenceId.equals(conferenceId);

                Log.d(TAG, "onCall: join conference " + conference
                        + " invalidId:=" + invalidId
                        + " inConference:=" + inConference
                        + " sameConference:=" + sameConference
                        + " canOverride:=" + (inConference && !sameConference)
                        + " mConferenceId:=" + mConferenceId
                        + " conferenceId:=" + conferenceId);
                Log.d(TAG, "onCall: Not checking for conference joined. Method internally used -- in case of regression please contact support for this specific");
                if (invalidId/* || (inConference && !sameConference)*/) {
                    try {
                        throw new InConferenceException();
                    } catch (InConferenceException exception) {
                        solver.reject(exception);
                        return;
                    }
                }

                //remove the timeout, a new one should be started when user will be invited
                removeTimeoutCallbacks();

                Log.d(TAG, "joining " + conferenceId);

                setIsInConference(true);
                mConferenceId = conferenceId;

                conference.setConferenceState(ConferenceState.JOINING);

                AudioService service = VoxeetSdk.audio();
                if (null != service) {
                    service.enableAec(true);
                    service.enableNoiseSuppressor(true);
                }

                boolean isSuccess = initMedia(conference.isListener());
                Conference correspondingConference = conference.getConference();

                if (!isSuccess) {
                    Log.d(TAG, "onCall: InitMedia failed... new state = left");
                    try {
                        setIsInConference(false);
                        conference.setConferenceState(ConferenceState.LEFT);
                        closeMedia();
                        ConferenceStateEvent error = new ConferenceStateEvent(correspondingConference, correspondingConference.getState());
                        mEventBus.post(error);
                        throw new PromiseConferenceJoinedErrorException(error, null);
                    } catch (PromiseConferenceJoinedErrorException exception) {
                        solver.reject(exception);
                    }
                    return;
                }

                mEventBus.post(new ConferenceStateEvent(correspondingConference, correspondingConference.getState()));

                final Call<ResumeConference> user = mConferenceObservableProvider.joinConference(conferenceId, new JoinParameters(DeviceType.ANDROID, conference.isListener()));

                HttpHelper.enqueue(user, new HttpHelper.HttpCallback<ResumeConference>() {
                    @Override
                    public void onSuccess(@NonNull ResumeConference object, @NonNull Response<ResumeConference> response) {
                        createOrSetConferenceWithParams(object.getConferenceId(),
                                object.getConferenceAlias());
                        initMedia(conference.isListener());


                        String automaticTelecomModePrefix = getAutomaticTelecomModePrefix();

                        boolean hasAutomaticPrefix = null != automaticTelecomModePrefix && !TextUtils.isEmpty(automaticTelecomModePrefix);
                        Log.d(TAG, "onSuccess: trying to join conference, will an attempt to be telecom be made ? " + hasAutomaticPrefix);
                        ConferenceInformation information = getCurrentConferenceInformation();
                        if (null != information) {

                            if (hasAutomaticPrefix && !TextUtils.isEmpty(object.getConferenceAlias())
                                    && object.getConferenceAlias().startsWith(automaticTelecomModePrefix)) {
                                Log.d(TAG, "onSuccess: the conference is now in telecom mode");
                                information.setTelecomMode(true);
                            } else {
                                Log.d(TAG, "onSuccess: the prefix does not match - if the telecom mode is required, it must be done programmatically at this point");
                            }

                            information.setConferenceState(ConferenceState.JOINED);
                        }

                        onConferenceResumedInternal(object, solver);
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<ResumeConference> response) {
                        HttpException.dumpErrorResponse(response);

                        try {
                            e.printStackTrace();

                            setIsInConference(false);

                            //ConferenceInformation information = getCurrentConferenceInformation();
                            conference.setConferenceState(ConferenceState.LEFT);
                            closeMedia();
                            Conference correspondingConference = conference.getConference();
                            ConferenceStateEvent event = new ConferenceStateEvent(correspondingConference, correspondingConference.getState());
                            getEventBus().post(event);
                            throw new PromiseConferenceJoinedErrorException(event, e);
                        } catch (PromiseConferenceJoinedErrorException exception) {
                            solver.reject(exception);
                        }
                    }
                });
            }
        });
    }

    private void onConferenceResumedInternal(@NonNull ResumeConference response,
                                             @NonNull final Solver<Boolean> solver) {
        mConferenceId = response.getConferenceId();

        ConferenceInformation information = getCurrentConferenceInformation();

        //information should not be null in this case

        Conference conference = information.getConference();

        conference.setConferenceId(response.getConferenceId());
        conference.setConferenceAlias(response.getConferenceId());

        if (response.getConferenceAlias() != null) {
            conference.setConferenceAlias(response.getConferenceAlias());
        }

        information.participantsToConferenceUsers(response.getParticipants());

        setIsInConference(true);

        VoxeetSdk.audio().setAudioRoute(AudioRoute.ROUTE_PHONE);

        mEventBus.post(new ConferenceStateEvent(conference, conference.getState()));

        //now loop over every participants to send artifical participant "updated" event
        List<Participant> participants = response.getParticipants();
        int length = null != participants ? participants.size() : -1;

        Log.d(TAG, "onConferenceResumedInternal: resuming with participants ? " + participants + " " + length);
        if (null != participants) {
            for (Participant participant : participants) {
                Log.d(TAG, "onConferenceResumedInternal: call onUserAddedOrUpdated " + participant);
                onUserAddedOrUpdated(conference.getId(),
                        participant.getParticipantId(),
                        participant.getUserInfos(),
                        ConferenceUserStatus.fromString(participant.getStatus()),
                        true);
            }
        }

        if (null == response.getCandidates() && null == response.getDescription()) {
            Log.d(TAG, "onConferenceResumedInternal: candidates + description");
            solver.resolve(true);
            return;
        }

        Log.d(TAG, "onConferenceResumedInternal: having candidates and description");

        UserInfo info = VoxeetSdk.session().getUserInfo();
        String name = "";
        String externalId = "";
        String avatarUrl = "";
        String userId = "";
        if (null != info) {
            userId = VoxeetPreferences.id();
            name = info.getName();
            externalId = info.getExternalId();
            avatarUrl = info.getAvatarUrl();
        }

        handleAnswer(response.getConferenceId(),
                userId != null ? userId : "",
                externalId,
                name,
                avatarUrl,
                "ANDROID",
                true,
                response.getDescription(),
                response.getCandidates())
                .then(new PromiseExec<Boolean, Object>() {
                    @Override
                    public void onCall(@Nullable Boolean aBoolean, @NonNull Solver<Object> s) {
                        Log.d(TAG, "onCall: answer called, result is " + aBoolean + " " + VoxeetPreferences.id());
                        solver.resolve(true);
                    }
                })
                .error(new ErrorPromise() {
                    @Override
                    public void onError(@NonNull Throwable error) {
                        error.printStackTrace();

                        ExceptionManager.sendException(error);
                        solver.reject(error);
                    }
                });
    }

    /**
     * Make an api call to retrieve the list of invited users into any conference
     *
     * @param conferenceId the conference id from which retrieving the invited users
     * @return the promise to resolve
     */
    @NonNull
    public Promise<ConferenceUsersInvitedResult> getInvitedUsers(@NonNull String conferenceId) {
        return new GetConferenceStatus(this,
                VoxeetSdk.mediaDevice(),
                mConferenceObservableProvider,
                getConferenceInformation(conferenceId),
                getEventBus()).createPromise();
    }

    /**
     * This method attempts to leave the current mConference
     * <p>
     * The previous version used a boolean, the new method fingerprint is now void
     */
    @NonNull
    public Promise<Boolean> leave() {
        return new LeavePromise(this,
                VoxeetSdk.mediaDevice(),
                mConferenceObservableProvider,
                getCurrentConferenceInformation(),
                getEventBus())
                .createPromise();
    }


    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Public Event management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final InvitationReceivedEvent invitation) {

        String conferenceId = invitation.getConferenceId();

        if (null == conferenceId && null != invitation.getConference()) {
            conferenceId = invitation.getConference().getConferenceId();
        }

        Log.d(TAG, "onEvent: current mConferenceId " + mConferenceId + " vs " + invitation.getConferenceId() + " vs " + conferenceId);

        if (null != mConferenceId && mConferenceId.equals(conferenceId)) {
            Log.d(TAG, "onEvent: receiving invitation for our conference, we prevent this");
            return;
        }

        //we update the information for this conference only
        final ConferenceInformation information = getConferenceInformation(conferenceId);

        boolean is_own_user = invitation.getUserId().equals(VoxeetPreferences.id());
        ConferenceType type = ConferenceType.fromId(invitation.getConference().getConferenceType());

        if (isLive() && null != getConferenceId() && getConferenceId().equals(conferenceId)) {
            Log.d(TAG, "onEvent: receiving invitation for the same conference, invalidate the call");
            return;
        }

        if (!is_own_user || ConferenceType.SCHEDULED.equals(type)) {
            InvitationReceivedEvent.UserInviter inviter = invitation.getInviter();
            List<Invitation> invitations = invitation.getInvitations();

            final User[] foundUser = {null};
            if (null == information) {
                Log.d(TAG, "onEvent: INVALID INFORMATION FOR THE CONFERENCE");
            }

            if (null != inviter && null != inviter.externalId && null != inviter.userId) {
                Log.d(TAG, "onEvent: Invitation with inviter " + inviter.userId + " " + inviter.externalId);
                UserInfo info = new UserInfo(inviter.nickName, inviter.externalId, inviter.externalAvatarUrl);

                User inviter_user = new User(inviter.userId, info);

                foundUser[0] = inviter_user;
                if (null != information) {
                    information.getLastInvitationReceived().add(inviter_user);
                    information.getUserIdsCached().put(inviter.userId, inviter.externalId);
                }
            } else {
                Log.d(TAG, "onEvent: Invitation with invalid inviter");
            }

            final String finalConferenceId = conferenceId;
            getConferenceStatus(invitation.getConference().getConferenceId())
                    .then(new PromiseExec<ConferenceStatusResult, Object>() {
                        @Override
                        public void onCall(@Nullable ConferenceStatusResult result, @NonNull Solver<Object> internal_solver) {
                            Log.d(TAG, "onSuccess: " + result);

                            //removed the check for live Event since now it is able to have multiple

                            if (result.getConferenceUsers().size() > 0) {
                                Log.d(TAG, "onEvent: users " + result.getConferenceUsers().size());

                                String foundExternalId = null;

                                List<User> merged_list = new ArrayList<>();
                                //add every user from the mConference
                                for (SdkParticipant participant : result.participants) {
                                    merged_list.add(new User(participant));
                                }
                                //add every use from the invitation
                                merged_list.addAll(getLastInvitationUsers());

                                List<User> list = ConferenceUtils.findUsersMatching(currentUserOrEmpty(), merged_list);

                                for (User conferenceUser : list) {
                                    String userId = conferenceUser.getId();
                                    if (null == foundExternalId && null != information) {
                                        String externalId = conferenceUser.getUserInfo().getExternalId();
                                        String cachedExternalId = information.getUserIdsCached().get(userId);
                                        Log.d(TAG, "onEvent: " + userId + " " + externalId + " " + cachedExternalId);

                                        if (null == foundUser[0]) foundUser[0] = conferenceUser;
                                        if (null != userId && (null == externalId || userId.equals(externalId))) {
                                            //should be different than null
                                            externalId = cachedExternalId;
                                            if (null != externalId) {
                                                foundExternalId = cachedExternalId;
                                            }
                                        }
                                    }
                                }

                                if (foundUser[0] != null && null == foundExternalId) {
                                    foundExternalId = foundUser[0].getId();
                                    Log.d(TAG, "externalId is null, setting it to userId");
                                }

                                Map<String, String> infos = new HashMap<>();

                                if (null != foundUser[0]) {

                                }
                                infos.put(Constants.INVITER_ID, null != foundUser[0] ? foundUser[0].getId() : "");
                                infos.put(Constants.INVITER_NAME, null != foundUser[0] ? foundUser[0].getUserInfo().getName() : "");
                                //NOTIF_TYPE ?? useless or usefull ??
                                infos.put(Constants.NOTIF_TYPE, null != result.getType() ? result.getType() : "");
                                infos.put(Constants.INVITER_EXTERNAL_ID, null != foundUser[0] ? foundExternalId : "");
                                infos.put(Constants.INVITER_URL, null != foundUser[0] ? foundUser[0].getUserInfo().getAvatarUrl() : "");
                                infos.put(Constants.CONF_ID, null != result.conferenceId ? result.conferenceId : "");

                                NotificationCenterFactory.instance.onInvitationReceived(context, infos,
                                        Build.MANUFACTURER,
                                        Build.VERSION.SDK_INT);

                                mEventBus.post(new IncomingCallEvent(result.conferenceId));
                            }
                        }
                    })
                    .error(new ErrorPromise() {
                        @Override
                        public void onError(@NonNull Throwable error) {
                            error.printStackTrace();
                        }
                    });
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(OwnConferenceCreated ownConferenceStartedEvent) {
        if (null != ownConferenceStartedEvent.conferenceInfos) {
            mConferenceId = ownConferenceStartedEvent.conferenceInfos.getConferenceId();
        }

        Conference conference = getConference();

        if (null != conference) {
            conference.setConferenceInfos(ownConferenceStartedEvent.conferenceInfos);
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(RenegociationEndedEvent event) {
        Log.d(TAG, "onEvent: Renegociation " + event.getConferenceId()
                + " " + event.getType() + " " + event.getAnswerReceived()
                + " " + event.getOfferSent());
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final OfferCreated offer) {
        Conference conference = getConference();
        String conferenceId = null != conference ? conference.getId() : null;

        if (null != conferenceId && conferenceId.equals(offer.getConferenceId())) {

            User user = findUserById(offer.getUserId());

            if (null == user) {
                user = new User(offer.getUserId(), new UserInfo(offer.getName(), offer.getExternalId(), offer.getAvatarUrl()));
                Log.d(TAG, "run: WARNING obtained OfferCreated for user which did not existed " + user);
                conference.updateUser(user);

                mEventBus.post(new UserAddedEvent(conference, user));
            }

            handleAnswer(offer.getConferenceId(),
                    offer.getUserId(),
                    offer.getExternalId(),
                    offer.getName(),
                    offer.getAvatarUrl(),
                    offer.getDevice(),
                    offer.isMaster(),
                    offer.getDescription(),
                    offer.getCandidates())
                    .then(new PromiseExec<Boolean, Object>() {
                        @Override
                        public void onCall(@Nullable Boolean aBoolean, @NonNull Solver<Object> solver) {
                            Log.d(TAG, "onCall: answer called, result is " + aBoolean + " " + offer.getUserId() + " " + VoxeetPreferences.id());
                        }
                    })
                    .error(new ErrorPromise() {
                        @Override
                        public void onError(@NonNull Throwable error) {
                            error.printStackTrace();
                            ExceptionManager.sendException(error);
                        }
                    });
        } else {
            Log.d(TAG, "onEvent: OfferCreated for another conference");
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ParticipantAdded event) {
        onUserAddedOrUpdated(event.conferenceId,
                event.userId, new UserInfo(event.name, event.externalId, event.avatarUrl),
                ConferenceUserStatus.fromString(event.status),
                true);
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final ParticipantUpdated event) {
        onUserAddedOrUpdated(event.conferenceId,
                event.userId, null,
                ConferenceUserStatus.valueOf(event.status),
                false);
    }

    private void onUserAddedOrUpdated(@NonNull String conferenceId,
                                      @NonNull String userId,
                                      @Nullable UserInfo userInfo,
                                      @NonNull ConferenceUserStatus status,
                                      boolean added) {
        boolean known_user = false;
        MediaEngine media = VoxeetSdk.mediaDevice().getMedia();

        Conference conference = null;
        User user = findUserById(userId);
        ConferenceInformation information = getConferenceInformation(conferenceId);
        if (null != information) {
            conference = information.getConference();
            user = conference.findUserById(userId);

            if (null != user) {
                known_user = true;
                if (added && null != userInfo) {
                    user.setUserInfo(userInfo);
                }
            } else {
                user = new User(userId, userInfo);

                conference.updateUser(user);
            }
        }

        if (user != null) {
            user.updateStatus(status);
        }

        //update the current conference state specificly from the users point of view
        updateConferenceFromUsers();

        switch (status) {
            case CONNECTING:
                if (null != user && !user.getId().equals(VoxeetSdk.session().getUserId())) {
                    Log.d(TAG, "Cancelling timeout timer from user connecting");
                    removeTimeoutCallbacks();
                }
                break;
            case LEFT:
            case DECLINE:
                if (null != media && user != null) {
                    user.streamsHandler().removeAllStreams();
                    Log.d(TAG, "In mConference user with id: " + user + " status updated to " + status);
                    media.removePeer(userId);
                }
                break;
            default:
                Log.d(TAG, "status not managed updated to " + status);
        }

        //check if the timeout runnable was not triggered
        boolean timeout_triggered = null != timeoutRunnable && timeoutRunnable.isTriggered();

        if (!timeout_triggered) {
            switch (status) {
                case DECLINE:
                    if (!userId.equals(VoxeetSdk.session().getUserId())) {
                        Log.d(TAG, "Conference user with id: " + userId + " declined the call");

                        mEventBus.post(new ConferenceUserCallDeclinedEvent(conferenceId, userId, status));
                        break;
                    }
                case LEFT:
                    checkForTelecomEvent(conference.getId(), user.getId());
                default:
                    updateConferenceFromUsers();
                    //if decline but not managed -> fire
                    //else -> fire
                    if (known_user) {
                        mEventBus.post(new UserUpdatedEvent(conference, user));
                    } else {
                        mEventBus.post(new UserAddedEvent(conference, user));
                    }
            }
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceUserCallDeclinedEvent event) {
        checkForTelecomEvent(event.conferenceId, event.userId);
    }

    private void checkForTelecomEvent(String conferenceId, String userId) {
        ConferenceInformation currentConference = getCurrentConferenceInformation();
        //checking if the current conference is in telecom mode
        Log.d(TAG, "checkForTelecomEvent: having currentConference ? " + currentConference);
        if (null != currentConference && currentConference.isTelecomMode()) {
            Conference conference = currentConference.getConference();
            //check if the conference matches
            if (null != conferenceId && conferenceId.equals(conference.getId())) {
                String id = VoxeetPreferences.id();
                if (!TextUtils.isEmpty(userId) && null != id && !id.equals(userId)) {
                    leave().then(new PromiseExec<Boolean, Object>() {
                        @Override
                        public void onCall(@Nullable Boolean result, @NonNull Solver<Object> solver) {
                            Log.d(TAG, "onCall: conference left since declined/left event");
                        }
                    }).error(new ErrorPromise() {
                        @Override
                        public void onError(@NonNull Throwable error) {
                            Log.e(TAG, "onError: conference left since declined/left event", error);
                        }
                    });
                }
            }
        } else {
            Log.d(TAG, "checkForTelecomEvent: not a telecom mode");
        }
    }


    @Subscribe(threadMode = ThreadMode.MAIN, priority = 999)
    public void onEvent(final ConferenceDestroyedPush event) {
        if (null == mConferenceId || mConferenceId.equals(event.conferenceId)) {
            closeMedia();
        } else {
            Log.d(TAG, "onEvent: another conference has ended");
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN, priority = 999)
    public void onEvent(final ConferenceEnded event) {
        if (null == mConferenceId || mConferenceId.equals(event.conferenceId)) {
            closeMedia();
        } else {
            Log.d(TAG, "onEvent: another conference has ended");
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final QualityUpdated event) {
        for (ConferenceUser conferenceUser : event.getUser()) {
            User user = findUserById(conferenceUser.getUserId());
            if (user != null) {
                user.setQuality(conferenceUser.getQuality());

                mEventBus.post(new ConferenceUserQualityUpdatedEvent(user));
            }
        }
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Protected management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    @NonNull
    protected User createConferenceUser(String userId, String device, UserInfo userInfo) {
        //TODO add device information for the users ?
        return new User(userId, /*device,*/ userInfo);
    }

    protected void setConference(@NonNull Conference conference) {
        mConferenceId = conference.getId();
        ConferenceInformation information = getCurrentConferenceInformation();
        information.setConference(conference);
    }

    protected void setConferenceAlias(String alias) {
        Conference conference = getConference();
        conference.setConferenceAlias(alias);
        //null ? throw error B)
    }

    /**
     * Return true if the current implementation is SDK
     * false otherwise
     *
     * @return the state of this implementation
     */
    protected boolean isSDK() {
        return true;
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Private Promises management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    protected Promise<Integer> answer(final String peer, final SdpMessage message) {
        return new Promise<>(new PromiseSolver<Integer>() {
            @Override
            public void onCall(@NonNull final Solver<Integer> solver) {
                try {
                    if (null == mConferenceId) {
                        throw new NotInConferenceException();
                    }
                } catch (NotInConferenceException exception) {
                    solver.reject(exception);
                    return;
                }

                if (null == message) {
                    try {
                        ParticipantAddedErrorEvent event = new ParticipantAddedErrorEvent("Invalid message");
                        throw new PromiseParticipantAddedErrorEventException(event);
                    } catch (PromiseParticipantAddedErrorEventException exception) {
                        solver.reject(exception);
                    }
                }

                Log.d("SDKMEDIA", "answer: peer := " + peer + " message := " + message);

                final Call<ResponseBody> user = mConferenceObservableProvider.answerConference(mConferenceId, peer, message.getDescription());
                HttpHelper.enqueue(user, new HttpHelper.HttpCallback<ResponseBody>() {
                    @Override
                    public void onSuccess(@NonNull ResponseBody object, @NonNull Response<ResponseBody> response) {
                        if (response.code() != 200) {
                            try {
                                ParticipantAddedErrorEvent event = new ParticipantAddedErrorEvent(response.code() + "");
                                throw new PromiseParticipantAddedErrorEventException(event);
                            } catch (PromiseParticipantAddedErrorEventException exception) {
                                solver.reject(exception);
                            }
                        } else {
                            solver.resolve(response.code());
                        }
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<ResponseBody> response) {
                        HttpException.dumpErrorResponse(response);
                        try {
                            ParticipantAddedErrorEvent event = new ParticipantAddedErrorEvent(handleError(e));
                            throw new PromiseParticipantAddedErrorEventException(event);
                        } catch (PromiseParticipantAddedErrorEventException exception) {
                            solver.reject(exception);
                        }
                    }
                });
            }
        });
    }


    /**
     * resolve -> true everything went fine
     * resolve -> false an error occured
     * reject -> media exception
     *
     * @return a promise with the result of the call
     */
    protected Promise<Boolean> handleAnswer(@NonNull final String conferenceId,
                                            @NonNull final String userId,
                                            @NonNull final String externalId,
                                            @NonNull final String userName,
                                            @NonNull final String avatarUrl,
                                            @NonNull final String device,
                                            final boolean isMaster,
                                            @NonNull final OfferDescription offerDescription,
                                            @NonNull final List<OfferCandidate> offerCandidates) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                MediaSDK media = VoxeetSdk.mediaDevice().getMedia();

                if (null == media) {
                    try {
                        throw new MediaEngineException("handleAnswer media is null");
                    } catch (MediaEngineException exception) {
                        solver.reject(exception);
                    }
                }

                if (!conferenceId.equals(mConferenceId)) {
                    Log.d(TAG, "onCall: CONFERENCE IS NOT THE SAME ! ANSWER SHALL BE DISCARDED");
                }

                SdpDescription description = new SdpDescription(offerDescription.type, offerDescription.sdp);

                List<SdpCandidate> candidates = new ArrayList<>();
                for (OfferCandidate candidate : offerCandidates) {
                    Log.d(TAG, "onCall: sdpcandidate " + candidate.getMid() + " " + candidate.getmLine());
                    candidates.add(new SdpCandidate(candidate.getMid(), Integer.parseInt(candidate.getmLine()), candidate.getSdp()));
                }

                //String userId = offer.getUserId();//isSDK() ? offer.getExternalId() : offer.getUserId();
                Log.d(TAG, "handleAnswer: offer := " + userId + " " + externalId);
                Log.d("SDKMEDIA", "handleAnswer: " + userId + " " + externalId + " " + description.ssrc);


                //TODO in the callback, check for the conferenceId being the same !
                createAnswerForPeerThreaded(media, userId, description.ssrc, description, candidates, isMaster)
                        .then(new PromiseExec<SdpMessage, Object>() {
                            @Override
                            public void onCall(@Nullable SdpMessage message, @NonNull Solver<Object> internal_solver) {

                                try {
                                    if (null == mConferenceId) {
                                        Log.d(TAG, "onMessage: INVALID CONFERENCE ID WHEN OFFER IS RECEIVED");
                                        mConferenceId = conferenceId;
                                    }
                                    Conference conference = getConference();

                                    User user = null;

                                    for (User in_conf : conference.getUsers()) {
                                        if (in_conf.getId() != null && in_conf.getId().equals(userId)) {
                                            user = in_conf;
                                        }
                                    }

                                    //TODO update user add or updated ???

                                    UserInfo infos;
                                    if (userId.contains("11111"))
                                        infos = new UserInfo("Julie", externalId, "https://raw.githubusercontent.com/romainbenmansour/JCenter/master/user_1.png");
                                    else if (userId.contains("22222"))
                                        infos = new UserInfo("Sophie", externalId, "https://raw.githubusercontent.com/romainbenmansour/JCenter/master/user_2.png");
                                    else if (userId.contains("33333"))
                                        infos = new UserInfo("Mike", externalId, "https://raw.githubusercontent.com/romainbenmansour/JCenter/master/user_3.png");
                                    else
                                        infos = new UserInfo(userName, externalId, avatarUrl);

                                    if (user == null) {
                                        user = new User(userId, /*device,*/ infos);
                                        conference.updateUser(user);
                                    } else {
                                        user.setUserInfo(infos);
                                    }


                                    setUserPosition(userId, 0, 0.5);

                                    answer(userId, message)
                                            .then(new PromiseExec<Integer, Object>() {
                                                @Override
                                                public void onCall(@Nullable Integer result, @NonNull Solver<Object> internal_solver) {
                                                    Log.d(TAG, "onSuccess: " + result);

                                                    solver.resolve(true);
                                                }
                                            })
                                            .error(new ErrorPromise() {
                                                @Override
                                                public void onError(Throwable error) {
                                                    if (error instanceof PromiseParticipantAddedErrorEventException)
                                                        mEventBus.post(((PromiseParticipantAddedErrorEventException) error).getEvent());
                                                    else
                                                        error.printStackTrace();

                                                    solver.resolve(false);
                                                }
                                            });
                                } catch (Exception e) {
                                    Log.d(TAG, "onMessage: unlockPeerOperation" + e.getMessage());
                                    solver.resolve(false);
                                }
                            }
                        })
                        .error(new ErrorPromise() {
                            @Override
                            public void onError(@NonNull Throwable error) {
                                error.printStackTrace();
                                solver.reject(error);
                            }
                        });

            }
        });
    }
    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Private management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    protected User updateConferenceParticipants(String userId, ConferenceUserStatus status) {
        User user = findUserById(userId);
        Conference conference = getConference();

        if (null != user && null != conference) {
            //TODO check for owner information ?
            //TODO set quality ?
            user//.setIsOwner(conference.getOwnerProfile() == null || conference.getOwnerProfile().getUserId().equals(user.getProfile().getUserId()))
                    .updateStatus(status)
            //.setQuality(ConferenceQuality.ULTRA_HIGH.StringValue())
            ;

            return user;
        }
        return null;
    }

    /**
     * @return false if permission refused -> cancel ! true otherwise
     */
    protected boolean initMedia(boolean listenerMode) {
        Validate.notNull(mContext, "mContext");
        Validate.notNull(VoxeetPreferences.id(), "user id");
        MediaEngine media = VoxeetSdk.mediaDevice().getMedia();

        if (null == media) {
            try {
                if (!listenerMode && !Validate.hasMicrophonePermissions(context)) {
                    Log.d(TAG, "the app does not seem to have mic permission, disabling mic");
                    mute(true);
                    getEventBus().post(new PermissionRefusedEvent(PermissionRefusedEvent.Permission.MICROPHONE));
                    return false;
                }

                VoxeetSdk.audio().enable();
                VoxeetSdk.mediaDevice().createMedia(context, VoxeetPreferences.id(), mediaStreamListener, cameraEventsHandler,
                        isDefaultVideo || isVideoOn(), !listenerMode && Validate.hasMicrophonePermissions(context));
                media = VoxeetSdk.mediaDevice().getMedia();

                VoxeetSdk.audio().setSpeakerMode(isDefaultOnSpeaker);
                VoxeetSdk.audio().unsetMediaRoute().requestAudioFocus();

                if (isDefaultMute) {
                    media.mute();
                } else {
                    media.unMute();
                }
            } catch (MediaEngineException e) {
                e.printStackTrace();
                return false;
            }
        }

        return true;
    }

    protected void closeMedia() {
        MediaSDK media = VoxeetSdk.mediaDevice().getMedia();
        if (null != media) {
            Conference conference = getConference();
            ConferenceInformation information = getCurrentConferenceInformation();

            if (null != information) {
                information.setOwnVideoStarted(false);
                information.setScreenShareOn(false);
            }

            try {
                if (null != conference) {
                    for (User user : conference.getUsers())
                        if (user != null && user.getId() != null)
                            media.removePeer(user.getId());
                }
            } catch (Exception ex) {
                Log.e(TAG, "Error", ex);
            }

            VoxeetSdk.audio().abandonAudioFocusRequest().setMediaRoute().disable();

            VoxeetSdk.mediaDevice().releaseMedia();

            media.unsetStreamListener();
        }

        setIsInConference(false);
        mConferenceId = null;
    }

    /**
     * Remove any previous timeout callback
     * <p>
     * To be used when it is mandatory to remove such behaviour
     * for instance, when joining a new mConference
     * <p>
     * It happens since we do not want to manage every edge case :
     * - decline
     * - network error
     * - etc...
     */
    protected void removeTimeoutCallbacks() {
        if (timeoutRunnable != null) {
            timeoutRunnable.setCanceled(true);
            handler.removeCallbacks(timeoutRunnable);
        }
    }

    protected void sendTimeoutCallbacks() {
        if (mTimeOutTimer != -1) {

            timeoutRunnable = new TimeoutRunnable(this,
                    mEventBus,
                    mTimeOutTimer);

            handler.postDelayed(timeoutRunnable, mTimeOutTimer);
        }
    }

    protected VoxeetSdk getVoxeetSDK() {
        return mSDK;
    }

    @NonNull
    protected ConferenceInformation createOrSetConferenceWithParams(@NonNull String conferenceId, @Nullable String conferenceAlias) {
        Log.d(TAG, "createOrSetConferenceWithParams: set conference id := " + conferenceId);
        ConferenceInformation information = getConferenceInformation(conferenceId);

        Conference conference = information.getConference();

        conference.setConferenceId(conferenceId);
        if (null != conferenceAlias || null == conference.getAlias()) {
            conference.setConferenceAlias(conferenceAlias);
        }

        //the information is always valid
        return information;
    }

    /**
     * Get the current conference information holder. The instance of the class holding the conference representation
     * as well as other utility components
     *
     * @return the current conference on air or null
     */
    @Nullable
    public ConferenceInformation getCurrentConferenceInformation() {
        return getConferenceInformation(mConferenceId);
    }

    /**
     * Check if the current status has any on air participants
     *
     * @return true if at least one is on air
     */
    public boolean hasParticipants() {
        Conference conference = getConference();

        if (null == conference) return false;

        if (conference.hasLocalStreams(false)) return true;
        return conference.hasAny(ConferenceUserStatus.ON_AIR, false);
    }

    /**
     * Change the ICE feature capability
     *
     * @param new_state
     */
    public void setICERestartEnabled(boolean new_state) {
        isICERestartEnabled = new_state;
    }

    /**
     * Check the ICE enabled state
     *
     * @return the current ICE enabled state
     */
    public boolean isICERestartEnabled() {
        return isICERestartEnabled;
    }

    @NonNull
    public Context getContext() {
        return mContext;
    }


    @Nullable
    protected ConferenceInformation getConferenceInformation(@Nullable String conferenceId) {
        if (null == conferenceId) {
            return null;
        }
        return mConferenceInformationHolder.getInformation(conferenceId);
    }

    void joinLock() {
        try {
            joinLock.lock();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void joinUnlock() {
        try {
            if (joinLock.isLocked())
                joinLock.unlock();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void updateConferenceFromUsers() {
        ConferenceInformation information = getCurrentConferenceInformation();
        if (null != information) {
            ConferenceState state = information.getConferenceState();
            switch (state) {
                case JOINED:
                    if (hasParticipants())
                        information.setConferenceState(ConferenceState.FIRST_PARTICIPANT);
                    break;
                case FIRST_PARTICIPANT:
                    if (!hasParticipants())
                        information.setConferenceState(ConferenceState.NO_MORE_PARTICIPANT);
                    break;
                case NO_MORE_PARTICIPANT:
                    if (hasParticipants())
                        information.setConferenceState(ConferenceState.FIRST_PARTICIPANT);
                    break;
                default:
                    //nothing
            }
            state = information.getConferenceState();
        }
    }

    @NonNull
    private ErrorPromise manageError() {
        return new ErrorPromise() {
            @Override
            public void onError(@NonNull Throwable error) {
                error.printStackTrace();
            }
        };
    }

    @Nullable
    private String getUserId() {
        SessionService sessionService = VoxeetSdk.session();
        return null != sessionService ? sessionService.getUserId() : null;
    }

    void clearConferencesInformation() {
        this.mConferenceInformationHolder.clearConferencesInformation();
    }

    @NoDocumentation
    public void moveConference(@NonNull String conferenceId, @NonNull ConferenceInformation conferenceInformation) {
        Log.d(TAG, "moveConference: change " + conferenceInformation.getConference().getId() + " to " + conferenceId);
        mConferenceInformationHolder.moveConference(conferenceId, conferenceInformation);
    }

    /**
     * Submit a Create Answer For Peer into the new Threaded Pool
     *
     * @param media      the media instance to use
     * @param userId     the userId to create the answer for
     * @param ssrc       the ssrc
     * @param offer      the actual offer
     * @param candidates the candidates
     * @param master     is this peer a master one
     * @return the Promise to resolve to get the actual Sdpmessage from
     */
    private Promise<SdpMessage> createAnswerForPeerThreaded(@NonNull MediaSDK media,
                                                            @NonNull String userId,
                                                            long ssrc,
                                                            @NonNull SdpDescription offer,
                                                            @NonNull List<SdpCandidate> candidates,
                                                            boolean master) {
        return new Promise<>(new PromiseSolver<SdpMessage>() {
            @Override
            public void onCall(@NonNull Solver<SdpMessage> solver) {
                executorService.submit(new Runnable() {
                    @Override
                    public void run() {
                        try {
                            media.createAnswerForPeer(userId,
                                    ssrc,
                                    offer,
                                    candidates,
                                    master,
                                    new PendingPeerCallback() {
                                        @Override
                                        public void onMessage(@Nullable SdpMessage message) {
                                            solver.resolve(message);
                                        }
                                    });
                        } catch (MediaEngineException e) {
                            e.printStackTrace();
                            solver.reject(e);
                        }
                    }
                });
            }
        });
    }
}