package com.voxeet.sdk.core.services;

import android.support.annotation.NonNull;

import com.voxeet.sdk.core.VoxeetSdk;
import com.voxeet.sdk.core.network.endpoints.ISdkVideoPresentationRService;
import com.voxeet.sdk.core.services.abstracts.AbstractPresentationService;
import com.voxeet.sdk.core.services.presentation.PresentationState;
import com.voxeet.sdk.core.services.presentation.video.VideoPresentation;
import com.voxeet.sdk.json.VideoPresentationPaused;
import com.voxeet.sdk.json.VideoPresentationPlay;
import com.voxeet.sdk.json.VideoPresentationSeek;
import com.voxeet.sdk.json.VideoPresentationStarted;
import com.voxeet.sdk.json.VideoPresentationStopped;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.NoDocumentation;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.Nullable;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;

/**
 * Manage every outgoing or ingoing call related to Video stream from third party or outside files
 */
@Annotate
public class VideoPresentationService extends AbstractPresentationService<ISdkVideoPresentationRService, VideoPresentation> {

    private List<Solver<VideoPresentation>> mCacheStartedSolvers;
    private List<Solver<VideoPresentation>> mCacheStoppedSolvers;
    private List<Solver<VideoPresentation>> mCachePausedSolvers;
    private List<Solver<VideoPresentation>> mCachePlaySolvers;
    private List<Solver<VideoPresentation>> mCacheSeekSolvers;


    /**
     * Instantiates a new User service.
     *
     * @param instance the parent instance
     */
    @NoDocumentation
    public VideoPresentationService(VoxeetSdk instance) {
        super(instance, ISdkVideoPresentationRService.class);

        mCachePausedSolvers = new ArrayList<>();
        mCacheStartedSolvers = new ArrayList<>();
        mCacheStoppedSolvers = new ArrayList<>();
        mCachePausedSolvers = new ArrayList<>();
        mCachePlaySolvers = new ArrayList<>();
        mCacheSeekSolvers = new ArrayList<>();
        registerEventBus();
    }

    void onInternalServiceEvent(VideoPresentationStarted event) {
        VideoPresentation information = getPresentationInformation(event.key);
        information.url = event.url;
        presentations.add(information);

        information.state = PresentationState.STARTED;
        information.lastSeekTimestamp = event.timestamp;
        getEventBus().post(event);
        tryUnlock(information, mCacheStartedSolvers);
    }

    void onInternalServiceEvent(VideoPresentationStopped event) {
        VideoPresentation information = getPresentationInformation(event.key);
        information.state = PresentationState.STOP;
        presentations.remove(information);

        getEventBus().post(event);
        tryUnlock(information, mCacheStoppedSolvers);
    }

    void onInternalServiceEvent(VideoPresentationPaused event) {
        VideoPresentation information = getPresentationInformation(event.key);
        information.state = PresentationState.PAUSED;
        information.lastSeekTimestamp = event.timestamp;

        getEventBus().post(event);
        tryUnlock(information, mCachePausedSolvers);
    }

    void onInternalServiceEvent(VideoPresentationPlay event) {
        VideoPresentation information = getPresentationInformation(event.key);
        information.state = PresentationState.PLAY;
        information.lastSeekTimestamp = event.timestamp;

        getEventBus().post(event);
        tryUnlock(information, mCachePlaySolvers);
    }

    void onInternalServiceEvent(VideoPresentationSeek event) {
        VideoPresentation information = getPresentationInformation(event.key);
        information.state = PresentationState.SEEK;
        information.lastSeekTimestamp = event.timestamp;

        getEventBus().post(event);
        tryUnlock(information, mCacheSeekSolvers);
    }

    /**
     * Made a call to change the given video "start" parameter
     *
     * @return the promise to resolve
     */
    @NonNull
    public Promise<VideoPresentation> startVideoPresentation(final String url) {
        return new Promise<>(new PromiseSolver<VideoPresentation>() {
            @Override
            public void onCall(@NonNull final Solver<VideoPresentation> solver) {
                ISdkVideoPresentationRService.VideoPresentationUrl holder = new ISdkVideoPresentationRService.VideoPresentationUrl(url);
                consumeInternalCall(solver, mCacheStartedSolvers,
                        internalCall(getService().startVideoPresentation(getConferenceId(), holder)));
            }
        });
    }

    /**
     * Made a call to change the given video "stop" parameter
     *
     * @return the promise to resolve
     */
    @NonNull
    public Promise<VideoPresentation> stopVideoPresentation() {
        return new Promise<>(new PromiseSolver<VideoPresentation>() {
            @Override
            public void onCall(@NonNull final Solver<VideoPresentation> solver) {

                consumeInternalCall(solver, mCacheStoppedSolvers,
                        internalCall(getService().stopVideoPresentation(getConferenceId())));
            }
        });
    }

    /**
     * Made a call to change the given video "play" parameter
     *
     * @return the promise to resolve
     */
    @NonNull
    public Promise<VideoPresentation> playVideoPresentation() {
        return new Promise<>(new PromiseSolver<VideoPresentation>() {
            @Override
            public void onCall(@NonNull final Solver<VideoPresentation> solver) {

                consumeInternalCall(solver, mCachePlaySolvers,
                        internalCall(getService().playVideoPresentation(getConferenceId())));
            }
        });
    }

    /**
     * Made a call to change the given video "pause" parameter
     *
     * @param timestamp the timestamp the video paused at
     * @return the promise to resolve
     */
    @NonNull
    public Promise<VideoPresentation> pauseVideoPresentation(final long timestamp) {
        return new Promise<>(new PromiseSolver<VideoPresentation>() {
            @Override
            public void onCall(@NonNull final Solver<VideoPresentation> solver) {
                ISdkVideoPresentationRService.VideoPresentationSeek body_sent = new ISdkVideoPresentationRService.VideoPresentationSeek(timestamp);

                consumeInternalCall(solver, mCachePausedSolvers,
                        internalCall(getService().pauseVideoPresentation(getConferenceId(), body_sent)));
            }
        });
    }

    /**
     * Made a call to change the given video "seek" parameter
     *
     * @param timestamp the new timestamp
     * @return the promise to resolve
     */
    @NonNull
    public Promise<VideoPresentation> seekVideoPresentation(final long timestamp) {
        return new Promise<>(new PromiseSolver<VideoPresentation>() {
            @Override
            public void onCall(@NonNull final Solver<VideoPresentation> solver) {

                ISdkVideoPresentationRService.VideoPresentationSeek body_sent = new ISdkVideoPresentationRService.VideoPresentationSeek(timestamp);

                consumeInternalCall(solver, mCacheSeekSolvers,
                        internalCall(getService().seekVideoPresentation(getConferenceId(), body_sent)));
            }
        });
    }

    /**
     * If a presentation is currently playing, this method will return the description of this presentation
     * <p>
     * if none (or stopped) this method will return null
     * This method exists primarily because of the possible various Activities states
     *
     * @return a clone of the holder of the (key, url, state) or null if no presentation currently playing
     */
    @Nullable
    public VideoPresentation getCurrentPresentation() {
        for (VideoPresentation information : presentations) {
            if (!PresentationState.STOP.equals(information)) return information.clone();
        }
        return null;
    }

    @NonNull
    private VideoPresentation getPresentationInformation(@NonNull String key) {
        for (VideoPresentation information : presentations) {
            if (key.equals(information.key)) return information;
        }
        return new VideoPresentation(key, "");
    }

}