package com.voxeet.sdk.core.services;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.graphics.Point;
import android.media.projection.MediaProjectionManager;
import android.os.Build;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;
import android.view.WindowManager;

import com.voxeet.sdk.core.AbstractVoxeetService;
import com.voxeet.sdk.core.VoxeetSdk;
import com.voxeet.sdk.core.services.holder.ServiceProviderHolder;
import com.voxeet.sdk.core.services.screenshare.RequestScreenSharePermissionEvent;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.NoDocumentation;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.ErrorPromise;
import eu.codlab.simplepromise.solve.PromiseExec;
import eu.codlab.simplepromise.solve.Solver;

/**
 * Module to manage the various ScreenShare elements
 */
@Annotate
public class ScreenShareService extends AbstractVoxeetService {
    private static final int REQUEST_SCREEN_CAPTURE = 672;
    private static final String TAG = ScreenShareService.class.getSimpleName();
    private static final int DEFAULT_WIDTH = 640;
    private static final int DEFAULT_HEIGHT = 480;

    private int mLastResultCode;

    @Nullable
    private Point mLastPoint;

    @Nullable
    private Intent mLastData;

    @NoDocumentation
    public ScreenShareService(@NonNull VoxeetSdk instance) {
        super(instance, ServiceProviderHolder.DEFAULT);
    }

    /**
     * Start a screenshare if currently not started or stop it.
     * <p>
     * Warning : this method is only making call internally, no information is sent directly after this call.
     * <p>
     * Developers should use the startScreenshare or stopScreenshare only
     */
    @NoDocumentation
    @Deprecated
    public void toggleScreenShare() {
        VoxeetSdk.conference().toggleScreenShare();
    }

    /**
     * Start a screenshare instance
     *
     * @param intent intent information obtained after the user has granted screenshare permission
     * @return the promise to resolve
     */
    @NonNull
    public Promise<Boolean> startScreenShare(@NonNull Intent intent) {
        ConferenceService service = VoxeetSdk.conference();

        int width = DEFAULT_WIDTH;
        int height = DEFAULT_HEIGHT;

        if (null != mLastPoint) {
            width = mLastPoint.x;
            height = mLastPoint.y;
        }

        return service.startScreenShare(intent, width, height);
    }

    /**
     * Stop the current screenshare instance
     *
     * @return the promise to resolve
     */
    public Promise<Boolean> stopScreenShare() {
        return VoxeetSdk.conference().stopScreenShare();
    }

    /**
     * Change future screenshare propagation information
     *
     * @param point a point representing the widt and height of the screenshare information
     * @return the current service instance
     */
    public ScreenShareService setScreenSizeInformation(@NonNull Point point) {
        mLastPoint = point;

        return this;
    }

    /**
     * Send a screenshare request to the VoxeetAppCompatActivity's instance which will present the modal to the user.
     * <p>
     * After the permission has been granted, the VoxeetAppCompatActivity's will request the start call
     */
    public void sendRequestStartScreenShare() {
        getEventBus().post(new RequestScreenSharePermissionEvent());
    }

    /**
     * Utility method to make the proper permission dialog given any Activity instance.
     *
     * @param activity a valid activity which will be used to start the system's one and gets the callback result
     * @return the compatibility result
     */
    public boolean sendUserPermissionRequest(@NonNull Activity activity) {
        if (Build.VERSION_CODES.LOLLIPOP <= Build.VERSION.SDK_INT) {
            MediaProjectionManager manager = (MediaProjectionManager)
                    activity.getSystemService(Context.MEDIA_PROJECTION_SERVICE);
            Intent request = manager.createScreenCaptureIntent();
            activity.startActivityForResult(request, ScreenShareService.REQUEST_SCREEN_CAPTURE);
            return true;
        }
        return false;
    }

    /**
     * Given information obtained and stored from the activities lifecycle, it will automatically manage those and call the startScreenshare
     */
    public void consumeRightsToScreenShare() {
        if (mLastResultCode == Activity.RESULT_OK && null != mLastData) {
            startScreenShare(mLastData)
                    .then(new PromiseExec<Boolean, Object>() {
                        @Override
                        public void onCall(@Nullable Boolean result, @NonNull Solver<Object> solver) {
                            Log.d(TAG, "onCall: starting screen share");
                        }
                    })
                    .error(new ErrorPromise() {
                        @Override
                        public void onError(@NonNull Throwable error) {
                            Log.d(TAG, "onError: error while starting screen share");
                            error.printStackTrace();
                        }
                    });
            mLastData = null;
        }
    }

    /**
     * Manage the Activity result in the context of the Screen Share Feature
     * <p>
     * returns true if the logic was managed
     * false, otherwise
     *
     * @param requestCode the requestCode obtained it should match
     * @param resultCode  the resultCode obtained
     * @param data        the incoming data
     * @return true/false if successfully managed
     */
    public boolean onActivityResult(int requestCode, int resultCode, Intent data) {
        if (requestCode == REQUEST_SCREEN_CAPTURE) {
            if (Activity.RESULT_OK != resultCode) {
                VoxeetSdk.conference().onUserCanceledScreenShare();
            } else {
                mLastResultCode = resultCode;
                mLastData = data;
                return true;
            }
        }
        return false;
    }

    /**
     * From a givevn context, get the current Screen size information
     *
     * @param context the context from which retrieve the information
     * @return a valid point information
     */
    @NonNull
    public Point getScreenSize(@NonNull Context context) {
        Point point = new Point();

        WindowManager windowManager = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
        if (null != windowManager) {
            windowManager.getDefaultDisplay().getSize(point);
        }

        return point;
    }

    /**
     * From given screen information, get the scaled width according to the requested height
     *
     * @param screen_size a Point object populated with the screen's width/height
     * @param height      the requested height of the stream
     * @return the scaled up/down screen size
     */
    @NonNull
    public Point getScreenSizeScaled(@NonNull Point screen_size, int height) {
        Point point = new Point(screen_size);
        int screen_width = screen_size.x;
        int screen_height = screen_size.y;

        if (screen_height > 0) {
            point.y = height;
            point.x = (int) (screen_width * height * 1f / screen_height);
        }
        return point;
    }
}
