package com.voxeet.sdk.core.services;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.sdk.core.VoxeetSdk;
import com.voxeet.sdk.core.network.endpoints.ISdkConferenceRService;
import com.voxeet.sdk.core.services.conference.information.ConferenceInformation;
import com.voxeet.sdk.core.services.conference.promises.StartRecordingPromiseable;
import com.voxeet.sdk.core.services.conference.promises.StopRecordingPromiseable;
import com.voxeet.sdk.core.services.holder.ServiceProviderHolder;
import com.voxeet.sdk.json.RecordingStatusUpdateEvent;
import com.voxeet.sdk.models.Conference;
import com.voxeet.sdk.models.User;
import com.voxeet.sdk.models.v1.RecordingStatus;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.NoDocumentation;

import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;

import java.util.Date;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.ErrorPromise;
import eu.codlab.simplepromise.solve.PromiseExec;
import eu.codlab.simplepromise.solve.Solver;

/**
 * Conference SDK Service
 * <p>
 * Warning : always implements the error for each promises you may call to start resolve them
 * You can also use the execute method but in case of Exception, you will trigger errors...
 */
@Annotate
public class RecordingService extends com.voxeet.sdk.core.AbstractVoxeetService<ISdkConferenceRService> {

    private final static String TAG = RecordingService.class.getSimpleName();
    private ConferenceSdkObservableProvider mConferenceObservableProvider;

    private boolean isRecording = false;
    private Context mContext;

    private boolean isICERestartEnabled = false;

    @NoDocumentation
    public RecordingService(VoxeetSdk instance) {
        super(instance, new ServiceProviderHolder.Builder<ISdkConferenceRService>()
                .setRetrofit(instance.getRetrofit())
                .setService(ISdkConferenceRService.class)
                .setEventBus(instance.getEventBus())
                .setClient(instance.getClient())
                .build());

        mConferenceObservableProvider = new ConferenceSdkObservableProvider();
        mConferenceObservableProvider.setRetrofitInstantiatedProvider(getService());
        mContext = instance.getApplicationContext();

        registerEventBus();
    }

    public void toggleRecording() {
        Promise<Boolean> promise = isRecording ? stopRecording() : startRecording();
        promise.then(new PromiseExec<Boolean, Object>() {
            @Override
            public void onCall(@Nullable Boolean result, @NonNull Solver<Object> solver) {
                Log.d(TAG, "onSuccess: toggle done " + result);
            }
        }).error(new ErrorPromise() {
            @Override
            public void onError(Throwable error) {
                error.printStackTrace();
            }
        });
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Public Promises management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
    /**
     * resolve -> true
     * resolve -> false in case of non 200 response
     * reject -> network error
     *
     * @return a promise to resolve indicating the result of the request
     */
    public Promise<Boolean> startRecording() {
        ConferenceService service = VoxeetSdk.conference();
        return new StartRecordingPromiseable(service,
                VoxeetSdk.mediaDevice(),
                mConferenceObservableProvider,
                service.getCurrentConferenceInformation(),
                getEventBus()).createPromise();
    }

    /**
     * resolve -> true
     * resolve -> false not a 200 response
     * reject -> network error
     *
     * @return
     */
    public Promise<Boolean> stopRecording() {
        ConferenceService service = VoxeetSdk.conference();
        return new StopRecordingPromiseable(service,
                VoxeetSdk.mediaDevice(),
                mConferenceObservableProvider,
                service.getCurrentConferenceInformation(),
                getEventBus()).createPromise();
    }


    @Subscribe(threadMode = ThreadMode.MAIN, priority = 999)
    public void onEvent(final RecordingStatusUpdateEvent recordingStatusUpdateEvent) {
        ConferenceService service = VoxeetSdk.conference();
        if (null == service) return;
        String conferenceId = recordingStatusUpdateEvent.conferenceId;

        ConferenceInformation information = service.getConferenceInformation(conferenceId);
        if(null == information) return;

        Conference conference = information.getConference();
        User user = service.findUserById(recordingStatusUpdateEvent.userId);
        if (user != null) {

            Conference.RecordingInformation recording = new Conference.RecordingInformation();
            RecordingStatus status = RecordingStatus.valueOf(recordingStatusUpdateEvent.recordingStatus);
            if (status == RecordingStatus.RECORDING) {
                isRecording = true;
                recording.setStartRecordTimestamp(new Date(recordingStatusUpdateEvent.timeStamp));
                recording.setRecordingStatus(RecordingStatus.RECORDING);
                recording.setRecordingUser(recordingStatusUpdateEvent.userId);
                //TODO user owner ?
                //user.setIsRecordingOwner(true);
            } else {
                isRecording = false;
                recording.setStartRecordTimestamp(null);
                recording.setRecordingStatus(RecordingStatus.NOT_RECORDING);
                recording.setRecordingUser(null);
                //TODO user owner ?
                //user.setIsRecordingOwner(false);
            }

            conference.setRecordingInformation(recording);
        }
    }

}