package com.voxeet.sdk.core.services;

import android.support.annotation.NonNull;

import com.voxeet.sdk.core.AbstractVoxeetService;
import com.voxeet.sdk.core.VoxeetSdk;
import com.voxeet.sdk.core.network.endpoints.ISdkConferenceRService;
import com.voxeet.sdk.core.services.conference.promises.SendBroadcastMessagePromise;
import com.voxeet.sdk.core.services.holder.ServiceProviderHolder;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.NoDocumentation;

import eu.codlab.simplepromise.Promise;

/**
 * Module to manage incoming well formatted ChatMessage and redistribute them accross the EventBus
 */
@Annotate
public class CommandService extends AbstractVoxeetService<ISdkConferenceRService> {
    private ConferenceSdkObservableProvider mConferenceObservableProvider;

    @NoDocumentation
    public CommandService(@NonNull VoxeetSdk instance) {
        super(instance, new ServiceProviderHolder.Builder<ISdkConferenceRService>()
                .setRetrofit(instance.getRetrofit())
                .setService(ISdkConferenceRService.class)
                .setEventBus(instance.getEventBus())
                .setClient(instance.getClient())
                .build());

        mConferenceObservableProvider = new ConferenceSdkObservableProvider();
        mConferenceObservableProvider.setRetrofitInstantiatedProvider(getService());
    }


    /**
     * Send a message to any given conference. The message must be a string but it can hold any representation of strings :
     * - json
     * - base64
     * - ...
     *
     * @param conferenceId the conference id to which the message must be sent
     * @param message      the actual message content (any possible string)
     * @return a promise to resolve
     */
    @NonNull
    public Promise<Boolean> sendMessage(@NonNull String conferenceId, @NonNull final String message) {
        return new SendBroadcastMessagePromise(VoxeetSdk.conference(),
                VoxeetSdk.mediaDevice(),
                mConferenceObservableProvider,
                VoxeetSdk.conference().getConferenceInformation(conferenceId),
                getEventBus(),
                conferenceId,
                message).createPromise();
    }
}
