package com.voxeet.sdk.models.v2;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.android.media.MediaStream;
import com.voxeet.android.media.MediaStreamType;
import com.voxeet.sdk.models.User;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.NoDocumentation;

import java.util.List;

/**
 * Class dedicated to help insert, update, remove and get specific streams from a given user
 *
 * It assumes the User.streams method will resolve a reference to the given User's streams
 */
@Annotate
public class UserMediaStreamHandler {
    private User user;

    @NoDocumentation
    public UserMediaStreamHandler(@NonNull User user) {
        this.user = user;
    }

    /**
     * Check for the existence of a specific MediaStream Type into the current user lifecycle
     * @param type the MediaStream type to look for
     * @return return the search result
     */
    public boolean has(@NonNull MediaStreamType type) {
        for (MediaStream stream : user.streams()) {
            if (type.equals(stream.getType())) return true;
        }
        return false;
    }

    /**
     * Get the first MediaStream corresponding to a given type
     * @param type the type to look
     * @return the instance of such MediaStream or null
     */
    @Nullable
    public MediaStream getFirst(@NonNull MediaStreamType type) {
        for (MediaStream stream : user.streams()) {
            if (type.equals(stream.getType())) return stream;
        }
        return null;
    }

    /**
     * Manage the insert or update of the given stream
     * @param mediaStream the mediastream to insert or update
     * @return the user instance
     */
    @NonNull
    public User insertOrUpdate(@NonNull MediaStream mediaStream) {
        List<MediaStream> streams = user.streams();
        boolean found = false;
        for (MediaStream stream : streams) {
            if (stream.equals(mediaStream)) {
                found = true;
                break;
            }
        }

        if (!found) streams.add(mediaStream);
        else {
            streams.remove(mediaStream);
            streams.add(mediaStream);
        }
        return user;
    }

    /**
     * Clear the given User's streams
     * @return the User instance
     */
    @NonNull
    public User removeAllStreams() {
        List<MediaStream> streams = user.streams();
        streams.clear();
        return user;
    }

    /**
     * Remove a given mediastream from the User's list
     * @param mediaStream the MediaStream to remove
     * @return the result of the erasure in the list
     */
    @NonNull
    public boolean remove(@NonNull MediaStream mediaStream) {
        List<MediaStream> streams = user.streams();
        return streams.remove(mediaStream);
    }

    /**
     * Remove a given mediastream type from the User's list
     * @param type the given type to erase from the streams
     * @return if a given stream with the appropriate type has been removed
     */
    @NonNull
    public boolean remove(@NonNull MediaStreamType type) {
        List<MediaStream> streams = user.streams();
        MediaStream toRemove = null;
        for (MediaStream stream : streams) {
            if (type.equals(stream.getType())) {
                toRemove = stream;
                break;
            }
        }

        if (null != toRemove) {
            streams.remove(toRemove);
            return true;
        }

        return false;
    }
}
