package com.voxeet.sdk.media.audio;

import android.content.Context;
import android.media.Ringtone;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.audio.AudioRoute;
import com.voxeet.audio.AudioStackManager;
import com.voxeet.audio.listeners.IAudioRouteListener;
import com.voxeet.audio.utils.Constants;
import com.voxeet.sdk.events.sdk.AudioRouteChangeEvent;
import com.voxeet.sdk.media.MediaPowerManager;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.AudioType;
import com.voxeet.sdk.utils.SoundPool;
import com.voxeet.sdk.utils.Validate;

import org.greenrobot.eventbus.EventBus;

import java.util.HashMap;
import java.util.List;

@Annotate
public class SoundManager implements IAudioRouteListener {

    private AudioStackManager mAudioManager;
    private MediaPowerManager mMediaPowerManager;
    private Context mContext;
    private HashMap<Integer, SoundPool> _sound_pools;
    private HashMap<AudioType, String> mSounds;

    private boolean mEnable;

    private SoundManager() {
        disable();
    }

    public SoundManager(@NonNull Context context) {
        this();

        _sound_pools = new HashMap<>();

        mAudioManager = new AudioStackManager(context);
        mAudioManager.setMediaRoute();
        mAudioManager.configureVolumeStream(Constants.STREAM_VOICE_CALL,
                Constants.STREAM_MUSIC);
        mAudioManager.registerAudioRouteListener(this);
        mMediaPowerManager = new MediaPowerManager(context,
                currentRoute());

        mContext = context;

        mSounds = new HashMap<>();
        configure();
    }

    /**
     * Get the current available routes
     *
     * @return a non null route
     */
    @NonNull
    public List<AudioRoute> getAvailableRoutes() {
        return mAudioManager.availableRoutes();
    }

    public AudioRoute currentRoute() {
        return mAudioManager.outputRoute();
    }

    public boolean setAudioRoute(@NonNull AudioRoute route) {
        Validate.runningOnUiThread();
        return mAudioManager.setOutputRoute(route);
    }

    @Nullable
    public Ringtone getSystemRingtone() {
        return mAudioManager.getSystemRingtone();
    }

    public boolean isBluetoothHeadsetConnected() {
        return mAudioManager.isBluetoothHeadsetConnected();
    }

    public boolean isWiredHeadsetOn() {
        return mAudioManager.isWiredHeadsetOn();
    }

    public SoundManager setSpeakerMode(boolean isSpeaker) {
        mAudioManager.setSpeakerMode(isSpeaker);
        return this;
    }

    public boolean isSpeakerOn() {
        return AudioRoute.ROUTE_SPEAKER.equals(mAudioManager.outputRoute());
    }

    public SoundManager acquireLocks() {
        mMediaPowerManager.acquire();
        return this;
    }

    public SoundManager releaseLocks() {
        mMediaPowerManager.release();
        return this;
    }

    public SoundManager resetDefaultSoundType() {
        mAudioManager.resetDefaultSoundType();
        return this;
    }

    private int getUiSoundsStreamType() {
        return mAudioManager.getDefaultSoundStreamType();
    }

    private SoundManager forceVolumeControlStream(int type) {
        mAudioManager.forceVolumeControlStream(type);
        return this;
    }

    public SoundManager enableMedia() {
        mEnable = true;
        if (null != mAudioManager) {
            mAudioManager.setMediaRoute();
            mAudioManager.enable();
        }
        return this;
    }

    public SoundManager setMediaRoute() {
        mAudioManager.setMediaRoute();
        return this;
    }

    public SoundManager unsetMediaRoute() {
        mAudioManager.unsetMediaRoute();
        return this;
    }

    public SoundManager abandonAudioFocusRequest() {
        mAudioManager.abandonAudioFocus();
        return this;
    }

    public SoundManager requestAudioFocus() {
        mAudioManager.requestAudioFocus();
        return this;
    }

    public SoundManager checkOutputRoute() {
        if(isEnabled()) {
            mAudioManager.checkOutputRoute();
        }
        return this;
    }

    private SoundPool getSoundPool(int soundMode) {
        SoundPool pool = _sound_pools.get(soundMode);
        if (pool == null) {
            pool = new SoundPool(mContext, soundMode, getVolume(soundMode));
            _sound_pools.put(soundMode, pool);
        }
        return pool;
    }

    private void configure() {
        //default disable the audio manager
        mAudioManager.disable();

        mSounds.put(AudioType.RING, "out.mp3");
        mSounds.put(AudioType.HANGUP, "leave.mp3");

        setSound(AudioType.RING, mSounds.get(AudioType.RING));
        setSound(AudioType.HANGUP, mSounds.get(AudioType.HANGUP));
    }

    public boolean setSound(@NonNull AudioType type, @NonNull String assetName) {
        return setSound(type, assetName, Constants.STREAM_VOICE_CALL);
    }

    public boolean setSound(@NonNull AudioType type, @NonNull String assetName, int soundMode) {
        return getSoundPool(soundMode).release(type).setShortResource(type, assetName);
    }

    public SoundManager playSoundType(@NonNull AudioType type) {
        return playSoundType(type, Constants.STREAM_VOICE_CALL);
    }

    public SoundManager playSoundType(@NonNull AudioType type, int soundMode) {
        if (isEnabled()) {
            getSoundPool(soundMode).playShortResource(type, mSounds.get(type));
        }
        return this;
    }

    public SoundManager playSoundTypeForce(@NonNull AudioType type) {
        return playSoundType(type, Constants.STREAM_VOICE_CALL);
    }

    public SoundManager playSoundTypeForce(@NonNull AudioType type, int soundMode) {
        getSoundPool(soundMode).playShortResource(type, mSounds.get(type));
        return this;
    }

    public SoundManager stopSoundType(@NonNull AudioType audioType) {
        return stopSoundType(audioType, Constants.STREAM_VOICE_CALL);
    }

    public SoundManager stopSoundType(@NonNull AudioType audioType, int soundMode) {
        getSoundPool(soundMode).stop(audioType);
        return this;
    }

    public SoundManager stop(int soundMode) {
        getSoundPool(soundMode).stop();
        return this;
    }

    public SoundManager stop() {
        stop(Constants.STREAM_VOICE_CALL);
        return this;
    }

    public SoundManager onConferencePreJoinedEvent() {
        mAudioManager.checkOutputRoute();
        return this;
    }

    public SoundManager onConferenceDestroyedPush() {
        abandonAudioFocusRequest();
        return this;
    }

    public SoundManager enable() {
        mEnable = true;
        if (null != mAudioManager) {
            mAudioManager.enable();
        }
        return this;
    }

    public SoundManager disable() {
        mEnable = false;
        if (null != mAudioManager) {
            mAudioManager.disable();
        }
        return this;
    }

    private boolean isEnabled() {
        return mEnable;
    }

    private float getVolume(int soundMode) {
        switch (soundMode) {
            case Constants.STREAM_VOICE_CALL:
                return 0.1f;
            default:
                return 1.f;
        }
    }

    @Override
    public void onAudioRouteChanged() {
        EventBus.getDefault().post(new AudioRouteChangeEvent());
    }
}
