package com.voxeet.sdk.core.services;

import android.support.annotation.NonNull;

import com.voxeet.sdk.core.VoxeetSdk;
import com.voxeet.sdk.core.network.endpoints.ISdkVideoPresentationRService;
import com.voxeet.sdk.core.services.abstracts.AbstractPresentationService;
import com.voxeet.sdk.json.VideoPresentationPaused;
import com.voxeet.sdk.json.VideoPresentationPlay;
import com.voxeet.sdk.json.VideoPresentationSeek;
import com.voxeet.sdk.json.VideoPresentationStarted;
import com.voxeet.sdk.json.VideoPresentationStopped;
import com.voxeet.sdk.utils.Annotate;

import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;

import java.util.ArrayList;
import java.util.List;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;

/**
 * Manage every outgoing or ingoing call related to Video stream from third party or outside files
 */
@Annotate
public class SDKVideoPresentationService extends AbstractPresentationService<ISdkVideoPresentationRService> {

    private List<Solver<VideoPresentationStarted>> mCacheStartedSolvers;
    private List<Solver<VideoPresentationStopped>> mCacheStoppedSolvers;
    private List<Solver<VideoPresentationPaused>> mCachePausedSolvers;
    private List<Solver<VideoPresentationPlay>> mCachePlaySolvers;
    private List<Solver<VideoPresentationSeek>> mCacheSeekSolvers;

    /**
     * Instantiates a new User service.
     *
     * @param instance the parent instance
     */
    public SDKVideoPresentationService(VoxeetSdk instance) {
        super(instance, ISdkVideoPresentationRService.class);

        mCachePausedSolvers = new ArrayList<>();
        mCacheStartedSolvers = new ArrayList<>();
        mCacheStoppedSolvers = new ArrayList<>();
        mCachePausedSolvers = new ArrayList<>();
        mCachePlaySolvers = new ArrayList<>();
        mCacheSeekSolvers = new ArrayList<>();
        registerEventBus();
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(VideoPresentationStarted event) {
        tryUnlock(event, mCacheStartedSolvers);
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(VideoPresentationStopped event) {
        tryUnlock(event, mCacheStoppedSolvers);
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(VideoPresentationPaused event) {
        tryUnlock(event, mCachePausedSolvers);
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(VideoPresentationPlay event) {
        tryUnlock(event, mCachePlaySolvers);
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(VideoPresentationSeek event) {
        tryUnlock(event, mCacheSeekSolvers);
    }

    /**
     * Made a call to change the given video "start" parameter
     *
     * @return the promise to resolve
     */
    @NonNull
    public Promise<VideoPresentationStarted> startVideoPresentation(final String url) {
        return new Promise<>(new PromiseSolver<VideoPresentationStarted>() {
            @Override
            public void onCall(@NonNull final Solver<VideoPresentationStarted> solver) {
                ISdkVideoPresentationRService.VideoPresentationUrl holder = new ISdkVideoPresentationRService.VideoPresentationUrl(url);
                consumeInternalCall(solver, mCacheStartedSolvers,
                        internalCall(getService().startVideoPresentation(getConferenceId(), holder)));
            }
        });
    }

    /**
     * Made a call to change the given video "stop" parameter
     *
     * @return the promise to resolve
     */
    @NonNull
    public Promise<VideoPresentationStopped> stopVideoPresentation() {
        return new Promise<>(new PromiseSolver<VideoPresentationStopped>() {
            @Override
            public void onCall(@NonNull final Solver<VideoPresentationStopped> solver) {

                consumeInternalCall(solver, mCacheStoppedSolvers,
                        internalCall(getService().stopVideoPresentation(getConferenceId())));
            }
        });
    }

    /**
     * Made a call to change the given video "play" parameter
     *
     * @return the promise to resolve
     */
    @NonNull
    public Promise<VideoPresentationPlay> playVideoPresentation() {
        return new Promise<>(new PromiseSolver<VideoPresentationPlay>() {
            @Override
            public void onCall(@NonNull final Solver<VideoPresentationPlay> solver) {

                consumeInternalCall(solver, mCachePlaySolvers,
                        internalCall(getService().playVideoPresentation(getConferenceId())));
            }
        });
    }

    /**
     * Made a call to change the given video "pause" parameter
     *
     * @param timestamp the timestamp the video paused at
     * @return the promise to resolve
     */
    @NonNull
    public Promise<VideoPresentationPaused> pauseVideoPresentation(final long timestamp) {
        return new Promise<>(new PromiseSolver<VideoPresentationPaused>() {
            @Override
            public void onCall(@NonNull final Solver<VideoPresentationPaused> solver) {
                ISdkVideoPresentationRService.VideoPresentationSeek body_sent = new ISdkVideoPresentationRService.VideoPresentationSeek(timestamp);

                consumeInternalCall(solver, mCachePausedSolvers,
                        internalCall(getService().pauseVideoPresentation(getConferenceId(), body_sent)));
            }
        });
    }

    /**
     * Made a call to change the given video "seek" parameter
     *
     * @param timestamp the new timestamp
     * @return the promise to resolve
     */
    @NonNull
    public Promise<VideoPresentationSeek> seekVideoPresentation(final long timestamp) {
        return new Promise<>(new PromiseSolver<VideoPresentationSeek>() {
            @Override
            public void onCall(@NonNull final Solver<VideoPresentationSeek> solver) {

                ISdkVideoPresentationRService.VideoPresentationSeek body_sent = new ISdkVideoPresentationRService.VideoPresentationSeek(timestamp);

                consumeInternalCall(solver, mCacheSeekSolvers,
                        internalCall(getService().seekVideoPresentation(getConferenceId(), body_sent)));
            }
        });
    }

}