package com.voxeet.sdk.core.services;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.sdk.core.VoxeetSdk;
import com.voxeet.sdk.core.network.endpoints.ISdkFilePresentationRService;
import com.voxeet.sdk.core.services.abstracts.AbstractPresentationService;
import com.voxeet.sdk.core.services.videopresentation.AbstractMediaPlayerProvider;
import com.voxeet.sdk.events.error.HttpException;
import com.voxeet.sdk.json.FileConverted;
import com.voxeet.sdk.json.FilePresentationStarted;
import com.voxeet.sdk.json.FilePresentationStopped;
import com.voxeet.sdk.json.FilePresentationUpdated;
import com.voxeet.sdk.models.v1.File;
import com.voxeet.sdk.models.v1.FilePresentationConverted;
import com.voxeet.sdk.utils.Annotate;

import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;

import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.UUID;
import java.util.concurrent.CopyOnWriteArrayList;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;
import okhttp3.MediaType;
import okhttp3.MultipartBody;
import okhttp3.RequestBody;
import okhttp3.ResponseBody;
import retrofit2.Call;
import retrofit2.Callback;
import retrofit2.Response;

@Annotate
public class SDKFilePresentationService extends AbstractPresentationService<ISdkFilePresentationRService> {
    private static final String TAG = SDKFilePresentationService.class.getSimpleName();

    private HashMap<String, Solver<FilePresentationConverted>> mCacheSolvers;
    private HashMap<String, Solver<FilePresentationStarted>> mCacheStartedSolvers;
    private HashMap<String, Solver<FilePresentationStopped>> mCacheStoppedSolvers;
    private HashMap<String, Solver<FilePresentationUpdated>> mCacheUpdatedSolvers;

    /**
     * Instantiates a new User service.
     *
     * @param instance the parent instance
     */
    public SDKFilePresentationService(VoxeetSdk instance) {
        super(instance, ISdkFilePresentationRService.class);

        mCacheSolvers = new HashMap<>();
        mCacheStartedSolvers = new HashMap<>();
        mCacheStoppedSolvers = new HashMap<>();
        mCacheUpdatedSolvers = new HashMap<>();
        registerEventBus();
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(FilePresentationStarted event) {
        tryUnlock(event.fileId, event, mCacheStartedSolvers);
    }


    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(FilePresentationStopped event) {
        tryUnlock(event.getFileId(), event, mCacheStoppedSolvers);
    }


    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(FilePresentationUpdated event) {
        tryUnlock(event.fileId, event, mCacheUpdatedSolvers);
    }


    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(FileConverted converted) {
        if (null != converted) {
            List<File> list = converted.getFiles();

            for (File file : list) {
                Log.d(TAG, "onEvent: " + file.getName());
                String key = findSolverFor(file.getName());
                Log.d(TAG, "onEvent: " + key);
                if (null != key) {
                    FilePresentationConverted temp = new FilePresentationConverted(
                            file.getName(),
                            file.getFileId(),
                            file.getSize(),
                            file.getNbImageConverted());

                    tryUnlock(key, temp, mCacheSolvers);
                }
            }
        }
    }

    /**
     * Get the url for an image
     *
     * @param fileId     the remote file id
     * @param pageNumber the requested page number
     * @return a formatted string
     */
    public String getImage(String fileId, int pageNumber) {
        return String.format(Locale.getDefault(), "%s/v1/files/%s/converted/%d?token=%s",
                getURLRoot(getVoxeetSdkInstance()), fileId, pageNumber, getInternalJwtToken(getVoxeetSdkInstance()));
    }

    /**
     * Get the url for a thumbnail
     *
     * @param fileId     the remote file id
     * @param pageNumber the requested page number
     * @return a formatted string
     */
    public String getThumbnail(String fileId, int pageNumber) {
        return String.format(Locale.getDefault(), "%s/v1/files/%s/converted/%d/thumbnail?token=%s",
                getURLRoot(getVoxeetSdkInstance()), fileId, pageNumber, getInternalJwtToken(getVoxeetSdkInstance()));
    }

    /**
     * Given a local file, starts uploading it and wait for its management server side
     *
     * @param file a valid file to upload
     * @return the promise to resolve
     */
    public Promise<FilePresentationConverted> convertFile(@NonNull final java.io.File file) {
        return new Promise<>(new PromiseSolver<FilePresentationConverted>() {
            @Override
            public void onCall(@NonNull final Solver<FilePresentationConverted> solver) {
                final String uuid = UUID.randomUUID().toString();

                String appended_name = uuid + file.getName();
                RequestBody requestFile = RequestBody.create(MediaType.parse("multipart/form-data"), file);

                // MultipartBody.Part is used to send also the actual file name
                MultipartBody.Part body = MultipartBody.Part.createFormData("file", appended_name, requestFile);


                // finally, execute the request
                getService().convertFile(requestFile, body)
                        .enqueue(new Callback<ResponseBody>() {
                            @Override
                            public void onResponse(Call<ResponseBody> call, Response<ResponseBody> response) {
                                //will wait for conversion result, so we put into cache...
                                if(response.isSuccessful()) {
                                    mCacheSolvers.put(uuid, solver);
                                } else {
                                    solver.reject(HttpException.throwResponse(response));
                                }
                            }

                            @Override
                            public void onFailure(Call<ResponseBody> call, Throwable e) {
                                Log.d(TAG, "onError: " + e);
                                solver.reject(e);
                            }
                        });
            }
        });
    }

    /**
     * Start a presentation for a given uploaded and managed file
     *
     * @param body the valid descriptor for the file
     * @return the promise to resolve
     */
    @NonNull
    public Promise<FilePresentationStarted> startPresentation(@NonNull final FilePresentationConverted body) {
        return startPresentation(body, 0);
    }


    /**
     * Start a presentation for a given uploaded and managed file. Also point to a given page (represented by the position)
     *
     * @param body the valid descriptor for the file
     * @param position the position to start it
     * @return the promise to resolve
     */
    @NonNull
    public Promise<FilePresentationStarted> startPresentation(@NonNull final FilePresentationConverted body,
                                                              final int position) {
        return new Promise<>(new PromiseSolver<FilePresentationStarted>() {
            @Override
            public void onCall(@NonNull final Solver<FilePresentationStarted> solver) {
                ISdkFilePresentationRService.FilePresentationId body_sent = new ISdkFilePresentationRService.FilePresentationId(body.getFileId(), body.getName(), position, body.getNbImageConverted());

                consumeInternalCall(solver, body.getFileId(), mCacheStartedSolvers,
                        internalCall(getService().startFilePresentation(getConferenceId(), body_sent)));
            }
        });
    }


    /**
     * Stop a presentation for a given uploaded and managed file
     *
     * @param fileId the fileId to manage
     * @return the promise to resolve
     */
    @NonNull
    public Promise<FilePresentationStopped> stopPresentation(@NonNull final String fileId) {
        return new Promise<>(new PromiseSolver<FilePresentationStopped>() {
            @Override
            public void onCall(@NonNull final Solver<FilePresentationStopped> solver) {
                ISdkFilePresentationRService.FilePresentationId body_sent = new ISdkFilePresentationRService.FilePresentationId(fileId);

                consumeInternalCall(solver, fileId, mCacheStoppedSolvers,
                        internalCall(getService().startFilePresentation(getConferenceId(), body_sent)));
            }
        });
    }

    /**
     * Go to a specific position in the current presentation
     *
     * @param fileId the fileId
     * @param position the position to go to
     * @return the promise to resolve
     */
    @NonNull
    public Promise<FilePresentationUpdated> updatePresentation(@NonNull final String fileId,
                                                               final int position) {
        return new Promise<>(new PromiseSolver<FilePresentationUpdated>() {
            @Override
            public void onCall(@NonNull final Solver<FilePresentationUpdated> solver) {

                ISdkFilePresentationRService.FilePresentationId body_sent = new ISdkFilePresentationRService.FilePresentationId(fileId, position);

                consumeInternalCall(solver, fileId, mCacheUpdatedSolvers,
                        internalCall(getService().startFilePresentation(getConferenceId(), body_sent)));
            }
        });
    }

    @Nullable
    private String findSolverFor(@NonNull String name) {
        for (String value : mCacheSolvers.keySet())
            if (name.indexOf(value) == 0) return value;
        return null;
    }
}