package com.voxeet.sdk.core.services;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.android.media.stats.LocalStats;
import com.voxeet.sdk.core.AbstractVoxeetService;
import com.voxeet.sdk.core.VoxeetSdk;
import com.voxeet.sdk.core.services.holder.ServiceProviderHolder;
import com.voxeet.sdk.core.services.localstats.LocalStatsHandler;
import com.voxeet.sdk.core.services.localstats.LocalStatsUserInfo;
import com.voxeet.sdk.events.sdk.QualityIndicators;
import com.voxeet.sdk.json.ConferenceStats;
import com.voxeet.sdk.media.MediaSDK;
import com.voxeet.sdk.models.Conference;
import com.voxeet.sdk.models.User;
import com.voxeet.sdk.utils.Annotate;

import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;

import java.util.HashMap;
import java.util.List;

/**
 * Service responsible for statistics management from the WebRTC implementation. Every statistics are local only
 */
@Annotate
public class LocalStatsService extends AbstractVoxeetService implements Runnable {
    public final static int NEXT_LOOP_FETCH_MS = 500;

    private VoxeetSdk mInstance;

    private LocalStatsHandler mHandler;
    private HashMap<String, HashMap<String, LocalStatsUserInfo>> mConferencesHolder;

    public LocalStatsService(@NonNull VoxeetSdk instance) {
        super(instance, ServiceProviderHolder.DEFAULT);

        mConferencesHolder = new HashMap<>();
        mHandler = null;
        mInstance = instance;
    }

    /**
     * Retrieve the in-memory holder of the user specifically designed for the user
     *
     * @param conferenceId the conferenceId - not the alias
     * @param userId       the userId - not the externalId
     * @return a valid user info holder
     */
    public LocalStatsUserInfo getUserInfo(@NonNull String conferenceId, @NonNull String userId) {
        HashMap<String, LocalStatsUserInfo> holder = getUserStatsHolder(conferenceId);
        LocalStatsUserInfo info = holder.get(userId);

        if (null == info) {
            info = new LocalStatsUserInfo(userId);
            holder.put(userId, info);
        }
        return info;
    }

    /**
     * Get the specified user stats from a local point of view
     *
     * @param peer the user id
     * @return LocalStats or null
     */
    @Nullable
    public LocalStats getLocalStats(@Nullable String peer) {
        if (null == peer) return null;
        MediaSDK media = VoxeetSdk.mediaDevice().getMedia();
        if (null != media) return media.getLocalStats(peer);
        return null;
    }

    /**
     * Start an autofetch loop which will get the stats information at various times
     *
     * @return an indicator telling if the session is now started or already started
     */
    public boolean startAutoFetch() {
        if (null == mHandler) {
            mHandler = new LocalStatsHandler(this, NEXT_LOOP_FETCH_MS);
            mHandler.startAutoFetch();
            return true;
        } else {
            return false;
        }
    }

    /**
     * Stop any autofetching loop
     *
     * @return an indicator telling if it's now stoppped or was already stopped
     */
    public boolean stopAutoFetch() {
        if (null != mHandler) {
            mHandler.stopAutoFetch();
            mHandler = null;
            return true;
        }
        return false;
    }

    @Override
    public void run() {
        //List<LocalStats> stats = new ArrayList<>();
        Conference conference = VoxeetSdk.conference().getConference();

        if (null != conference) {
            List<User> users = conference.getUsers();
            for (int i = 0; i < users.size(); i++) {
                User user = users.get(i);
                LocalStats stat = getLocalStats(user.getId());
                getUserInfo(conference.getId(), user.getId()).addFromLocalStats(stat);
            }
        } else {
            //no conference ?
        }
    }

    /**
     * Retrieve the in-memory holder of the conference information specifically designed for the stats
     *
     * @param conferenceId the conferenceId - not the alias
     * @return a non null HashMap instance
     */
    @NonNull
    private HashMap<String, LocalStatsUserInfo> getUserStatsHolder(@NonNull String conferenceId) {
        HashMap<String, LocalStatsUserInfo> holder = mConferencesHolder.get(conferenceId);

        if (null == holder) {
            holder = new HashMap<>();
            mConferencesHolder.put(conferenceId, holder);
        }
        return holder;
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceStats stats) {
        String conferenceId = VoxeetSdk.conference().getConferenceId();
        if (stats.getConference_id() != null && stats.getConference_id().equals(conferenceId)) {
            float mos = stats.getScore(VoxeetSdk.user().getUserId());

            getEventBus().post(new QualityIndicators(mos));
        }
    }
}
