package com.voxeet.sdk.core.services;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.audio.AudioRoute;
import com.voxeet.sdk.core.AbstractVoxeetService;
import com.voxeet.sdk.core.VoxeetSdk;
import com.voxeet.sdk.core.services.holder.ServiceProviderHolder;
import com.voxeet.sdk.events.sdk.ConferencePreJoinedEvent;
import com.voxeet.sdk.json.ConferenceDestroyedPush;
import com.voxeet.sdk.json.ConferenceEnded;
import com.voxeet.sdk.media.audio.SoundManager;
import com.voxeet.sdk.media.sensors.ConferenceLock;
import com.voxeet.sdk.media.sensors.ProximitySensor;
import com.voxeet.sdk.media.sensors.ScreenSensor;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.AudioType;
import com.voxeet.sdk.utils.NoDocumentation;

import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;
import org.webrtc.voiceengine.WebRtcAudioUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * Manage and server access to the various Audio related API in the Voxeet SDK
 */
@Annotate
public class AudioService extends AbstractVoxeetService {

    private static SoundManager sSoundManager;
    private static final String TAG = AudioService.class.getSimpleName();
    private List<ConferenceLock> locks = new ArrayList<>();

    /**
     * Instantiate a new Audio Service
     *
     * @param instance the voxeet parent instance
     */
    @NoDocumentation
    public AudioService(@NonNull VoxeetSdk instance) {
        super(instance, ServiceProviderHolder.DEFAULT);

        AudioService.preInitSounds(instance.getApplicationContext());
        instance.getEventBus().register(this);

        locks.add(new ProximitySensor(instance.getApplicationContext()));
        locks.add(new ScreenSensor(instance.getApplicationContext()));
    }

    /**
     * Retrieve the static SoundManager
     *
     * @return the instance of the SoundManager or null if it has not been initialized yet
     */
    @Nullable
    public static SoundManager getSoundManager() {
        return sSoundManager;
    }

    /**
     * Given an ApplicationContext, try to initialize the SoundManager. To use for environment where the SDK may not be initialized yet such as Cordova or React Native
     *
     * @param applicationContext a valid instance
     * @return a boolean indicating if it has been initialized or if it was already initialized
     */
    public static boolean preInitSounds(@NonNull Context applicationContext) {
        if (null == sSoundManager) {
            sSoundManager = new SoundManager(applicationContext);
            return true;
        }

        return false;
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferencePreJoinedEvent event) {
        enableAec(true);
        enableNoiseSuppressor(true);

        updateSensors(currentRoute());
        sSoundManager.onConferencePreJoinedEvent();
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceEnded event) {
        sSoundManager.onConferenceDestroyedPush();
        releaseSensors();
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceDestroyedPush event) {
        sSoundManager.onConferenceDestroyedPush();
        releaseSensors();
    }

    /**
     * Get the current available routes
     *
     * @return the list of all possible routes
     */
    @NonNull
    public List<AudioRoute> getAvailableRoutes() {
        return sSoundManager.getAvailableRoutes();
    }

    /**
     * Get the current route used by the SoundManager instance
     *
     * @return the current selected route
     */
    @NonNull
    public AudioRoute currentRoute() {
        return sSoundManager.currentRoute();
    }

    /**
     * Change the AudioRoute
     * This method will also update the various states of the sensors
     *
     * @param route the new route to use
     * @return a boolean indicating if a switch has been made
     */
    public boolean setAudioRoute(@NonNull AudioRoute route) {
        boolean new_switch = sSoundManager.setAudioRoute(route);
        updateSensors(currentRoute());

        return new_switch;
    }

    /**
     * Check for any route side effect to be managed internally
     *
     * @return the current instance
     */
    @NonNull
    public AudioService checkOutputRoute() {
        sSoundManager.checkOutputRoute();
        return this;
    }

    /**
     * Change the mode to speaker
     *
     * @param isSpeaker activate or deactivate the speakers
     */
    public void setSpeakerMode(boolean isSpeaker) {
        sSoundManager.setSpeakerMode(isSpeaker);
        updateSensors(currentRoute());
    }

    /**
     * Get the current bluetooth device connectivity
     *
     * @return a boolean indicating if a device is currently connected or not(/connecting)
     */
    public boolean isBluetoothHeadsetConnected() {
        return sSoundManager.isBluetoothHeadsetConnected();
    }

    /**
     * Get the current wired headset connectivity state
     *
     * @return a boolean indicating if wired headsets are connected to the device
     */
    public boolean isWiredHeadsetOn() {
        return sSoundManager.isWiredHeadsetOn();
    }

    /**
     * Check for the speaker mode
     * @return a boolean indicating if the speakers are activated
     */
    public boolean isSpeakerOn() {
        return sSoundManager.isSpeakerOn();
    }

    /**
     * Acquire the system's locks for the sensors
     * @return the current instance
     */
    @NonNull
    public AudioService acquireLocks() {
        sSoundManager.acquireLocks();
        return this;
    }

    /**
     * Release the system's locks of the sensors
     * @return the current instance
     */
    @NonNull
    public AudioService releaseLocks() {
        sSoundManager.releaseLocks();
        return this;
    }

    /**
     * Reset the system audio to media/system
     * @return the current instance
     */
    @NonNull
    public AudioService resetDefaultSoundType() {
        sSoundManager.resetDefaultSoundType();
        return this;
    }

    /**
     * Abandon the current focus made onto the Audio system type
     *
     * @return the current instance
     */
    @NonNull
    public AudioService abandonAudioFocusRequest() {
        sSoundManager.abandonAudioFocusRequest();
        return this;
    }

    /**
     * Ask for a focus onto the current available/valid AudioRoute
     *
     * @return the current instance
     */
    @NonNull
    public AudioService requestAudioFocus() {
        sSoundManager.requestAudioFocus();
        return this;
    }

    /**
     * Given a sound type, attach a specific asset from its name into the phone mode
     *
     * @param type the Sound Type to attach to
     * @param assetName the asset's name
     * @return the attach success
     */
    public boolean setSound(@NonNull AudioType type, @NonNull String assetName) {
        return sSoundManager.setSound(type, assetName);
    }

    /**
     * Given a sound type, attach a specific asset from its name and set it to the specific sound mode pool
     * @param type the Sound Type to attach to
     * @param assetName the asset's name
     * @param soundMode the sound mode which will reflect the pool mode to link
     * @return the attach success
     */
    public boolean setSound(@NonNull AudioType type, @NonNull String assetName, int soundMode) {
        return sSoundManager.setSound(type, assetName, soundMode);
    }

    /**
     * Play a given Audio from its type into the phone mode
     * @param type the AudioType to play
     * @return the current instance
     */
    @NonNull
    public AudioService playSoundType(@NonNull AudioType type) {
        sSoundManager.playSoundType(type);
        return this;
    }

    /**
     * Play a given Audio from its type
     * @param type the AudioType to play
     * @param soundMode the sound mode to use
     * @return the current instance
     */
    @NonNull
    public AudioService playSoundType(@NonNull AudioType type, int soundMode) {
        sSoundManager.playSoundType(type, soundMode);
        return this;
    }

    /**
     * Stop a sound given its type into the phone mode
     * @param audioType the requested type
     * @return
     */
    @NonNull
    public AudioService stopSoundType(@NonNull AudioType audioType) {
        sSoundManager.stopSoundType(audioType);
        return this;
    }

    /**
     * Stop a sound given its type and mode
     *
     * @param audioType the requested type
     * @param soundMode the mode it was working on
     * @return the current instance
     */
    @NonNull
    public AudioService stopSoundType(@NonNull AudioType audioType, int soundMode) {
        sSoundManager.stopSoundType(audioType, soundMode);
        return this;
    }

    /**
     * Stop the service but does not free the holds or ressources
     *
     * @return the current instance
     */
    @NonNull
    public AudioService stop() {
        sSoundManager.stop();
        return this;
    }

    /**
     * Enable the service letting it manage sounds, mode modification etc...
     *
     * @return the current instance
     */
    @NonNull
    public AudioService enable() {
        sSoundManager.enable();
        return this;
    }

    /**
     * Enable the media mode in the manager, does not hold it but will accept various implementation feature
     * @return the current instance
     */
    @NonNull
    public AudioService enableMedia() {
        sSoundManager.enableMedia();
        return this;
    }

    /**
     * Unset the Media mode, does not restore the phone mode but won't hold any specific internal feature
     * @return the current instance
     */
    @NonNull
    public AudioService unsetMediaRoute() {
        sSoundManager.unsetMediaRoute();
        return this;
    }

    /**
     * Change the Audio output to the Media mode (instead of the Phone mode)
     * @return the current instance
     */
    @NonNull
    public AudioService setMediaRoute() {
        sSoundManager.setMediaRoute();
        return this;
    }

    /**
     * Disable the AudioService and stop the various elements in it
     * @return the current instance
     */
    @NonNull
    public AudioService disable() {
        sSoundManager.disable();
        stop();
        return this;
    }

    /**
     * Update the states of the sensors given a specified AudioRoute
     *
     * @param route a valid route to update the sensors with
     */
    public void updateSensors(@NonNull AudioRoute route) {
        for (ConferenceLock lock : locks) {
            if (!route.useProximitySensor() && lock.isProximity()) {
                lock.release();
            } else {
                lock.acquire();
            }
        }
    }

    /**
     * Release the sensors
     */
    public void releaseSensors() {
        Log.d("ProximitySensor", "releaseSensors: ");
        for (ConferenceLock lock : locks) {
            lock.release();
        }
    }

    /**
     * Activate or deactivate the AEC mode
     *
     * @param enable the new state to use
     * @return success or failure
     */
    public boolean enableAec(boolean enable) {
        WebRtcAudioUtils.setWebRtcBasedAcousticEchoCanceler(enable);
        return true;
    }

    /**
     * Activate or deactivate the NS mode
     *
     * @param enable the new state to use
     * @return success or failure
     */
    public boolean enableNoiseSuppressor(boolean enable) {
        WebRtcAudioUtils.setWebRtcBasedNoiseSuppressor(enable);
        return true;
    }
}
