package com.voxeet.sdk.core;

import android.content.Context;
import android.content.Intent;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.util.Log;

import com.voxeet.sdk.core.preferences.VoxeetPreferences;
import com.voxeet.sdk.core.services.AudioService;
import com.voxeet.sdk.core.services.ChatService;
import com.voxeet.sdk.core.services.ConferenceService;
import com.voxeet.sdk.core.services.LocalStatsService;
import com.voxeet.sdk.core.services.MediaDeviceService;
import com.voxeet.sdk.core.services.SDKFilePresentationService;
import com.voxeet.sdk.core.services.SDKVideoPresentationService;
import com.voxeet.sdk.core.services.ScreenShareService;
import com.voxeet.sdk.core.services.UserService;
import com.voxeet.sdk.core.services.authenticate.WebSocketState;
import com.voxeet.sdk.core.services.authenticate.token.RefreshTokenCallback;
import com.voxeet.sdk.events.sdk.SocketStateChangeEvent;
import com.voxeet.sdk.exceptions.ExceptionManager;
import com.voxeet.sdk.exceptions.VoxeetSentry;
import com.voxeet.sdk.factories.VoxeetIntentFactory;
import com.voxeet.sdk.json.ConferenceDestroyedPush;
import com.voxeet.sdk.utils.AbstractVoxeetEnvironmentHolder;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.NoDocumentation;
import com.voxeet.sdk.utils.Validate;

import org.greenrobot.eventbus.EventBus;
import org.greenrobot.eventbus.EventBusException;

import java.net.UnknownHostException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import eu.codlab.simplepromise.Promise;
import okhttp3.OkHttpClient;
import retrofit2.Retrofit;

@Annotate
public final class VoxeetSdk {

    private final static String TAG = VoxeetSdk.class.getSimpleName();
    private final static long DEFAULT_TIMEOUT_MS = 60 * 1000;

    private static Context CurrentApplication = null;
    private static VoxeetSdk CurrentInstance;

    private VoxeetHttp _voxeet_http;
    private Context _application_context;
    private EventBus _event_bus;
    private AbstractVoxeetEnvironmentHolder _voxeet_environment_holder;
    private HashMap<Class<? extends AbstractVoxeetService>, AbstractVoxeetService> _services;
    private VoxeetSentry voxeetSentry;

    protected static void setInstance(@NonNull VoxeetSdk sdk) {
        CurrentInstance = sdk;

        Validate.hasInternetPermissions(sdk.getApplicationContext(), true);
        VoxeetPreferences.init(sdk.getApplicationContext(),
                sdk.getVoxeetEnvironmentHolder());
    }

    @Nullable
    public static VoxeetSdk instance() {
        return CurrentInstance;
    }

    public static void setApplication(@NonNull Context application) {
        CurrentApplication = application;
    }

    /**
     * This function initializes the Voxeet SDK, the behavior of Voxeet SDK functions are
     * undetermined if this function is not called. It should be called as early as possible.
     * <p>
     * Using a null value for the UserInfo? Prefer the signature without it then
     *
     * @param appId    the app id
     * @param password the password
     */
    public static synchronized void initialize(@NonNull String appId,
                                               @NonNull String password) {
        if (null == CurrentInstance) {
            VoxeetSdk sdk = new VoxeetSdk(CurrentApplication,
                    appId,
                    password,
                    null,
                    null,
                    new VoxeetEnvironmentHolder(CurrentApplication),
                    false);

            VoxeetSdk.setInstance(sdk);
        } else {
            Log.d(TAG, "initialize: Instance already started !");
        }
    }

    /**
     * This function initializes the Voxeet SDK, the behavior of Voxeet SDK functions are
     * undetermined if this function is not called. It should be called as early as possible.
     * <p>
     * This method will invoke the SDK with the thirdparty authentication feature
     * <p>
     * Using a null value for the UserInfo? Prefer the signature without it then
     *
     * @param accessToken  the default accessToken to use
     * @param refreshToken the refreshToken callback
     */
    public static synchronized void initialize(@NonNull String accessToken,
                                               @NonNull RefreshTokenCallback refreshToken) {
        if (null == CurrentInstance) {
            VoxeetSdk sdk = new VoxeetSdk(CurrentApplication,
                    null,
                    null,
                    accessToken,
                    refreshToken,
                    new VoxeetEnvironmentHolder(CurrentApplication),
                    false);
            VoxeetSdk.setInstance(sdk);
        } else {
            Log.d(TAG, "initialize: Instance already started !");
        }
    }

    private VoxeetSdk(@NonNull Context application_context,
                      @Nullable String appId,
                      @Nullable String password,
                      @Nullable String tokenAccess,
                      @Nullable RefreshTokenCallback tokenRefresh,
                      @NonNull AbstractVoxeetEnvironmentHolder holder,
                      boolean debug) {
        initExceptionIfOk(application_context);

        _application_context = application_context;
        _event_bus = EventBus.getDefault();

        //make sure the looper is the main looper for promise execution
        Promise.setHandler(new Handler(Looper.getMainLooper()));

        Log.d("VoxeetSDK", "VoxeetSdk: version := " + holder.getVersionName());

        _voxeet_environment_holder = holder;

        VoxeetHttp.Builder builder = new VoxeetHttp.Builder()
                .setApplication(application_context)
                .setVoxeetSDK(this)
                .setServerUrl(_voxeet_environment_holder.getServerUrl())
                .setServerPort(_voxeet_environment_holder.getServerPort())
                .setAppId(appId)
                .setPassword(password)
                .setTokenAccess(tokenAccess)
                .setTokenRefresh(tokenRefresh)
                .setVoxeetServiceListener(new VoxeetHttp.VoxeetServiceListener() {

                    @Override
                    public void onIdentifySuccess(VoxeetHttp provider) {
                        user().connectSocket(provider);
                    }

                    @Override
                    public void onIdentifyError(String error) {
                        if (null == error) error = "identfify error";
                        _event_bus.post(new SocketStateChangeEvent(WebSocketState.CLOSED.name()));
                    }

                    @Override
                    public void onNetworkError(UnknownHostException error) {
                        _event_bus.post(new SocketStateChangeEvent(WebSocketState.CLOSED.name()));
                    }
                })
                .setDebug(debug);

        _voxeet_http = builder.build();


        _services = new HashMap<>();

        VoxeetPreferences.init(getApplicationContext(), getVoxeetEnvironmentHolder());
        initServices();
    }

    @Nullable
    public static ChatService chat() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(ChatService.class);
    }

    @Nullable
    public static ConferenceService conference() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(ConferenceService.class);
    }

    @Nullable
    public static MediaDeviceService mediaDevice() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(MediaDeviceService.class);
    }

    @Nullable
    public static ScreenShareService screenShare() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(ScreenShareService.class);
    }

    @Nullable
    public static UserService user() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(UserService.class);
    }

    @Nullable
    public static SDKFilePresentationService filePresentation() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(SDKFilePresentationService.class);
    }

    @Nullable
    public static SDKVideoPresentationService videoPresentation() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(SDKVideoPresentationService.class);
    }

    @Nullable
    public static AudioService audio() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(AudioService.class);
    }

    @Nullable
    public static LocalStatsService localStats() {
        if (null == instance()) return null;
        return instance().getServiceForKlass(LocalStatsService.class);
    }

    @NoDocumentation
    public boolean manageRemoteMessage(@NonNull Context context, @NonNull Map<String, String> remoteMessage) {
        if (remoteMessage.containsKey(VoxeetIntentFactory.NOTIF_TYPE)) {
            String notificationType = remoteMessage.get(VoxeetIntentFactory.NOTIF_TYPE);
            if (null == notificationType || TextUtils.isEmpty(notificationType))
                notificationType = "";

            switch (notificationType) {
                case VoxeetIntentFactory.NOTIF_TYPE_INVITATION_RECEIVED:
                    Intent intent = VoxeetIntentFactory.buildFrom(context, VoxeetPreferences.getDefaultActivity(), remoteMessage);
                    if (intent != null)
                        getApplicationContext().startActivity(intent);
                    break;
                case VoxeetIntentFactory.NOTIF_TYPE_LOCAL_CONFERENCE_DESTROYED:
                    String conferenceId = "";
                    if (remoteMessage.containsKey(VoxeetIntentFactory.CONF_ID)) {
                        conferenceId = remoteMessage.get(VoxeetIntentFactory.CONF_ID);
                    }
                    if (null == conferenceId || TextUtils.isEmpty(conferenceId)) conferenceId = "";
                    getEventBus().post(new ConferenceDestroyedPush(conferenceId));
                    break;
                default:
            }

            return true;
        }
        return false;
    }

    public Retrofit getRetrofit() {
        return _voxeet_http.getRetrofit();
    }

    public OkHttpClient getClient() {
        return _voxeet_http.getClient();
    }

    public Context getApplicationContext() {
        return _application_context;
    }

    public AbstractVoxeetEnvironmentHolder getVoxeetEnvironmentHolder() {
        return _voxeet_environment_holder;
    }

    public EventBus getEventBus() {
        return _event_bus;
    }

    /**
     * Register object onto the SDK. To be called before creating/joining any conference to trigger events
     *
     * @param subscriber the subscriber
     */
    public boolean register(@Nullable Object subscriber) {
        try {
            //MediaSDK.setContext(context);
            EventBus eventBus = getEventBus();

            if (null != subscriber && !eventBus.isRegistered(subscriber))
                eventBus.register(subscriber);
        } catch (EventBusException error) {
            error.printStackTrace();
            return false;
        }

        return true;
    }

    /**
     * Unregister Media. To be called in the on destroy of your activty/fragment
     *
     * @param subscriber the subscriber
     */
    public void unregister(@NonNull Object subscriber) {
        try {
            if (getEventBus().isRegistered(subscriber))
                getEventBus().unregister(subscriber);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    protected HashMap<Class<? extends AbstractVoxeetService>, AbstractVoxeetService> getServices() {
        return _services;
    }

    protected VoxeetHttp getVoxeetHttp() {
        return _voxeet_http;
    }

    private void initExceptionIfOk(@NonNull Context application_context) {
        Log.d(TAG, "initExceptionIfOk: checking for exception to log internally");
        voxeetSentry = new VoxeetSentry(application_context);
        ExceptionManager.register(voxeetSentry);
        Log.d(TAG, "initExceptionIfOk: finished to try implementing error management");
    }

    private <T extends AbstractVoxeetService> T getServiceForKlass(Class<T> klass) {
        T service = null;
        Iterator<Class<? extends AbstractVoxeetService>> iterator = _services.keySet().iterator();
        while (iterator.hasNext()) {
            Class<? extends AbstractVoxeetService> next = iterator.next();
            AbstractVoxeetService item = _services.get(next);


            if (klass.isInstance(item)) {
                service = (T) item;
            }
        }

        if (service == null) {
            Log.d(TAG, klass.getSimpleName() + " not found in the list of services");
        }
        return service;
    }

    private void initServices() {
        getServices().put(ConferenceService.class, new ConferenceService(this, DEFAULT_TIMEOUT_MS));
        getServices().put(UserService.class, new UserService(this));
        getServices().put(SDKFilePresentationService.class, new SDKFilePresentationService(this));
        getServices().put(SDKVideoPresentationService.class, new SDKVideoPresentationService(this));
        getServices().put(AudioService.class, new AudioService(this));
        getServices().put(ScreenShareService.class, new ScreenShareService(this));
        getServices().put(MediaDeviceService.class, new MediaDeviceService(this));
        getServices().put(LocalStatsService.class, new LocalStatsService(this));
        getServices().put(ChatService.class, new ChatService(this));
    }


}
