package com.voxeet.sdk.core;

import android.util.Log;

import com.google.gson.Gson;
import com.voxeet.sdk.core.network.websocket.VoxeetWebSocket;
import com.voxeet.sdk.events.BaseEvent;
import com.voxeet.sdk.events.error.SocketErrorEvent;
import com.voxeet.sdk.events.sdk.MessageReceived;
import com.voxeet.sdk.events.sdk.SocketConnectEvent;
import com.voxeet.sdk.events.sdk.SocketDisconnectEvent;
import com.voxeet.sdk.events.sdk.SocketStateChangeEvent;
import com.voxeet.sdk.events.success.ConferenceUpdated;
import com.voxeet.sdk.events.websocket.RenegociationEndedEvent;
import com.voxeet.sdk.factories.EventsFactory;
import com.voxeet.sdk.json.BadgeUpdatedEvent;
import com.voxeet.sdk.json.BroadcastEvent;
import com.voxeet.sdk.json.ConferenceDestroyedPush;
import com.voxeet.sdk.json.ConferenceEnded;
import com.voxeet.sdk.json.ConferenceMessageReceived;
import com.voxeet.sdk.json.ConferenceStats;
import com.voxeet.sdk.json.ParticipantAdded;
import com.voxeet.sdk.json.ConferenceUserSwitch;
import com.voxeet.sdk.json.ContactAdded;
import com.voxeet.sdk.json.Event;
import com.voxeet.sdk.json.EventNames;
import com.voxeet.sdk.json.FileAddedEvent;
import com.voxeet.sdk.json.FileConverted;
import com.voxeet.sdk.json.FileDeletedEvent;
import com.voxeet.sdk.json.FilePresentationStarted;
import com.voxeet.sdk.json.FilePresentationStopped;
import com.voxeet.sdk.json.FilePresentationUpdated;
import com.voxeet.sdk.json.InterfaceEvent;
import com.voxeet.sdk.json.InvitationReceivedEvent;
import com.voxeet.sdk.json.MeetActivityAddedEvent;
import com.voxeet.sdk.json.MeetActivityDeletedEvent;
import com.voxeet.sdk.json.MeetActivityUpdatedEvent;
import com.voxeet.sdk.json.MeetingAddedOrUpdatedEvent;
import com.voxeet.sdk.json.MeetingDeletedEvent;
import com.voxeet.sdk.json.MeetingEventAddedOrUpdatedEvent;
import com.voxeet.sdk.json.MeetingReadTimeStampUpdatedEvent;
import com.voxeet.sdk.json.OfferCreated;
import com.voxeet.sdk.json.OwnConferenceCreated;
import com.voxeet.sdk.json.OwnConferenceUserSwitch;
import com.voxeet.sdk.json.OwnContactRemovedEvent;
import com.voxeet.sdk.json.OwnExternalInvitationSentEvent;
import com.voxeet.sdk.json.OwnProfileUpdated;
import com.voxeet.sdk.json.OwnUserInvited;
import com.voxeet.sdk.json.ParticipantUpdated;
import com.voxeet.sdk.json.PeerConnectionStatusUpdated;
import com.voxeet.sdk.json.PlayEmoSphere;
import com.voxeet.sdk.json.QualityUpdated;
import com.voxeet.sdk.json.RecordingStatusUpdateEvent;
import com.voxeet.sdk.json.TypingDetection;
import com.voxeet.sdk.json.UserInvited;
import com.voxeet.sdk.json.VideoPresentationPaused;
import com.voxeet.sdk.json.VideoPresentationPlay;
import com.voxeet.sdk.json.VideoPresentationSeek;
import com.voxeet.sdk.json.VideoPresentationStarted;
import com.voxeet.sdk.json.VideoPresentationStopped;
import com.voxeet.sdk.json.WhisperInvitationAccepted;
import com.voxeet.sdk.json.WhisperInvitationDeclined;
import com.voxeet.sdk.json.WhisperInvitationReceived;
import com.voxeet.sdk.json.WhisperLeft;

import org.greenrobot.eventbus.EventBus;

public class VoxeetDispatcher {

    private static final String TAG = VoxeetDispatcher.class.getSimpleName();

    private static EventBus eventBus = EventBus.getDefault();

    /**
     * Map and dispatch events through the eventbus.
     *
     * @param eventName the event name
     * @param message   the message
     */
    public static void dispatch(String eventName, String message) {
        Log.d(TAG, "dispatch: " + eventName + " := " + message);
        try {
            InterfaceEvent event;
            switch (eventName) {
                case VoxeetWebSocket.SOCKET_DISCONNECTED:
                    event = handleDisconnect(message);
                    break;
                case VoxeetWebSocket.SOCKET_ERROR:
                    event = handleError(message);
                    break;
                case VoxeetWebSocket.SOCKET_MESSAGE:
                    event = handleMessage(message);
                    break;
                case VoxeetWebSocket.SOCKET_CONNECTED:
                    event = handleSocketConnect(message);
                    break;
                case VoxeetWebSocket.SOCKET_STATE_CHANGE:
                    event = handleStateChange(message);
                    break;
                default:
                    throw new IllegalStateException("Unknown event type");
            }

            eventBus.post(event);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * Dispatch a socket error event.
     */
    private static BaseEvent handleError(String message) {
        return new SocketErrorEvent(message);
    }

    /**
     * Dispatch a socket disconnected event.
     */
    private static BaseEvent handleDisconnect(String message) {
        return new SocketDisconnectEvent(message);
    }

    /**
     * Dispatch a socket state change event.
     */
    private static BaseEvent handleStateChange(String message) {
        return new SocketStateChangeEvent(message);
    }

    /**
     * Dispatch a socket connected event.
     */
    private static BaseEvent handleSocketConnect(String message) {
        return new SocketConnectEvent(message);
    }

    /**
     * Dispatch all other events.
     */
    private static InterfaceEvent handleMessage(String message) {
        InterfaceEvent busEvent;

        Event event = EventsFactory.decode(message);
        switch (event.getType()) {
            case EventNames.OFFER_CREATED:
                busEvent = (OfferCreated) event;
                break;
            case EventNames.CONFERENCE_MESSAGE_RECEIVED:
                busEvent = handleConferenceMessage((ConferenceMessageReceived) event);
                break;
            case EventNames.CONFERENCE_DESTROYED:
                busEvent = (ConferenceDestroyedPush) event;
                break;
            case EventNames.CONTACT_PROFILE_UPDATED:
                busEvent = event;
                break;
            case EventNames.PARTICIPANT_UPDATED:
                busEvent = (ParticipantUpdated) event;
                break;
            case EventNames.OWN_CONFERENCE_CREATED:
                busEvent = (OwnConferenceCreated) event;
                break;
            case EventNames.OWN_PROFILE_UPDATED:
                busEvent = (OwnProfileUpdated) event;
                break;
            case EventNames.OWN_EXTERNAL_INVITATION:
                busEvent = (OwnExternalInvitationSentEvent) event;
                break;
            case EventNames.OWN_CONTACT_REMOVED:
                busEvent = (OwnContactRemovedEvent) event;
                break;
            case EventNames.MEETING_ADDED_OR_UPDATED:
                busEvent = (MeetingAddedOrUpdatedEvent) event;
                break;
            case EventNames.MEETING_DELETED:
                busEvent = (MeetingDeletedEvent) event;
                break;
            case EventNames.BADGE_UPDATE_EVENT:
                busEvent = (BadgeUpdatedEvent) event;
                break;
            case EventNames.RENEGOCIATION_ENDED:
                busEvent = (RenegociationEndedEvent) event;
                break;
            case EventNames.BROADCAST_EVENT:
                busEvent = handleBroadcast((BroadcastEvent) event);
                break;
            case EventNames.RECORDING_STATUS_UPDATE:
                busEvent = (RecordingStatusUpdateEvent) event;
                break;
            case EventNames.MEET_ACTIVITY_DELETED:
                busEvent = (MeetActivityDeletedEvent) event;
                break;
            case EventNames.MEET_ACTIVITY_UPDATED:
                busEvent = (MeetActivityUpdatedEvent) event;
                break;
            case EventNames.MEET_ACTIVITY_ADDED:
                busEvent = (MeetActivityAddedEvent) event;
                break;
            case EventNames.MEET_READ_TIMESTAMP_UPDATED:
                busEvent = (MeetingReadTimeStampUpdatedEvent) event;
                break;
            case EventNames.INVITATION_RECEIVED:
                busEvent = (InvitationReceivedEvent) event;
                break;
            case EventNames.MEETING_EVENT_ADDED_OR_UPDATED:
                busEvent = (MeetingEventAddedOrUpdatedEvent) event;
                break;
            case EventNames.FILE_DELETED:
                busEvent = (FileDeletedEvent) event;
                break;
            case EventNames.FILE_ADDED:
                busEvent = (FileAddedEvent) event;
                break;
            case EventNames.QUALITY_UPDATED:
                busEvent = (QualityUpdated) event;
                break;
            case EventNames.WHISPER_INVITATION_RECEIVED:
                busEvent = (WhisperInvitationReceived) event;
                break;
            case EventNames.WHISPER_INVITATION_ACCEPTED:
                busEvent = (WhisperInvitationAccepted) event;
                break;
            case EventNames.WHISPER_INVITATION_DECLINED:
                busEvent = (WhisperInvitationDeclined) event;
                break;
            case EventNames.WHISPER_LEFT:
                busEvent = (WhisperLeft) event;
                break;
            case EventNames.OWN_USER_INVITED:
                busEvent = (OwnUserInvited) event;
                break;
            case EventNames.USER_INVITED:
                busEvent = (UserInvited) event;
                break;
            case EventNames.PARTICIPANT_SWITCH:
                busEvent = (ConferenceUserSwitch) event;
                break;
            case EventNames.PARTICIPANT_ADDED:
                busEvent = (ParticipantAdded) event;
                break;
            case EventNames.OWN_USER_SWITCH:
                busEvent = (OwnConferenceUserSwitch) event;
                break;
            case EventNames.CONFERENCE_ENDED:
                busEvent = (ConferenceEnded) event;
                break;
            case EventNames.CONFERENCE_UPDATED:
                busEvent = (ConferenceUpdated) event;
                break;
            case EventNames.FILE_CONVERTED:
                busEvent = (FileConverted) event;
                break;
            case EventNames.FILE_PRESENTATION_STARTED:
                busEvent = (FilePresentationStarted) event;
                break;
            case EventNames.FILE_PRESENTATION_STOPPED:
                busEvent = (FilePresentationStopped) event;
                break;
            case EventNames.FILE_PRESENTATION_UPDATED:
                busEvent = (FilePresentationUpdated) event;
                break;
            case EventNames.VIDEO_PRESENTATION_STARTED:
                busEvent = (VideoPresentationStarted) event;
                break;
            case EventNames.VIDEO_PRESENTATION_STOPPED:
                busEvent = (VideoPresentationStopped) event;
                break;
            case EventNames.VIDEO_PRESENTATION_PLAY:
                busEvent = (VideoPresentationPlay) event;
                break;
            case EventNames.VIDEO_PRESENTATION_PAUSED:
                busEvent = (VideoPresentationPaused) event;
                break;
            case EventNames.VIDEO_PRESENTATION_SEEK:
                busEvent = (VideoPresentationSeek) event;
                break;
            case EventNames.CONTACT_ADDED:
                busEvent = (ContactAdded) event;
                break;
            case EventNames.PEER_CONNECTION_UPDATED:
                busEvent = (PeerConnectionStatusUpdated) event;
                break;
            case EventNames.CONFERENCE_STATS:
                busEvent = new Gson().fromJson(message, ConferenceStats.class);
                break;
            default:
                throw new IllegalStateException("Unknown type of event");
        }
        return busEvent;
    }

    /**
     * Dispatch conference related events
     */
    private static InterfaceEvent handleConferenceMessage(ConferenceMessageReceived event) {
        Event event1 = EventsFactory.decode(event.message);
        if (event1 == null)
            return new MessageReceived(event.userId, event.conferenceId, event.message);
        else {
            switch (event1.getType()) {
                case EventNames.VIDEO_PRESENTATION_STARTED:
                    return (VideoPresentationStarted) event1;
                case EventNames.VIDEO_PRESENTATION_STOPPED:
                    return (VideoPresentationStopped) event1;
                case EventNames.VIDEO_PRESENTATION_PLAY:
                    return (VideoPresentationPlay) event1;
                case EventNames.VIDEO_PRESENTATION_PAUSED:
                    return (VideoPresentationPaused) event1;
                case EventNames.VIDEO_PRESENTATION_SEEK:
                    return (VideoPresentationSeek) event1;
                case EventNames.PLAY_EMO_SPHERE:
                    return (PlayEmoSphere) event1;
                default:
                    return new MessageReceived(event.userId, event.conferenceId, event.message);
            }
        }
    }

    /**
     * Dispatch a conference broadcast based event.
     */
    private static InterfaceEvent handleBroadcast(BroadcastEvent event) {
        Event event1 = EventsFactory.decode(event.message);
        switch (event1.getType()) {
            case EventNames.TYPING_DETECTION:
                return (TypingDetection) event1;
            default:
                throw new IllegalStateException("Unknown broadcast type event");
        }
    }
}
