package com.voxeet.sdk.core.services;

import android.content.Context;
import android.os.Build;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.android.media.MediaEngine;
import com.voxeet.android.media.MediaEngineException;
import com.voxeet.sdk.core.services.media.VideoSinkHolder;
import com.voxeet.sdk.media.MediaSDK;
import com.voxeet.android.media.MediaStream;
import com.voxeet.sdk.media.camera.CameraEnumeratorLollipopWrapper;
import com.voxeet.sdk.media.camera.CameraEnumeratorPreLollipopWrapper;
import com.voxeet.sdk.media.camera.CameraInformationProvider;
import com.voxeet.android.media.crypto.AbstractNativeMediaCryptoCallback;
import com.voxeet.android.media.stats.LocalStats;
import com.voxeet.sdk.core.AbstractVoxeetService;
import com.voxeet.sdk.core.VoxeetSdkTemplate;
import com.voxeet.sdk.core.services.holder.ServiceProviderHolder;
import com.voxeet.sdk.events.error.CameraSwitchErrorEvent;
import com.voxeet.sdk.events.success.CameraSwitchSuccessEvent;
import com.voxeet.sdk.utils.Validate;
import com.voxeet.sdk.utils.annotate;

import org.webrtc.CameraVideoCapturer;
import org.webrtc.EglBaseMethods;
import org.webrtc.VideoSink;

import java.util.concurrent.CopyOnWriteArrayList;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;

@annotate
public class MediaService extends AbstractVoxeetService {

    private CameraInformationProvider enumerator;
    private AbstractNativeMediaCryptoCallback mediaCryptoCallback;
    private MediaSDK media;
    private boolean isAudio3DEnabled;

    private CopyOnWriteArrayList<VideoSinkHolder> videoSinkHolders;

    public MediaService(@NonNull VoxeetSdkTemplate instance) {
        super(instance, ServiceProviderHolder.DEFAULT);

        videoSinkHolders = new CopyOnWriteArrayList<>();

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            enumerator = new CameraEnumeratorLollipopWrapper(instance.getApplicationContext());
        } else {
            enumerator = new CameraEnumeratorPreLollipopWrapper(instance.getApplicationContext());
        }

        enumerator.setDefaultCameraFront(true);

        isAudio3DEnabled = false;
    }

    public boolean hasMedia() {
        return null != media;
    }

    public MediaSDK getMedia() {
        return media;
    }

    /**
     * @param peer
     * @return
     * @deprecated use {@link LocalStatsService#getLocalStats(String)}
     */
    @Deprecated
    @Nullable
    public LocalStats getLocalStats(@NonNull String peer) {
        if (null == peer) return null; //may happen in wrong implementation
        MediaSDK media = getMedia();
        if (null != media) return media.getLocalStats(peer);
        return null;
    }

    public void releaseMedia() {
        for (VideoSinkHolder holder : videoSinkHolders) {
            MediaStream stream = holder.getMediaStream();
            if(holder.hasVideoSink()) {
                media.unattachMediaStream(holder, stream);
            }
        }
        videoSinkHolders.clear();

        if (null != media) media.stop();

        media = null;
    }

    public void createMedia(Context context, String userId, MediaEngine.StreamListener mediaStreamListener,
                            CameraVideoCapturer.CameraEventsHandler cameraEventsHandler,
                            boolean videoOn, boolean useMic) throws MediaEngineException {
        media = new MediaSDK(context, userId, mediaStreamListener, cameraEventsHandler, videoOn, useMic, mediaCryptoCallback);
        media.setAudio3D(isAudio3DEnabled);
    }

    public EglBaseMethods.Context getEglContext() {
        if (null != media) {
            return media.getEglBase().getEglBaseContext();
        } else {
            return null;
        }
    }

    /**
     * Set a MediaCryptoCallback to be used "before" joining the conference
     * <p>
     * Note : it's not released automatically, it must be un/-set before any conference
     *
     * @param callback the callback to use, null to unset the callback
     */
    public void setCryptoCallback(@Nullable AbstractNativeMediaCryptoCallback callback) {
        this.mediaCryptoCallback = callback;
    }

    public boolean attachMediaStream(@NonNull MediaStream stream, @NonNull VideoSink videoSink) {
        MediaEngine media = this.media;
        if(null == media) return false;

        VideoSinkHolder holder = getOrCreate(videoSink);

        if(!holder.isAttachedTo(stream)) {
            holder.setAttached(stream);
            media.attachMediaStream(holder, stream);
        }

        return true;
    }

    public boolean unAttachMediaStream(@NonNull MediaStream stream, @NonNull VideoSink videoSink) {
        MediaEngine media = this.media;
        if(null == media) return false;

        VideoSinkHolder holder = getOrCreate(videoSink);

        if(holder.isAttachedTo(stream)) {
            holder.unattach();
            media.unattachMediaStream(holder, stream);
        }
        return true;
    }

    public boolean isAudio3DEnabled() {
        return isAudio3DEnabled;
    }

    /**
     * Set the 3D Audio management to on/off. If no conferences or the media was not initialized, the state will retain for the next calls.
     *
     * The SDK keeps the value, it must be recalled with a new value to be forced-updated
     *
     * @param enable the new state for 3d audio processing
     * @return true if everything is fine, false if the media was not initialized
     */
    public boolean setAudio3DEnabled(boolean enable) {
        isAudio3DEnabled = enable;
        if(null != media) return media.setAudio3D(enable);
        return false;
    }

    @NonNull
    public Promise<Boolean> switchCamera() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                Validate.notNull(getMedia(), "media");

                getMedia().switchCamera(new CameraVideoCapturer.CameraSwitchHandler() {
                    @Override
                    public void onCameraSwitchDone(boolean isFrontCamera) {
                        getCameraInformationProvider().setDefaultCameraFront(isFrontCamera);

                        getEventBus().post(new CameraSwitchSuccessEvent(isFrontCamera));
                        solver.resolve(true);
                    }

                    @Override
                    public void onCameraSwitchError(String errorDescription) {
                        getEventBus().post(new CameraSwitchErrorEvent(errorDescription));
                        solver.resolve(false);
                    }
                });
            }
        });
    }

    @NonNull
    public CameraInformationProvider getCameraInformationProvider() {
        return enumerator;
    }


    private VideoSinkHolder getOrCreate(@NonNull VideoSink videoSink) {
        VideoSinkHolder holder = new VideoSinkHolder(videoSink);
        int found = videoSinkHolders.indexOf(holder);
        if(0 > found) {
            videoSinkHolders.add(holder);
            return holder;
        }
        return videoSinkHolders.get(found);
    }

}
