package com.voxeet.sdk.core.abs.promises;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.android.media.MediaEngineException;
import com.voxeet.sdk.core.abs.ConferenceSdkObservableProvider;
import com.voxeet.sdk.exceptions.ExceptionManager;
import com.voxeet.sdk.media.camera.CameraInformationProvider;
import com.voxeet.sdk.core.abs.AbstractPromiseable;
import com.voxeet.sdk.core.abs.ConferenceService;
import com.voxeet.sdk.core.abs.information.ConferenceInformation;
import com.voxeet.sdk.core.abs.information.ConferenceUserType;
import com.voxeet.sdk.utils.HttpHelper;
import com.voxeet.sdk.core.preferences.VoxeetPreferences;
import com.voxeet.sdk.core.services.MediaService;
import com.voxeet.sdk.events.error.HttpException;
import com.voxeet.sdk.events.error.PermissionRefusedEvent;
import com.voxeet.sdk.events.promises.PromisePermissionRefusedEventException;
import com.voxeet.sdk.events.success.StartVideoAnswerEvent;
import com.voxeet.sdk.json.StartVideoResponse;
import com.voxeet.sdk.utils.Validate;

import org.greenrobot.eventbus.EventBus;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.ErrorPromise;
import eu.codlab.simplepromise.solve.PromiseExec;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;
import retrofit2.Call;
import retrofit2.Response;

public class StartVideoPromise extends AbstractPromiseable<Boolean> {
    private final static String TAG = StartVideoPromise.class.getSimpleName();
    private final boolean isDefaultFrontFacing;

    public StartVideoPromise(@NonNull ConferenceService parent,
                             @NonNull MediaService mediaService,
                             @NonNull ConferenceSdkObservableProvider provider,
                             @Nullable ConferenceInformation information,
                             @NonNull EventBus eventBus,
                             boolean isDefaultFrontFacing) {
        super(parent, mediaService, provider, information, eventBus);

        this.isDefaultFrontFacing = isDefaultFrontFacing;
    }

    @NonNull
    @Override
    public Promise<Boolean> createPromise() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                final ConferenceInformation information = getInformation();
                try {
                    CameraInformationProvider provider = getMediaService().getCameraInformationProvider();
                    provider.setDefaultCameraFront(isDefaultFrontFacing);

                    log("starting video");
                    if (!Validate.hasCameraPermissions(getParent().getContext())) {
                        PermissionRefusedEvent event = new PermissionRefusedEvent(PermissionRefusedEvent.Permission.CAMERA);
                        getEventBus().post(event);
                        throw new PromisePermissionRefusedEventException(event);
                    }

                    if (!getMediaService().hasMedia()) {
                        Log.d(TAG, "startVideo: media is null");
                        throw new MediaEngineException("Media is null, invalid state");
                    }

                    if (null == information) {
                        Log.d(TAG, "startVideo: information is null");
                        throw new MediaEngineException("ConferenceInformation is null, invalid state");
                    }

                    if (information.isOwnVideoStarted()) {
                        Log.d(TAG, "startVideo: already started... please wait if you wanted to stop");
                        solver.resolve(true);//new StartVideoAnswerEvent(false, true));
                        return;
                    }

                    if(ConferenceUserType.LISTENER.equals(information.getConferenceUserType())) {
                        Log.d(TAG, "startVideo: you are a listener, you can't start the video");
                        solver.resolve(false);//new StartVideoAnswerEvent(false, true));
                        return;
                    }

                    String defaultCameraName = isDefaultFrontFacing ? provider.getFrontCameraName() : provider.getBackCameraName();

                    if (null == defaultCameraName) {
                        throw new MediaEngineException("DefaultCamera is null, invalid state");
                    }

                    information.setOwnVideoStarted(true);

                    getMediaService().getMedia().startVideo(defaultCameraName);

                    final Call<StartVideoResponse> startVideo = getProvider().getStartVideoObservable(getInformation().getConference().getConferenceId(), VoxeetPreferences.id());
                    HttpHelper.enqueue(startVideo, new HttpHelper.HttpCallback<StartVideoResponse>() {
                        @Override
                        public void onSuccess(@NonNull StartVideoResponse object, @NonNull Response<StartVideoResponse> response) {
                            //remove start video since it is not to be used
                            //media.startVideo(mDefaultCamera);

                            getInformation().setOwnVideoStarted(true);

                            final StartVideoAnswerEvent event = new StartVideoAnswerEvent(true);
                            getEventBus().post(event);
                            createVideoAnswer(object.getUserId(), object.getDescription(), object.getCandidates())
                                    .then(new PromiseExec<Boolean, Object>() {
                                        @Override
                                        public void onCall(@Nullable Boolean aBoolean, @NonNull Solver<Object> internal_solver) {
                                            solver.resolve(true);
                                        }
                                    })
                                    .error(new ErrorPromise() {
                                        @Override
                                        public void onError(Throwable error) {
                                            ExceptionManager.sendException(error);
                                            error.printStackTrace();
                                            solver.resolve(true);
                                        }
                                    });
                        }

                        @Override
                        public void onFailure(@NonNull Throwable e, @Nullable Response<StartVideoResponse> response) {
                            HttpException.dumpErrorResponse(response);

                            e.printStackTrace();
                            information.setOwnVideoStarted(false);
                            StartVideoAnswerEvent event = new StartVideoAnswerEvent(false);
                            getEventBus().post(event);
                            solver.reject(e);
                        }
                    });
                    //explicitly keep the list of the different exception
                    //this methods throws
                } catch (PromisePermissionRefusedEventException exception) {
                    solver.reject(exception);
                } catch (MediaEngineException exception) {
                    solver.reject(exception);
                } catch (Exception exception) {
                    solver.reject(exception);
                }
            }
        });
    }
}
