package com.voxeet.sdk.core;

import android.util.Log;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.voxeet.sdk.core.network.websocket.VoxeetWebSocket;
import com.voxeet.sdk.events.BaseEvent;
import com.voxeet.sdk.events.error.SocketErrorEvent;
import com.voxeet.sdk.events.success.BadgeUpdate;
import com.voxeet.sdk.events.success.ConferenceDestroyedPushEvent;
import com.voxeet.sdk.events.success.ConferenceEndedEvent;
import com.voxeet.sdk.events.success.ConferenceStatsEvent;
import com.voxeet.sdk.events.success.ConferenceUpdated;
import com.voxeet.sdk.events.success.ConferenceUpdatedEvent;
import com.voxeet.sdk.events.success.ConferenceUserAddedEvent;
import com.voxeet.sdk.events.success.ConferenceUserSwitchEvent;
import com.voxeet.sdk.events.success.ContactAddedEvent;
import com.voxeet.sdk.events.success.FileConvertedEvent;
import com.voxeet.sdk.events.success.FilePresentationChangeEvent;
import com.voxeet.sdk.events.success.FilePresentationStartEvent;
import com.voxeet.sdk.events.success.FilePresentationStartedEvent;
import com.voxeet.sdk.events.success.FilePresentationStopEvent;
import com.voxeet.sdk.events.success.FilePresentationStoppedEvent;
import com.voxeet.sdk.events.success.FilePresentationUpdatedEvent;
import com.voxeet.sdk.events.success.InvitationReceived;
import com.voxeet.sdk.events.success.MeetingActivityAdded;
import com.voxeet.sdk.events.success.MeetingActivityDeleted;
import com.voxeet.sdk.events.success.MeetingActivityUpdated;
import com.voxeet.sdk.events.success.MeetingAddedOrUpdated;
import com.voxeet.sdk.events.success.MeetingDeleted;
import com.voxeet.sdk.events.success.MeetingEventAddedOrUpdated;
import com.voxeet.sdk.events.success.MeetingReadTimeStampUpdated;
import com.voxeet.sdk.events.success.MessageReceived;
import com.voxeet.sdk.events.success.OfferCreatedEvent;
import com.voxeet.sdk.events.success.OwnConferenceStartedEvent;
import com.voxeet.sdk.events.success.OwnConferenceUserSwitchEvent;
import com.voxeet.sdk.events.success.OwnContactRemoved;
import com.voxeet.sdk.events.success.OwnExternalInvitationSent;
import com.voxeet.sdk.events.success.OwnProfileUpdatedEvent;
import com.voxeet.sdk.events.success.OwnUserInvitedEvent;
import com.voxeet.sdk.events.success.ParticipantUpdatedEvent;
import com.voxeet.sdk.events.success.PeerConnectionStatusUpdatedEvent;
import com.voxeet.sdk.events.success.PlayEmoSphereEvent;
import com.voxeet.sdk.events.success.ProfileUpdated;
import com.voxeet.sdk.events.success.QualityUpdatedEvent;
import com.voxeet.sdk.events.success.RecordingStatusUpdate;
import com.voxeet.sdk.events.success.RenegociationEndedEvent;
import com.voxeet.sdk.events.success.RenegociationUpdate;
import com.voxeet.sdk.events.success.SocketConnectEvent;
import com.voxeet.sdk.events.success.SocketDisconnectEvent;
import com.voxeet.sdk.events.success.SocketStateChangeEvent;
import com.voxeet.sdk.events.success.TypingDetectionEvent;
import com.voxeet.sdk.events.success.UserInvitedEvent;
import com.voxeet.sdk.events.success.VideoPresentationPausedEvent;
import com.voxeet.sdk.events.success.VideoPresentationPlayEvent;
import com.voxeet.sdk.events.success.VideoPresentationSeekEvent;
import com.voxeet.sdk.events.success.VideoPresentationStartedEvent;
import com.voxeet.sdk.events.success.VideoPresentationStoppedEvent;
import com.voxeet.sdk.events.success.WhisperInviteAcceptedEvent;
import com.voxeet.sdk.events.success.WhisperInviteDeclinedEvent;
import com.voxeet.sdk.events.success.WhisperInviteReceivedEvent;
import com.voxeet.sdk.events.success.WhisperLeftEvent;
import com.voxeet.sdk.factories.EventsFactory;
import com.voxeet.sdk.json.BadgeUpdatedEvent;
import com.voxeet.sdk.json.BroadcastEvent;
import com.voxeet.sdk.json.ConferenceDestroyedPush;
import com.voxeet.sdk.json.ConferenceEnded;
import com.voxeet.sdk.json.ConferenceMessageReceived;
import com.voxeet.sdk.json.ConferenceStats;
import com.voxeet.sdk.json.ConferenceUserAdded;
import com.voxeet.sdk.json.ConferenceUserSwitch;
import com.voxeet.sdk.json.ContactAdded;
import com.voxeet.sdk.json.Event;
import com.voxeet.sdk.json.EventNames;
import com.voxeet.sdk.json.FileAdded;
import com.voxeet.sdk.json.FileAddedEvent;
import com.voxeet.sdk.json.FileConverted;
import com.voxeet.sdk.json.FileDeleted;
import com.voxeet.sdk.json.FileDeletedEvent;
import com.voxeet.sdk.json.FilePresentationChange;
import com.voxeet.sdk.json.FilePresentationStart;
import com.voxeet.sdk.json.FilePresentationStarted;
import com.voxeet.sdk.json.FilePresentationStop;
import com.voxeet.sdk.json.FilePresentationStopped;
import com.voxeet.sdk.json.FilePresentationUpdated;
import com.voxeet.sdk.json.InvitationReceivedEvent;
import com.voxeet.sdk.json.MeetActivityAddedEvent;
import com.voxeet.sdk.json.MeetActivityDeletedEvent;
import com.voxeet.sdk.json.MeetActivityUpdatedEvent;
import com.voxeet.sdk.json.MeetingAddedOrUpdatedEvent;
import com.voxeet.sdk.json.MeetingDeletedEvent;
import com.voxeet.sdk.json.MeetingEventAddedOrUpdatedEvent;
import com.voxeet.sdk.json.MeetingReadTimeStampUpdatedEvent;
import com.voxeet.sdk.json.OfferCreated;
import com.voxeet.sdk.json.OwnConferenceCreated;
import com.voxeet.sdk.json.OwnConferenceUserSwitch;
import com.voxeet.sdk.json.OwnContactRemovedEvent;
import com.voxeet.sdk.json.OwnExternalInvitationSentEvent;
import com.voxeet.sdk.json.OwnProfileUpdated;
import com.voxeet.sdk.json.OwnUserInvited;
import com.voxeet.sdk.json.ParticipantUpdated;
import com.voxeet.sdk.json.PeerConnectionStatusUpdated;
import com.voxeet.sdk.json.PlayEmoSphere;
import com.voxeet.sdk.json.ProfileUpdatedEvent;
import com.voxeet.sdk.json.QualityUpdated;
import com.voxeet.sdk.json.RecordingStatusUpdateEvent;
import com.voxeet.sdk.json.TypingDetection;
import com.voxeet.sdk.json.UserInvited;
import com.voxeet.sdk.json.VideoPresentationPaused;
import com.voxeet.sdk.json.VideoPresentationPlay;
import com.voxeet.sdk.json.VideoPresentationSeek;
import com.voxeet.sdk.json.VideoPresentationStarted;
import com.voxeet.sdk.json.VideoPresentationStopped;
import com.voxeet.sdk.json.WhisperInvitationAccepted;
import com.voxeet.sdk.json.WhisperInvitationDeclined;
import com.voxeet.sdk.json.WhisperInvitationReceived;
import com.voxeet.sdk.json.WhisperLeft;

import org.greenrobot.eventbus.EventBus;

public class VoxeetDispatcher {

    private static final String TAG = VoxeetDispatcher.class.getSimpleName();

    private static EventBus eventBus = EventBus.getDefault();

    /**
     * Map and dispatch events through the eventbus.
     *
     * @param eventName the event name
     * @param message   the message
     */
    public static void dispatch(String eventName, String message) {
        Log.d(TAG, "dispatch: " + eventName + " := " + message);
        try {
            BaseEvent event;
            switch (eventName) {
                case VoxeetWebSocket.SOCKET_DISCONNECTED:
                    event = handleDisconnect(message);
                    break;
                case VoxeetWebSocket.SOCKET_ERROR:
                    event = handleError(message);
                    break;
                case VoxeetWebSocket.SOCKET_MESSAGE:
                    event = handleMessage(message);
                    break;
                case VoxeetWebSocket.SOCKET_CONNECTED:
                    event = handleSocketConnect(message);
                    break;
                case VoxeetWebSocket.SOCKET_STATE_CHANGE:
                    event = handleStateChange(message);
                    break;
                default:
                    throw new IllegalStateException("Unknown event type");
            }

            eventBus.post(event);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * Dispatch a socket error event.
     */
    private static BaseEvent handleError(String message) {
        return new SocketErrorEvent(message);
    }

    /**
     * Dispatch a socket disconnected event.
     */
    private static BaseEvent handleDisconnect(String message) {
        return new SocketDisconnectEvent(message);
    }

    /**
     * Dispatch a socket state change event.
     */
    private static BaseEvent handleStateChange(String message) {
        return new SocketStateChangeEvent(message);
    }

    /**
     * Dispatch a socket connected event.
     */
    private static BaseEvent handleSocketConnect(String message) {
        return new SocketConnectEvent(message);
    }

    /**
     * Dispatch all other events.
     */
    private static BaseEvent handleMessage(String message) {
        BaseEvent busEvent;

        Event event = EventsFactory.decode(message);
        switch (event.getType()) {
            case EventNames.OFFER_CREATED:
                busEvent = new OfferCreatedEvent(message, (OfferCreated) event);
                break;
            case EventNames.CONFERENCE_MESSAGE_RECEIVED:
                busEvent = handleConferenceMessage((ConferenceMessageReceived) event);
                break;
            case EventNames.CONFERENCE_DESTROYED:
                busEvent = new ConferenceDestroyedPushEvent(message, (ConferenceDestroyedPush) event);
                break;
            case EventNames.CONTACT_PROFILE_UPDATED:
                busEvent = new ProfileUpdated(message, (ProfileUpdatedEvent) event);
                break;
            case EventNames.PARTICIPANT_UPDATED:
                ParticipantUpdated participantUpdated = (ParticipantUpdated) event;
                busEvent = new ParticipantUpdatedEvent(message, participantUpdated);
                break;
            case EventNames.OWN_CONFERENCE_CREATED:
                busEvent = new OwnConferenceStartedEvent(message, (OwnConferenceCreated) event);
                break;
            case EventNames.OWN_PROFILE_UPDATED:
                busEvent = new OwnProfileUpdatedEvent(message, (OwnProfileUpdated) event);
                break;
            case EventNames.OWN_EXTERNAL_INVITATION:
                busEvent = new OwnExternalInvitationSent(message, (OwnExternalInvitationSentEvent) event);
                break;
            case EventNames.OWN_CONTACT_REMOVED:
                busEvent = new OwnContactRemoved(message, (OwnContactRemovedEvent) event);
                break;
            case EventNames.MEETING_ADDED_OR_UPDATED:
                busEvent = new MeetingAddedOrUpdated(message, (MeetingAddedOrUpdatedEvent) event);
                break;
            case EventNames.MEETING_DELETED:
                busEvent = new MeetingDeleted(message, (MeetingDeletedEvent) event);
                break;
            case EventNames.BADGE_UPDATE_EVENT:
                busEvent = new BadgeUpdate(message, (BadgeUpdatedEvent) event);
                break;
            case EventNames.RENEGOCIATION_ENDED:
                busEvent = new RenegociationUpdate(message, (RenegociationEndedEvent) event);
                break;
            case EventNames.BROADCAST_EVENT:
                busEvent = handleBroadcast((BroadcastEvent) event);
                break;
            case EventNames.RECORDING_STATUS_UPDATE:
                busEvent = new RecordingStatusUpdate(message, (RecordingStatusUpdateEvent) event);
                break;
            case EventNames.MEET_ACTIVITY_DELETED:
                busEvent = new MeetingActivityDeleted(message, (MeetActivityDeletedEvent) event);
                break;
            case EventNames.MEET_ACTIVITY_UPDATED:
                busEvent = new MeetingActivityUpdated(message, (MeetActivityUpdatedEvent) event);
                break;
            case EventNames.MEET_ACTIVITY_ADDED:
                busEvent = new MeetingActivityAdded(message, (MeetActivityAddedEvent) event);
                break;
            case EventNames.MEET_READ_TIMESTAMP_UPDATED:
                busEvent = new MeetingReadTimeStampUpdated(message, (MeetingReadTimeStampUpdatedEvent) event);
                break;
            case EventNames.INVITATION_RECEIVED:
                busEvent = new InvitationReceived(message, (InvitationReceivedEvent) event);
                break;
            case EventNames.MEETING_EVENT_ADDED_OR_UPDATED:
                busEvent = new MeetingEventAddedOrUpdated(message, (MeetingEventAddedOrUpdatedEvent) event);
                break;
            case EventNames.FILE_DELETED:
                busEvent = new FileDeleted(message, (FileDeletedEvent) event);
                break;
            case EventNames.FILE_ADDED:
                busEvent = new FileAdded(message, (FileAddedEvent) event);
                break;
            case EventNames.QUALITY_UPDATED:
                busEvent = new QualityUpdatedEvent(message, (QualityUpdated) event);
                break;
            case EventNames.WHISPER_INVITATION_RECEIVED:
                busEvent = new WhisperInviteReceivedEvent(message, (WhisperInvitationReceived) event);
                break;
            case EventNames.WHISPER_INVITATION_ACCEPTED:
                busEvent = new WhisperInviteAcceptedEvent(message, (WhisperInvitationAccepted) event);
                break;
            case EventNames.WHISPER_INVITATION_DECLINED:
                busEvent = new WhisperInviteDeclinedEvent(message, (WhisperInvitationDeclined) event);
                break;
            case EventNames.WHISPER_LEFT:
                busEvent = new WhisperLeftEvent(message, (WhisperLeft) event);
                break;
            case EventNames.OWN_USER_INVITED:
                busEvent = new OwnUserInvitedEvent(message, (OwnUserInvited) event);
                break;
            case EventNames.USER_INVITED:
                busEvent = new UserInvitedEvent(message, (UserInvited) event);
                break;
            case EventNames.PARTICIPANT_SWITCH:
                busEvent = new ConferenceUserSwitchEvent(message, (ConferenceUserSwitch) event);
                break;
            case EventNames.PARTICIPANT_ADDED:
                busEvent = new ConferenceUserAddedEvent(message, (ConferenceUserAdded) event);
                break;
            case EventNames.OWN_USER_SWITCH:
                busEvent = new OwnConferenceUserSwitchEvent(message, (OwnConferenceUserSwitch) event);
                break;
            case EventNames.CONFERENCE_ENDED:
                busEvent = new ConferenceEndedEvent(message, (ConferenceEnded) event);
                break;
            case EventNames.CONFERENCE_UPDATED:
                busEvent = new ConferenceUpdatedEvent(message, (ConferenceUpdated) event);
                break;
            case EventNames.FILE_CONVERTED:
                busEvent = new FileConvertedEvent(message, (FileConverted) event);
                break;
            case EventNames.FILE_PRESENTATION_STARTED:
                busEvent = new FilePresentationStartedEvent(message, (FilePresentationStarted) event);
                break;
            case EventNames.FILE_PRESENTATION_STOPPED:
                busEvent = new FilePresentationStoppedEvent(message, (FilePresentationStopped) event);
                break;
            case EventNames.FILE_PRESENTATION_UPDATED:
                busEvent = new FilePresentationUpdatedEvent(message, (FilePresentationUpdated) event);
                break;
            case EventNames.VIDEO_PRESENTATION_STARTED:
                busEvent = new VideoPresentationStartedEvent(message, (VideoPresentationStarted) event);
                break;
            case EventNames.VIDEO_PRESENTATION_STOPPED:
                busEvent = new VideoPresentationStoppedEvent(message, (VideoPresentationStopped) event);
                break;
            case EventNames.VIDEO_PRESENTATION_PLAY:
                busEvent = new VideoPresentationPlayEvent(message, (VideoPresentationPlay) event);
                break;
            case EventNames.VIDEO_PRESENTATION_PAUSED:
                busEvent = new VideoPresentationPausedEvent(message, (VideoPresentationPaused) event);
                break;
            case EventNames.VIDEO_PRESENTATION_SEEK:
                busEvent = new VideoPresentationSeekEvent(message, (VideoPresentationSeek) event);
                break;
            case EventNames.CONTACT_ADDED:
                busEvent = new ContactAddedEvent(message, (ContactAdded) event);
                break;
            case EventNames.PEER_CONNECTION_UPDATED:
                busEvent = new PeerConnectionStatusUpdatedEvent(message, (PeerConnectionStatusUpdated) event);
                break;
            case EventNames.CONFERENCE_STATS:
                ConferenceStats stats = new Gson().fromJson(message, ConferenceStats.class);
                busEvent = new ConferenceStatsEvent(message, stats);
                break;
            default:
                throw new IllegalStateException("Unknown type of event");
        }
        return busEvent;
    }

    /**
     * Dispatch conference related events
     */
    private static BaseEvent handleConferenceMessage(ConferenceMessageReceived event) {
        Event event1 = EventsFactory.decode(event.getMessage());
        if (event1 == null)
            return new MessageReceived(event.getUserId(), event.getConferenceId(), event.getMessage());
        else {
            switch (event1.getType()) {
                case EventNames.FILE_PRESENTATION_CHANGE:
                    return new FilePresentationChangeEvent(event.getMessage(), event.getUserId(), event.getConferenceId(), (FilePresentationChange) event1);
                case EventNames.FILE_PRESENTATION_START:
                    return new FilePresentationStartEvent(event.getMessage(), event.getUserId(), event.getConferenceId(), (FilePresentationStart) event1);
                case EventNames.FILE_PRESENTATION_STOP:
                    return new FilePresentationStopEvent(event.getMessage(), event.getUserId(), event.getConferenceId(), (FilePresentationStop) event1);
                case EventNames.VIDEO_PRESENTATION_STARTED:
                    return new VideoPresentationStartedEvent(event.getMessage(), (VideoPresentationStarted) event1);
                case EventNames.VIDEO_PRESENTATION_STOPPED:
                    return new VideoPresentationStoppedEvent(event.getMessage(), (VideoPresentationStopped) event1);
                case EventNames.VIDEO_PRESENTATION_PLAY:
                    return new VideoPresentationPlayEvent(event.getMessage(), (VideoPresentationPlay) event1);
                case EventNames.VIDEO_PRESENTATION_PAUSED:
                    return new VideoPresentationPausedEvent(event.getMessage(), (VideoPresentationPaused) event1);
                case EventNames.VIDEO_PRESENTATION_SEEK:
                    return new VideoPresentationSeekEvent(event.getMessage(), (VideoPresentationSeek) event1);
                case EventNames.PLAY_EMO_SPHERE:
                    return new PlayEmoSphereEvent(event.getMessage(), event.getUserId(), event.getConferenceId(), (PlayEmoSphere) event1);
                default:
                    return new MessageReceived(event.getUserId(), event.getConferenceId(), event.getMessage());
            }
        }
    }

    /**
     * Dispatch a conference broadcast based event.
     */
    private static BaseEvent handleBroadcast(BroadcastEvent event) {
        Event event1 = EventsFactory.decode(event.getMessage());
        switch (event1.getType()) {
            case EventNames.TYPING_DETECTION:
                return new TypingDetectionEvent(event.getMessage(), (TypingDetection) event1, event.getUserId(), event.getMeetingId());
            default:
                throw new IllegalStateException("Unknown broadcast type event");
        }
    }
}
