package com.voxeet.sdk.utils;

import android.util.Log;

import com.voxeet.android.media.MediaStream;
import com.voxeet.sdk.core.VoxeetSdkTemplate;
import com.voxeet.sdk.events.success.ConferenceDestroyedPushEvent;
import com.voxeet.sdk.events.success.ConferenceUserJoinedEvent;
import com.voxeet.sdk.events.success.ConferenceUserLeftEvent;
import com.voxeet.sdk.events.success.ConferenceUserUpdatedEvent;
import com.voxeet.sdk.events.success.ParticipantUpdatedEvent;
import com.voxeet.sdk.events.success.ScreenStreamAddedEvent;
import com.voxeet.sdk.events.success.ScreenStreamRemovedEvent;
import com.voxeet.sdk.json.ConferenceDestroyedPush;
import com.voxeet.sdk.models.abs.ConferenceUser;

import org.greenrobot.eventbus.EventBus;
import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;

public abstract class EventCallback {
    private static final String TAG = EventCallback.class.getSimpleName();

    public void register(VoxeetSdkTemplate template) {
        EventBus eventBus = template.getEventBus();

        if (!eventBus.isRegistered(this)) {
            eventBus.register(this);
        }
    }

    public void unregister(VoxeetSdkTemplate template) {
        EventBus eventBus = template.getEventBus();

        if (eventBus.isRegistered(this)) {
            eventBus.unregister(this);
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceDestroyedPushEvent event) {
        onConferenceDestroyed(event.getPush().getConferenceId());
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceDestroyedPush event) {
        onConferenceDestroyed(event.getConferenceId());
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ParticipantUpdatedEvent event) {
        Log.d(TAG, "onEvent: ParticipantUpdatedEvent " + event + " " + event.getUserId() + " " + event.getStatus() + " " + event.getConfId());
        onParticipantUpdated(event.getConfId(), event.getUserId(), event.getStatus());
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceUserLeftEvent userLeft) {
        onParticipantLeft(userLeft.getUser());
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceUserJoinedEvent userJoined) {
        onUserJoined(userJoined.getUser(), userJoined.getMediaStream());
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceUserUpdatedEvent userUpdated) {
        onUserUpdated(userUpdated.getUser(), userUpdated.getMediaStream());
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ScreenStreamAddedEvent streamAdded) {
        onScreenAdded(streamAdded.getPeer(), streamAdded.getMediaStream());
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ScreenStreamRemovedEvent streamRemoved) {
        onScreenRemoved(streamRemoved.getPeer());
    }

    protected abstract void onConferenceDestroyed(String conferenceId);

    protected abstract void onParticipantUpdated(String confId, String userId, String status);

    protected abstract void onParticipantLeft(ConferenceUser user);

    protected abstract void onUserJoined(ConferenceUser user, MediaStream mediaStream);

    protected abstract void onUserUpdated(ConferenceUser user, MediaStream mediaStream);

    protected abstract void onScreenAdded(String peer, MediaStream mediaStream);

    protected abstract void onScreenRemoved(String peer);
}
