package com.voxeet.sdk.core.services;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.graphics.Point;
import android.media.projection.MediaProjectionManager;
import android.os.Build;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;
import android.view.WindowManager;

import com.voxeet.sdk.core.AbstractVoxeetService;
import com.voxeet.sdk.core.VoxeetSdkTemplate;
import com.voxeet.sdk.core.abs.ConferenceService;
import com.voxeet.sdk.core.services.holder.ServiceProviderHolder;
import com.voxeet.sdk.utils.annotate;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.ErrorPromise;
import eu.codlab.simplepromise.solve.PromiseExec;
import eu.codlab.simplepromise.solve.Solver;

@annotate
public class ScreenShareService extends AbstractVoxeetService {
    private static final int REQUEST_SCREEN_CAPTURE = 672;
    private static final String TAG = ScreenShareService.class.getSimpleName();
    private static final int DEFAULT_WIDTH = 640;
    private static final int DEFAULT_HEIGHT = 480;

    private VoxeetSdkTemplate mInstance;

    private int mLastResultCode;

    @Nullable
    private Point mLastPoint;

    @Nullable
    private Intent mLastData;


    public ScreenShareService(@NonNull VoxeetSdkTemplate instance) {
        super(instance, ServiceProviderHolder.DEFAULT);

        mInstance = instance;
    }

    public void toggleScreenShare() {
        mInstance.getConferenceService().toggleScreenShare();
    }

    public Promise<Boolean> startScreenShare(@NonNull Intent intent) {
        ConferenceService service = mInstance.getConferenceService();

        int width = DEFAULT_WIDTH;
        int height = DEFAULT_HEIGHT;

        if (null != mLastPoint) {
            width = mLastPoint.x;
            height = mLastPoint.y;
        }

        return service.startScreenShare(intent, width, height);
    }

    public Promise<Boolean> stopScreenShare() {
        return mInstance.getConferenceService().stopScreenShare();
    }

    public ScreenShareService setScreenSizeInformation(@NonNull Point point) {
        mLastPoint = point;

        return this;
    }

    public void sendRequestStartScreenShare() {
        getEventBus().post(new RequestScreenSharePermissionEvent());
    }

    public boolean sendUserPermissionRequest(@NonNull Activity activity) {
        if (Build.VERSION_CODES.LOLLIPOP <= Build.VERSION.SDK_INT) {
            MediaProjectionManager manager = (MediaProjectionManager)
                    activity.getSystemService(Context.MEDIA_PROJECTION_SERVICE);
            Intent request = manager.createScreenCaptureIntent();
            activity.startActivityForResult(request, ScreenShareService.REQUEST_SCREEN_CAPTURE);
            return true;
        }
        return false;
    }

    public void consumeRightsToScreenShare() {
        if (mLastResultCode == Activity.RESULT_OK && null != mLastData) {
            startScreenShare(mLastData)
                    .then(new PromiseExec<Boolean, Object>() {
                        @Override
                        public void onCall(@Nullable Boolean result, @NonNull Solver<Object> solver) {
                            Log.d(TAG, "onCall: starting screen share");
                        }
                    })
                    .error(new ErrorPromise() {
                        @Override
                        public void onError(@NonNull Throwable error) {
                            Log.d(TAG, "onError: error while starting screen share");
                            error.printStackTrace();
                        }
                    });
            mLastData = null;
        }
    }

    /**
     * Manage the Activity result in the context of the Screen Share Feature
     * <p>
     * returns true if the logic was managed
     * false, otherwise
     *
     * @param requestCode the requestCode obtained it should match
     * @param resultCode  the resultCode obtained
     * @param data        the incoming data
     * @return true/false if successfully managed
     */
    public boolean onActivityResult(int requestCode, int resultCode, Intent data) {
        if (requestCode == REQUEST_SCREEN_CAPTURE) {
            if (Activity.RESULT_OK != resultCode) {
                mInstance.getConferenceService().onUserCanceledScreenShare();
            } else {
                mLastResultCode = resultCode;
                mLastData = data;
                return true;
            }
        }
        return false;
    }

    public static class RequestScreenSharePermissionEvent {
        public RequestScreenSharePermissionEvent() {
            //empty constructor
        }
    }

    /**
     * From a givevn context, get the current Screen size information
     *
     * @param context the context from which retrieve the information
     * @return a valid point information
     */
    @NonNull
    public Point getScreenSize(@NonNull Context context) {
        Point point = new Point();

        WindowManager windowManager = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
        if (null != windowManager) {
            windowManager.getDefaultDisplay().getSize(point);
        }

        return point;
    }

    /**
     * From given screen information, get the scaled width according to the requested height
     *
     * @param screen_size a Point object populated with the screen's width/height
     * @param height      the requested height of the stream
     * @return the scaled up/down screen size
     */
    @NonNull
    public Point getScreenSizeScaled(@NonNull Point screen_size, int height) {
        Point point = new Point(screen_size);
        int screen_width = screen_size.x;
        int screen_height = screen_size.y;

        if (screen_height > 0) {
            point.y = height;
            point.x = (int) (screen_width * height * 1f / screen_height);
        }
        return point;
    }
}
