package com.voxeet.sdk.core.services;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.sdk.media.sensors.ConferenceLock;
import com.voxeet.sdk.media.sensors.ProximitySensor;
import com.voxeet.sdk.media.sensors.ScreenSensor;
import com.voxeet.audio.AudioRoute;
import com.voxeet.sdk.audio.SoundManager;
import com.voxeet.sdk.core.AbstractVoxeetService;
import com.voxeet.sdk.core.VoxeetSdkTemplate;
import com.voxeet.sdk.core.services.holder.ServiceProviderHolder;
import com.voxeet.sdk.events.success.ConferenceEndedEvent;
import com.voxeet.sdk.events.success.ConferencePreJoinedEvent;
import com.voxeet.sdk.json.ConferenceDestroyedPush;
import com.voxeet.sdk.utils.AudioType;
import com.voxeet.sdk.utils.annotate;

import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;
import org.webrtc.voiceengine.WebRtcAudioUtils;

import java.util.ArrayList;
import java.util.List;

@annotate
public class AudioService extends AbstractVoxeetService {

    private static SoundManager sSoundManager;
    private static final String TAG = AudioService.class.getSimpleName();
    private List<ConferenceLock> locks = new ArrayList<>();

    /**
     * Instantiate a new Audio Service
     *
     * @param instance the voxeet parent instance
     */
    public AudioService(@NonNull VoxeetSdkTemplate instance) {
        super(instance, ServiceProviderHolder.DEFAULT);

        AudioService.preInitSounds(instance.getApplicationContext());
        instance.getEventBus().register(this);

        locks.add(new ProximitySensor(instance.getApplicationContext()));
        locks.add(new ScreenSensor(instance.getApplicationContext()));
    }

    @Nullable
    public static SoundManager getSoundManager() {
        return sSoundManager;
    }

    public static boolean preInitSounds(@NonNull Context applicationContext) {
        if (null == sSoundManager) {
            sSoundManager = new SoundManager(applicationContext);
            return true;
        }

        return false;
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferencePreJoinedEvent event) {
        enableAec(true);
        enableNoiseSuppressor(true);

        updateSensors(currentRoute());
        sSoundManager.onConferencePreJoinedEvent();
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceEndedEvent event) {
        sSoundManager.onConferenceDestroyedPush();
        releaseSensors();
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceDestroyedPush event) {
        sSoundManager.onConferenceDestroyedPush();
        releaseSensors();
    }

    /**
     * Get the current available routes
     *
     * @return a non null route
     */
    @NonNull
    public List<AudioRoute> getAvailableRoutes() {
        return sSoundManager.getAvailableRoutes();
    }

    public AudioRoute currentRoute() {
        return sSoundManager.currentRoute();
    }

    public boolean setAudioRoute(@NonNull AudioRoute route) {
        boolean new_switch = sSoundManager.setAudioRoute(route);
        updateSensors(currentRoute());

        return new_switch;
    }

    public AudioService checkOutputRoute() {
        sSoundManager.checkOutputRoute();
        return this;
    }

    public void setSpeakerMode(boolean isSpeaker) {
        sSoundManager.setSpeakerMode(isSpeaker);
        updateSensors(currentRoute());
    }

    public boolean isBluetoothHeadsetConnected() {
        return sSoundManager.isBluetoothHeadsetConnected();
    }

    public boolean isWiredHeadsetOn() {
        return sSoundManager.isWiredHeadsetOn();
    }

    public boolean isSpeakerOn() {
        return sSoundManager.isSpeakerOn();
    }

    public AudioService acquireLocks() {
        sSoundManager.acquireLocks();
        return this;
    }

    public AudioService releaseLocks() {
        sSoundManager.releaseLocks();
        return this;
    }

    public AudioService resetDefaultSoundType() {
        sSoundManager.resetDefaultSoundType();
        return this;
    }

    public AudioService abandonAudioFocusRequest() {
        sSoundManager.abandonAudioFocusRequest();
        return this;
    }

    public AudioService requestAudioFocus() {
        sSoundManager.requestAudioFocus();
        return this;
    }

    public boolean setSound(@NonNull AudioType type, @NonNull String assetName) {
        return sSoundManager.setSound(type, assetName);
    }

    public boolean setSound(@NonNull AudioType type, @NonNull String assetName, int soundMode) {
        return sSoundManager.setSound(type, assetName, soundMode);
    }

    public AudioService playSoundType(@NonNull AudioType type) {
        sSoundManager.playSoundType(type);
        return this;
    }

    public AudioService playSoundType(@NonNull AudioType type, int soundMode) {
        sSoundManager.playSoundType(type, soundMode);
        return this;
    }

    public AudioService stopSoundType(@NonNull AudioType audioType) {
        sSoundManager.stopSoundType(audioType);
        return this;
    }

    public AudioService stopSoundType(@NonNull AudioType audioType, int soundMode) {
        sSoundManager.stopSoundType(audioType, soundMode);
        return this;
    }

    public AudioService stop() {
        sSoundManager.stop();
        return this;
    }

    public AudioService enable() {
        sSoundManager.enable();
        return this;
    }

    public AudioService enableMedia() {
        sSoundManager.enableMedia();
        return this;
    }

    public AudioService unsetMediaRoute() {
        sSoundManager.unsetMediaRoute();
        return this;
    }

    public AudioService setMediaRoute() {
        sSoundManager.setMediaRoute();
        return this;
    }

    public AudioService disable() {
        sSoundManager.disable();
        stop();
        return this;
    }

    public void updateSensors(@NonNull AudioRoute route) {
        for (ConferenceLock lock : locks) {
            if (!route.useProximitySensor() && lock.isProximity()) {
                lock.release();
            } else {
                lock.acquire();
            }
        }
    }

    public void releaseSensors() {
        Log.d("ProximitySensor", "releaseSensors: ");
        for (ConferenceLock lock : locks) {
            lock.release();
        }
    }

    public boolean enableAec(boolean enable) {
        WebRtcAudioUtils.setWebRtcBasedAcousticEchoCanceler(enable);
        return true;
    }

    public boolean enableNoiseSuppressor(boolean enable) {
        WebRtcAudioUtils.setWebRtcBasedNoiseSuppressor(enable);
        return true;
    }
}
