package com.voxeet.sdk.core.abs.promises;

import android.content.Intent;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.android.media.MediaEngineException;
import com.voxeet.sdk.core.abs.AbstractPromiseable;
import com.voxeet.sdk.core.abs.ConferenceService;
import com.voxeet.sdk.core.abs.information.ConferenceInformation;
import com.voxeet.sdk.core.abs.ConferenceSdkObservableProvider;
import com.voxeet.sdk.exceptions.ExceptionManager;
import com.voxeet.sdk.utils.HttpHelper;
import com.voxeet.sdk.core.preferences.VoxeetPreferences;
import com.voxeet.sdk.core.services.MediaService;
import com.voxeet.sdk.events.error.HttpException;
import com.voxeet.sdk.events.promises.NotInConferenceException;
import com.voxeet.sdk.events.success.StartScreenShareAnswerEvent;
import com.voxeet.sdk.json.StartScreenSharingResponse;

import org.greenrobot.eventbus.EventBus;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.ErrorPromise;
import eu.codlab.simplepromise.solve.PromiseExec;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;
import retrofit2.Call;
import retrofit2.Response;

public class StartScreensharePromise extends AbstractPromiseable<Boolean> {
    private final static String TAG = StartScreensharePromise.class.getSimpleName();
    private final Intent intent;
    private final int width;
    private final int height;

    public StartScreensharePromise(@NonNull ConferenceService parent,
                                   @NonNull MediaService mediaService,
                                   @NonNull ConferenceSdkObservableProvider provider,
                                   @Nullable ConferenceInformation information,
                                   @NonNull EventBus eventBus,
                                   @NonNull Intent intent,
                                   int width,
                                   int height) {
        super(parent, mediaService, provider, information, eventBus);
        this.intent = intent;
        this.width = width;
        this.height = height;
    }

    @NonNull
    @Override
    public Promise<Boolean> createPromise() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                try {
                    final ConferenceInformation information = getInformation();

                    if (null == information || !getParent().isInConference()) {
                        Log.d(TAG, "stopVideo: not in conf");

                        if (getMediaService().hasMedia()) getMediaService().getMedia().stopVideo();
                        throw new NotInConferenceException();
                    }

                    String conferenceId = information.getConference().getConferenceId();

                    if (information.isScreenShareOn()) {
                        solver.resolve(true);
                    }

                    if (!hasMedia()) {
                        Log.d(TAG, "startScreenShare: media is null");
                        throw new MediaEngineException("Media is null, invalid state");
                    }

                    information.setScreenShareOn(true);
                    getMedia().startScreenCapturer(intent, width, height);

                    final Call<StartScreenSharingResponse> startScreenShare = getProvider().getStartScreenShareObservable(conferenceId, VoxeetPreferences.id());
                    HttpHelper.enqueue(startScreenShare, new HttpHelper.HttpCallback<StartScreenSharingResponse>() {
                        @Override
                        public void onSuccess(@NonNull StartScreenSharingResponse object, @NonNull Response<StartScreenSharingResponse> response) {
                            StartScreenSharingResponse body = response.body();
                            information.setScreenShareOn(true);

                            final StartScreenShareAnswerEvent event = new StartScreenShareAnswerEvent(true);
                            getEventBus().post(event);
                            createVideoAnswer(body.getUserId(), body.getDescription(), body.getCandidates())
                                    .then(new PromiseExec<Boolean, Object>() {
                                        @Override
                                        public void onCall(@Nullable Boolean aBoolean, @NonNull Solver<Object> internal_solver) {
                                            Log.d(TAG, "onCall: createVideoAnswer := having result " + aBoolean);
                                            solver.resolve(true);
                                        }
                                    })
                                    .error(new ErrorPromise() {
                                        @Override
                                        public void onError(Throwable error) {
                                            ExceptionManager.sendException(error);
                                            error.printStackTrace();
                                            solver.resolve(true);
                                        }
                                    });
                        }

                        @Override
                        public void onFailure(@NonNull Throwable e, @Nullable Response<StartScreenSharingResponse> response) {
                            HttpException.dumpErrorResponse(response);

                            e.printStackTrace();
                            information.setScreenShareOn(false);

                            StartScreenShareAnswerEvent event = new StartScreenShareAnswerEvent(false);
                            getEventBus().post(event);
                            solver.reject(e);
                        }
                    });
                } catch (Exception e) {
                    e.printStackTrace();
                    solver.reject(e);
                } catch (NotInConferenceException e) {
                    e.printStackTrace();
                    solver.reject(e);
                }
            }
        });
    }
}
