package com.voxeet.sdk.factories;

import android.content.Context;
import android.content.Intent;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;
import android.view.WindowManager;

import com.voxeet.sdk.core.preferences.VoxeetPreferences;
import com.voxeet.sdk.json.ConferenceDestroyedPush;

import java.util.Map;

public class VoxeetIntentFactory {

    public static final String NOTIF_TYPE_INVITATION_RECEIVED = "1";
    public static final String NOTIF_TYPE_CONTACT_SUBSCRIPTION_RECEIVED = "2";
    public static final String NOTIF_TYPE_OWN_EVENTS_CLEARED = "3";
    public static final String NOTIF_TYPE_OWN_ACTIVITIES_CLEARED = "4";
    public static final String NOTIF_TYPE_LOCAL_CONFERENCE_DESTROYED = "5";
    public static final String NOTIF_TYPE_OWN_CONFERENCE_CREATED = "6";
    public static final String NOTIF_TYPE_CONTACTS_SUGGESTION = "7";
    public static final String NOTIF_TYPE_MESSAGE_RECEIVED = "8";

    /**
     * Error message when failing to create activity class.
     */
    private final static String defaultError = "Default activity to be triggered when receiving a notification has not been set. Please use setPathToActivity() with the path of the desired activity.";

    public static final String CONF_ID = "ConfId";

    public static final String INVITER_NAME = "DisplayName";

    public static final String INVITER_EXTERNAL_ID = "ExternalId";

    public static final String INVITER_ID = "UserId";

    public static final String INVITER_URL = "AvatarUrl";

    public static final String NOTIF_TYPE = "NotifType";

    public static final String[] DEFAULT_NOTIFICATION_KEYS = new String[]{
            INVITER_ID,
            INVITER_NAME,
            NOTIF_TYPE,
            INVITER_EXTERNAL_ID,
            INVITER_URL,
            CONF_ID
    };

    private static final String TAG = VoxeetIntentFactory.class.getSimpleName();

    /**
     * Get the class to implement from the saved activity
     *
     * @param path a valid or null class
     * @return a valid Class object or null if no class is to be used
     */
    @Nullable
    private static Class<?> createClass(@Nullable String path) {
        if (path == null)
            throw new IllegalStateException(defaultError);

        try {
            return Class.forName(path);
        } catch (ClassNotFoundException e) {
            return null;
        }
    }

    /**
     * Build the intent from a specific notification
     *
     * @param context a valid context
     * @param path a valid or null path of the class which will be called
     * @param data
     * @return a valid intent or null
     */
    @Nullable
    public static Intent buildFrom(@NonNull Context context, @Nullable String path, @NonNull Map<String, String> data) {
        Class<?> newClass = createClass(path);

        for (String string : data.keySet()) {
            Log.d(TAG, "buildFrom: "+ string +" "+data.get(string));
        }

        if (newClass != null) {
            Intent intent = new Intent();
            intent.setClass(context, newClass);

            for (String key : DEFAULT_NOTIFICATION_KEYS) {
                if(data.containsKey(key)) {
                    intent.putExtra(key, data.get(key));
                }
            }

            //force conference join since it is a push
            intent.putExtra("join", true);
            intent.putExtra("callMode", 0x0001);
            intent.addFlags(Intent.FLAG_FROM_BACKGROUND);
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);


            //TODO check usefullness
            intent.addFlags(WindowManager.LayoutParams.FLAG_SHOW_WHEN_LOCKED |
                    WindowManager.LayoutParams.FLAG_DISMISS_KEYGUARD |
                    WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON |
                    WindowManager.LayoutParams.FLAG_TURN_SCREEN_ON
            );

            return intent;
        }
        return null;
    }


    /**
     * Build the intent from a specific notification
     *
     * @param context a valid context
     * @param path a valid or null path of the class which will be called
     * @param extra a valid extra bundle
     * @return a valid intent or null
     */
    @Nullable
    public static Intent buildFrom(@NonNull Context context, @Nullable String path, @NonNull Bundle extra) {
        Class<?> newClass = createClass(path);

        if (newClass != null) {
            Intent intent = new Intent();
            intent.setClass(context, newClass);

            for (String key : DEFAULT_NOTIFICATION_KEYS) {
                if(extra.containsKey(key)) {
                    intent.putExtra(key, extra.getString(key));
                }
            }

            //force conference join since it is a push
            intent.putExtra("join", true);
            intent.putExtra("callMode", 0x0001);
            intent.addFlags(Intent.FLAG_FROM_BACKGROUND);
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);


            //TODO check usefullness
            intent.addFlags(WindowManager.LayoutParams.FLAG_SHOW_WHEN_LOCKED |
                    WindowManager.LayoutParams.FLAG_DISMISS_KEYGUARD |
                    WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON |
                    WindowManager.LayoutParams.FLAG_TURN_SCREEN_ON
            );

            return intent;
        }
        return null;
    }
}
