package com.voxeet.sdk.core;

import android.app.Application;
import android.content.Context;
import android.content.Intent;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.util.Log;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.ErrorPromise;
import eu.codlab.simplepromise.solve.PromiseExec;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;

import com.voxeet.sdk.core.abs.ConferenceService;
import com.voxeet.sdk.core.preferences.VoxeetPreferences;
import com.voxeet.sdk.core.services.AudioService;
import com.voxeet.sdk.core.services.LocalStatsService;
import com.voxeet.sdk.core.services.MediaService;
import com.voxeet.sdk.core.services.SDKFilePresentationService;
import com.voxeet.sdk.core.services.SDKVideoPresentationService;
import com.voxeet.sdk.core.services.ScreenShareService;
import com.voxeet.sdk.core.services.UserService;
import com.voxeet.sdk.core.services.authenticate.token.RefreshTokenCallback;
import com.voxeet.sdk.factories.VoxeetIntentFactory;
import com.voxeet.sdk.json.ConferenceDestroyedPush;
import com.voxeet.sdk.json.UserInfo;
import com.voxeet.sdk.utils.EventCallback;
import com.voxeet.sdk.utils.Validate;

import java.util.Map;

public final class VoxeetSdk extends VoxeetSdkTemplate {

    private static Application CurrentApplication = null;

    private static VoxeetSdk CurrentInstance;
    private final static String TAG = VoxeetSdk.class.getSimpleName();

    protected static void setInstance(@NonNull VoxeetSdk sdk) {
        CurrentInstance = sdk;

        Validate.hasInternetPermissions(sdk.getApplicationContext(), true);
        VoxeetPreferences.init(sdk.getApplicationContext(),
                sdk.getVoxeetEnvironmentHolder());
    }

    public static VoxeetSdk getInstance() {
        return CurrentInstance;
    }

    @Nullable
    public static VoxeetSdk instance() {
        return getInstance();
    }

    public static void setApplication(@NonNull Application application) {
        CurrentApplication = application;
    }

    /**
     * This function initializes the Voxeet SDK, the behavior of Voxeet SDK functions are
     * undetermined if this function is not called. It should be called as early as possible.
     * <p>
     * Using a null value for the UserInfo? Prefer the signature without it then
     *
     * @param appId    the app id
     * @param password the password
     * @param userInfo the user info
     */
    public static synchronized void initialize(@NonNull String appId,
                                               @NonNull String password,
                                               @Nullable final UserInfo userInfo) {
        initialize(CurrentApplication, appId, password, userInfo);
    }


    /**
     * This function initializes the Voxeet SDK, the behavior of Voxeet SDK functions are
     * undetermined if this function is not called. It should be called as early as possible.
     *
     * @param appId    the app id
     * @param password the password
     */
    public static synchronized void initialize(@NonNull String appId,
                                               @NonNull String password) {
        initialize(appId, password, null);
    }

    /**
     * This function initializes the Voxeet SDK, the behavior of Voxeet SDK functions are
     * undetermined if this function is not called. It should be called as early as possible.
     * <p>
     * Using a null value for the UserInfo? Prefer the signature without it then
     *
     * @param application The application context
     * @param appId       the app id
     * @param password    the password
     * @param userInfo    the user info
     */
    public static synchronized void initialize(@NonNull final Application application,
                                               @NonNull String appId,
                                               @NonNull String password,
                                               @Nullable final UserInfo userInfo) {
        if (null == CurrentInstance) {
            VoxeetSdk sdk = new VoxeetSdk(application,
                    appId,
                    password,
                    userInfo);

            VoxeetSdk.setInstance(sdk);
        } else {
            Log.d(TAG, "initialize: Instance already started !");
        }
    }

    /**
     * This function initializes the Voxeet SDK, the behavior of Voxeet SDK functions are
     * undetermined if this function is not called. It should be called as early as possible.
     *
     * @param application The application context
     * @param appId       the app id
     * @param password    the password
     */
    public static synchronized void initialize(@NonNull final Application application,
                                               @NonNull String appId,
                                               @NonNull String password) {
        initialize(application, appId, password, null);
    }

    /**
     * This function initializes the Voxeet SDK, the behavior of Voxeet SDK functions are
     * undetermined if this function is not called. It should be called as early as possible.
     * <p>
     * This method will invoke the SDK with the thirdparty authentication feature
     * <p>
     * Using a null value for the UserInfo? Prefer the signature without it then
     *
     * @param application  The application context
     * @param accessToken  the default accessToken to use
     * @param refreshToken the refreshToken callback
     * @param userInfo     the user info
     */
    public static synchronized void initialize(@NonNull final Application application,
                                               @NonNull String accessToken,
                                               @NonNull RefreshTokenCallback refreshToken,
                                               @Nullable final UserInfo userInfo) {
        if (null == CurrentInstance) {
            VoxeetSdk sdk = new VoxeetSdk(application,
                    accessToken,
                    refreshToken,
                    userInfo);

            VoxeetSdk.setInstance(sdk);
        } else {
            Log.d(TAG, "initialize: Instance already started !");
        }
    }

    /**
     * This function initializes the Voxeet SDK, the behavior of Voxeet SDK functions are
     * undetermined if this function is not called. It should be called as early as possible.
     * <p>
     * This method will invoke the SDK with the thirdparty authentication feature
     *
     * @param application  The application context
     * @param accessToken  the default accessToken to use
     * @param refreshToken the refreshToken callback
     */
    public static synchronized void initialize(@NonNull final Application application,
                                               @NonNull String accessToken,
                                               @NonNull RefreshTokenCallback refreshToken) {
        initialize(application, accessToken, refreshToken, null);
    }

    /**
     * This function initializes the Voxeet SDK, the behavior of Voxeet SDK functions are
     * undetermined if this function is not called. It should be called as early as possible.
     * <p>
     * This method will invoke the SDK with the thirdparty authentication feature
     * <p>
     * Using a null value for the UserInfo? Prefer the signature without it then
     *
     * @param accessToken  the default accessToken to use
     * @param refreshToken the refreshToken callback
     * @param userInfo     the user info
     */
    public static synchronized void initialize(@NonNull String accessToken,
                                               @NonNull RefreshTokenCallback refreshToken,
                                               @Nullable final UserInfo userInfo) {
        initialize(CurrentApplication, accessToken, refreshToken, userInfo);
    }

    /**
     * This function initializes the Voxeet SDK, the behavior of Voxeet SDK functions are
     * undetermined if this function is not called. It should be called as early as possible.
     * <p>
     * This method will invoke the SDK with the thirdparty authentication feature
     *
     * @param accessToken  the default accessToken to use
     * @param refreshToken the refreshToken callback
     */
    public static synchronized void initialize(@NonNull String accessToken,
                                               @NonNull RefreshTokenCallback refreshToken) {
        initialize(CurrentApplication, accessToken, refreshToken, null);
    }

    private VoxeetSdk(@NonNull Application application_context, @NonNull String appId, @NonNull String password, @Nullable UserInfo userInfo) {
        super(application_context, appId, password, userInfo, new VoxeetEnvironmentHolder(application_context), false);
    }

    private VoxeetSdk(@NonNull Application application_context, @NonNull String accessToken, @NonNull RefreshTokenCallback refreshToken, @Nullable UserInfo userInfo) {
        super(application_context, accessToken, refreshToken, userInfo, new VoxeetEnvironmentHolder(application_context), false);
    }

    public Promise<Boolean> logout() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                Log.d(TAG, "onCall: logout called");

                lockConnectAttempt();
                try {
                    throw new IllegalStateException("You have awaiting login, the logout automatically canceled those");
                } catch (Exception e) {
                    rejectLoginSockets(e);
                }
                clearLoginSockets();
                unlockConnectAttempt();

                Log.d(TAG, "onCall: clean awaiting sockets done");
                getConferenceService().logout().then(new PromiseExec<Boolean, Object>() {
                    @Override
                    public void onCall(@Nullable Boolean result, @NonNull Solver<Object> s) {
                        Log.d(TAG, "onCall: logout result := " + result + " ... propagating...");
                        solver.resolve(result);
                    }
                }).error(new ErrorPromise() {
                    @Override
                    public void onError(@NonNull Throwable error) {
                        solver.reject(error);
                    }
                });
            }
        });
    }

    @Override
    protected void initServices() {
        getServices().put(ConferenceService.class, new ConferenceService(this, DEFAULT_TIMEOUT_MS));
        getServices().put(UserService.class, new UserService(this));
        getServices().put(SDKFilePresentationService.class, new SDKFilePresentationService(this));
        getServices().put(SDKVideoPresentationService.class, new SDKVideoPresentationService(this));
        getServices().put(AudioService.class, new AudioService(this));
        getServices().put(ScreenShareService.class, new ScreenShareService(this));
        getServices().put(MediaService.class, new MediaService(this));
        getServices().put(LocalStatsService.class, new LocalStatsService(this));
    }

    @Nullable
    public static ConferenceService conference() {
        if(null == getInstance()) return null;
        return getInstance().getConferenceService();
    }

    @Nullable
    public static MediaService mediaDevice() {
        if(null == getInstance()) return null;
        return getInstance().getMediaService();
    }

    @Nullable
    public static ScreenShareService screenShare() {
        if(null == getInstance()) return null;
        return getInstance().getScreenShareService();
    }

    @Nullable
    public static UserService user() {
        if(null == getInstance()) return null;
        return getInstance().getUserService();
    }

    @Nullable
    public static SDKFilePresentationService filePresentation() {
        if(null == getInstance()) return null;
        return getInstance().getFilePresentationService();
    }

    @Nullable
    public static SDKVideoPresentationService videoPresentation() {
        if(null == getInstance()) return null;
        return getInstance().getVideoPresentationService();
    }

    @Nullable
    public static AudioService audio() {
        if(null == getInstance()) return null;
        return getInstance().getAudioService();
    }

    @Nullable
    public static LocalStatsService localStats() {
        if(null == getInstance()) return null;
        return getInstance().getLocalStatsService();
    }

    public UserService getUserService() {
        return getServiceForKlass(UserService.class);
    }

    public SDKFilePresentationService getFilePresentationService() {
        return getServiceForKlass(SDKFilePresentationService.class);
    }

    public SDKVideoPresentationService getVideoPresentationService() {
        return getServiceForKlass(SDKVideoPresentationService.class);
    }


    public boolean manageRemoteMessage(@NonNull Context context, @NonNull Map<String, String> remoteMessage) {
        if (remoteMessage.containsKey(VoxeetIntentFactory.NOTIF_TYPE)) {
            String notificationType = remoteMessage.get(VoxeetIntentFactory.NOTIF_TYPE);
            if (null == notificationType || TextUtils.isEmpty(notificationType)) notificationType = "";

            switch (notificationType) {
                case VoxeetIntentFactory.NOTIF_TYPE_INVITATION_RECEIVED:
                    Intent intent = VoxeetIntentFactory.buildFrom(context, VoxeetPreferences.getDefaultActivity(), remoteMessage);
                    if (intent != null)
                        getApplicationContext().startActivity(intent);
                    break;
                case VoxeetIntentFactory.NOTIF_TYPE_LOCAL_CONFERENCE_DESTROYED:
                    String conferenceId = "";
                    if (remoteMessage.containsKey(VoxeetIntentFactory.CONF_ID)) {
                        conferenceId = remoteMessage.get(VoxeetIntentFactory.CONF_ID);
                    }
                    if (null == conferenceId || TextUtils.isEmpty(conferenceId)) conferenceId = "";
                    getEventBus().post(new ConferenceDestroyedPush(conferenceId));
                    break;
                default:
            }

            return true;
        }
        return false;
    }

    @Override
    protected String getTag() {
        return VoxeetSdk.class.getSimpleName();
    }

    public void registerEventCallback(EventCallback callback) {
        callback.register(this);
    }
}
