package com.voxeet.sdk.utils;

import android.content.Context;
import android.content.res.AssetFileDescriptor;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;

public class SoundPool {
    private static final String TAG = SoundPool.class.getSimpleName();
    private Context mContext;
    private android.media.SoundPool mShortPlayer = null;
    private HashMap<AudioType, Sound> mSounds = new HashMap();
    private List<Integer> mLooping = new ArrayList<>();

    private SoundPool() {
        //nothing to do by default
    }

    public SoundPool(@NonNull Context pContext, int soundMode) {
        this();

        // setup Soundpool
        this.mContext = pContext;
        mShortPlayer = new android.media.SoundPool(4, soundMode, 0);
        mShortPlayer.setOnLoadCompleteListener(new android.media.SoundPool.OnLoadCompleteListener() {
            @Override
            public void onLoadComplete(android.media.SoundPool soundPool, int sampleId, int status) {
                Log.d(TAG, "onLoadComplete: a new sound is loaded sampleId:=" + sampleId);
                for (AudioType type : mSounds.keySet()) {
                    Sound sound = mSounds.get(type);
                    if (null != sound && sound.getSoundId() == sampleId) {
                        Log.d(TAG, "onLoadComplete: loaded ? " + sampleId + " " + status);
                        sound.setLoaded(true);
                        if (sound.isToPlay()) {
                            play(sound);
                        }
                    }
                }
            }
        });
    }

    private boolean isLooping(int id) {
        for (Integer integer : mLooping) {
            if (null != integer && integer == id) return true;
        }
        return false;
    }

    public SoundPool playShortResource(@NonNull AudioType type,
                                       @NonNull String assetName) {
        Log.d(TAG, "playShortResource: calling " + type);
        Sound sound = getOrCreate(type, assetName, true);

        if (null != sound) {
            sound.setLoop(type.isLoop());

            if (isLooping(sound.getSoundId())) {
                Log.d(TAG, "playShortResource: already playing");
            } else {
                Log.d(TAG, "playShortResource: playing resource " + sound.getSoundId());
                if (sound.isLoaded()) {
                    play(sound);
                } else {
                    sound.setPlay(true);
                }

                if (type.isLoop()) {
                    mLooping.add(sound.getSoundId());
                }
            }
        }

        return this;
    }

    @Nullable
    private Sound getOrCreate(@NonNull AudioType type,
                              @NonNull String assetName,
                              boolean play) {
        Sound sound = null;
        if (mSounds.containsKey(type)) {
            sound = mSounds.get(type);
        }
        try {
            if (null == sound) {
                sound = new Sound(assetName);

                Log.d(TAG, "playShortResource: not containing, adding it");
                AssetFileDescriptor descriptor = null;
                descriptor = mContext.getAssets().openFd(assetName);

                sound.setLoaded(false);
                sound.setPlay(play);
                int sampleId = this.mShortPlayer.load(descriptor, 1);
                sound.setId(sampleId);
                descriptor.close();
            }
        } catch (IOException e) {
            e.printStackTrace();
            sound = null;
        }

        if (null != sound) {
            mSounds.put(type, sound);
        }

        return sound;
    }

    public SoundPool stop() {
        List<AudioType> keys_to_remove = new ArrayList<>();
        Log.d(TAG, "stop: " + Arrays.toString(mLooping.toArray()));
        for (AudioType key : mSounds.keySet()) {
            Sound value = mSounds.get(key);
            Log.d(TAG, "stop: testing " + value.getName() + " " + value.getSoundId());

            if (value.isLoaded()) {
                if (value.isLoop()) {
                    Log.d(TAG, "stop: add to remove later " + value + " " + key);
                    value.unload()
                            .setLoaded(false)
                            .setLoop(false)
                            .setPlay(false);
                    keys_to_remove.add(key);
                    try {
                        mShortPlayer.pause(value.getStreamId());
                        mShortPlayer.stop(value.getStreamId());
                        mShortPlayer.unload(value.getSoundId());
                    } catch (Exception e) {
                        Log.d(TAG, "stop: exception...");
                        e.printStackTrace();
                    }
                }
            } else {
                value.setPlay(false)
                        .setLoop(false)
                        .setLoaded(false)
                        .unload();
            }
        }

        for (AudioType key : keys_to_remove) {
            Sound sound = mSounds.get(key);
            Log.d(TAG, "stop: remove " + key + " " + sound.getName() + " " + sound.getSoundId());
            mSounds.remove(key);
        }

        /*for (Sound id : mSounds.values()) {
            Log.d(TAG, "stop: stopping " + id.getName() + " " + id.getSoundId());
            try {
                mShortPlayer.stop(id.getStreamId());
            } catch (Exception e) {

            }
        }*/

        mLooping.clear();

        return this;
    }

    public SoundPool stop(@NonNull AudioType key) {
        Log.d(TAG, "stop: " + Arrays.toString(mLooping.toArray()));
        Sound value = mSounds.get(key);

        if(null == value) return this;

        if (value.isLoaded()) {
            Log.d(TAG, "stop: testing " + value.getName() + " " + value.getSoundId());

            if (value.isLoop()) {
                Log.d(TAG, "stop: add to remove later " + value + " " + key);
                value.unload()
                        .setLoaded(false)
                        .setLoop(false)
                        .setPlay(false);

                try {
                    mShortPlayer.pause(value.getStreamId());
                    mShortPlayer.stop(value.getStreamId());
                    mShortPlayer.unload(value.getSoundId());
                } catch (Exception e) {
                    Log.d(TAG, "stop: exception...");
                    e.printStackTrace();
                }
            }
        } else {
            value.setPlay(false)
                    .setLoop(false)
                    .setLoaded(false)
                    .unload();
        }

        mSounds.remove(key);
        mLooping.clear();

        return this;
    }

    public void release() {
        mSounds.clear();
        mShortPlayer.release();
        mShortPlayer = null;
    }

    public boolean setShortResource(@NonNull AudioType type,
                                    @NonNull String assetName) {
        Sound sound = getOrCreate(type, assetName, false);
        if (null != sound) {
            Log.d(TAG, "setShortResource: sound created ? " + sound.getName() + " " + sound.getSoundId());
            return true;
        }
        Log.d(TAG, "setShortResource: error while creating the sound");
        return false;
    }

    private SoundPool play(@NonNull Sound sound) {
        int streamId = mShortPlayer.play(sound.getSoundId(), 0.1f, 0.1f,
                0, sound.isLoop() ? -1 : 0, 1);
        sound.setStreamId(streamId);

        return this;
    }

    public SoundPool release(@NonNull AudioType type) {
        if (mSounds.containsKey(type)) {
            Sound sound = mSounds.get(type);
            if (null != sound) {
                mShortPlayer.stop(sound.getStreamId());
                mShortPlayer.unload(sound.getSoundId());
            }
        }
        return this;
    }
}
