package com.voxeet.sdk.core.services;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.sdk.media.MediaSDK;
import com.voxeet.android.media.stats.LocalStats;
import com.voxeet.sdk.core.AbstractVoxeetService;
import com.voxeet.sdk.core.VoxeetSdkTemplate;
import com.voxeet.sdk.core.services.holder.ServiceProviderHolder;
import com.voxeet.sdk.core.services.localstats.LocalStatsHandler;
import com.voxeet.sdk.core.services.localstats.LocalStatsUserInfo;
import com.voxeet.sdk.models.abs.Conference;
import com.voxeet.sdk.models.abs.ConferenceUser;

import java.util.HashMap;
import java.util.List;

public class LocalStatsService extends AbstractVoxeetService implements Runnable {
    public final static int NEXT_LOOP_FETCH_MS = 500;

    private VoxeetSdkTemplate mInstance;

    private LocalStatsHandler mHandler;
    private HashMap<String, HashMap<String, LocalStatsUserInfo>> mConferencesHolder;

    public LocalStatsService(@NonNull VoxeetSdkTemplate instance) {
        super(instance, ServiceProviderHolder.DEFAULT);

        mConferencesHolder = new HashMap<>();
        mHandler = null;
        mInstance = instance;
    }

    /**
     * Get the specified user stats from a local point of view
     *
     * @param peer the user id
     * @return LocalStats or null
     */
    @Nullable
    public LocalStats getLocalStats(@Nullable String peer) {
        if(null == peer) return null;
        MediaSDK media = mInstance.getMediaService().getMedia();
        if (null != media) return media.getLocalStats(peer);
        return null;
    }


    public boolean startAutoFetch() {
        if (null == mHandler) {
            mHandler = new LocalStatsHandler(this, NEXT_LOOP_FETCH_MS);
            mHandler.startAutoFetch();
            return true;
        } else {
            return false;
        }
    }

    public boolean stopAutoFetch() {
        if (null != mHandler) {
            mHandler.stopAutoFetch();
            mHandler = null;
            return true;
        }
        return false;
    }

    @Override
    public void run() {
        //List<LocalStats> stats = new ArrayList<>();
        Conference conference = mInstance.getConferenceService().getConference();

        if (null != conference) {
            List<ConferenceUser> users = conference.getConferenceUsers();
            for (int i = 0; i < users.size(); i++) {
                ConferenceUser user = users.get(i);
                LocalStats stat = getLocalStats(user.getUserId());
                getUserInfo(conference.getConferenceId(), user.getUserId()).addFromLocalStats(stat);
            }
        } else {
            //no conference ?
        }
    }

    /**
     * Retrieve the in-memory holder of the conference information specifically designed for the stats
     *
     * @param conferenceId the conferenceId - not the alias
     * @return a non null HashMap instance
     */
    @NonNull
    private HashMap<String, LocalStatsUserInfo> getUserStatsHolder(@NonNull String conferenceId) {
        HashMap<String, LocalStatsUserInfo> holder = mConferencesHolder.get(conferenceId);

        if (null == holder) {
            holder = new HashMap<>();
            mConferencesHolder.put(conferenceId, holder);
        }
        return holder;
    }

    /**
     * Retrieve the in-memory holder of the user specifically designed for the user
     *
     * @param conferenceId the conferenceId - not the alias
     * @param userId       the userId - not the externalId
     * @return a valid user info holder
     */
    public LocalStatsUserInfo getUserInfo(@NonNull String conferenceId, @NonNull String userId) {
        HashMap<String, LocalStatsUserInfo> holder = getUserStatsHolder(conferenceId);
        LocalStatsUserInfo info = holder.get(userId);

        if (null == info) {
            info = new LocalStatsUserInfo(userId);
            holder.put(userId, info);
        }
        return info;
    }
}
