package com.voxeet.sdk.core.abs.promises;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.sdk.core.abs.AbstractPromiseable;
import com.voxeet.sdk.core.abs.ConferenceService;
import com.voxeet.sdk.core.abs.information.ConferenceInformation;
import com.voxeet.sdk.core.abs.ConferenceSdkObservableProvider;
import com.voxeet.sdk.utils.HttpHelper;
import com.voxeet.sdk.core.preferences.VoxeetPreferences;
import com.voxeet.sdk.core.services.MediaService;
import com.voxeet.sdk.events.error.HttpException;
import com.voxeet.sdk.events.error.SdkLogoutErrorEvent;
import com.voxeet.sdk.events.success.SdkLogoutSuccessEvent;

import org.greenrobot.eventbus.EventBus;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;
import okhttp3.ResponseBody;
import retrofit2.Call;
import retrofit2.Response;

public class LogoutPromise extends AbstractPromiseable<Boolean> {
    private final static String TAG = LogoutPromise.class.getSimpleName();

    public LogoutPromise(@NonNull ConferenceService parent,
                         @NonNull MediaService mediaService,
                         @NonNull ConferenceSdkObservableProvider provider,
                         @Nullable ConferenceInformation information,
                         @NonNull EventBus eventBus) {
        super(parent, mediaService, provider, information, eventBus);
    }

    @NonNull
    @Override
    public Promise<Boolean> createPromise() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                final String id = VoxeetPreferences.id();
                if (null != id) {
                    Log.d(TAG, "Attempting to logout");

                    final Call<ResponseBody> user = getProvider().getLogOutObservable(VoxeetPreferences.token());
                    HttpHelper.enqueue(user, new HttpHelper.HttpCallback<ResponseBody>() {
                        @Override
                        public void onSuccess(@NonNull ResponseBody object, @NonNull Response<ResponseBody> response) {
                            String message;
                            if (response.code() == 200) {
                                message = "Logout success";

                                Log.d(TAG, message);

                                VoxeetPreferences.onLogout();
                                getVoxeetSDK().closeSocket();
                                getVoxeetSDK().cleanUserSession(id);

                                getEventBus().post(new SdkLogoutSuccessEvent(message));
                                solver.resolve(true);
                            } else {
                                message = "Logout failed";

                                Log.d(TAG, message);

                                VoxeetPreferences.onLogout();
                                getVoxeetSDK().closeSocket();
                                getVoxeetSDK().cleanUserSession(id);

                                getEventBus().post(new SdkLogoutErrorEvent(message));
                                solver.resolve(false);
                            }
                        }

                        @Override
                        public void onFailure(@NonNull Throwable e, @Nullable Response<ResponseBody> response) {
                            HttpException.dumpErrorResponse(response);

                            e.printStackTrace();
                            String message = "Logout failed";
                            Log.d(TAG, message);

                            VoxeetPreferences.onLogout();
                            getVoxeetSDK().closeSocket();
                            getVoxeetSDK().cleanUserSession(id);

                            getEventBus().post(new SdkLogoutErrorEvent(getParent().handleError(e)));
                            solver.reject(e);
                        }
                    });
                } else {
                    final String message = "Already logged out";
                    Log.d(TAG, message);
                    getEventBus().post(new SdkLogoutSuccessEvent("Already logged out"));
                    solver.resolve(true);
                }
            }
        });
    }
}
