package com.voxeet.sdk.core;

import android.annotation.SuppressLint;
import android.app.Application;
import android.provider.Settings;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.voxeet.sdk.converter.JacksonConverterFactory;
import com.voxeet.sdk.core.network.IVoxeetRService;
import com.voxeet.sdk.core.preferences.VoxeetPreferences;
import com.voxeet.sdk.core.services.authenticate.KeySecretTokenResponseProvider;
import com.voxeet.sdk.core.services.authenticate.OAuthTokenResponseProvider;
import com.voxeet.sdk.core.services.authenticate.VoxeetCookieJar;
import com.voxeet.sdk.core.services.authenticate.token.RefreshTokenCallback;
import com.voxeet.sdk.core.services.authenticate.token.TokenResponseProvider;
import com.voxeet.sdk.json.UserInfo;
import com.voxeet.sdk.models.UserTokenResponse;
import com.voxeet.sdk.networking.DeviceType;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.UnknownHostException;
import java.util.List;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLSession;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.ErrorPromise;
import eu.codlab.simplepromise.solve.PromiseExec;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;
import okhttp3.Cookie;
import okhttp3.Interceptor;
import okhttp3.OkHttpClient;
import okhttp3.Response;
import retrofit2.Retrofit;

/**
 * VoxeetHttp manages every network call made by the SDK
 * <p>
 * It will identify, re-identify and make the calls
 */
public class VoxeetHttp {

    private static final String TAG = VoxeetHttp.class.getSimpleName();
    private final VoxeetSdkTemplate sdk;

    private Application appContext;

    private String serverUrl;

    private String serverPort;

    /**
     * The Retrofit.
     */
    private Retrofit retrofit;

    private OkHttpClient client;

    private TokenResponseProvider tokenResponseProvider;

    private VoxeetServiceListener listener;

    private boolean debug = false;
    private OkHttpClient clientIdentify;
    private OkHttpClient clientSubIdentify;
    private Retrofit retrofitIdentify;
    private Retrofit retrofitSubIdentify;
    private VoxeetCookieJar cookieJar;

    /**
     * Instantiates a new Voxeet http.
     *
     * @param appContext   the Application Context
     * @param sdk          the current sdk instance
     * @param appId        the app id
     * @param password     the password
     * @param listener     the listener
     * @param tokenAccess  the token access
     * @param tokenRefresh the token refresher
     */
    private VoxeetHttp(Application appContext, VoxeetSdkTemplate sdk,
                       String serverUrl, String serverPort,
                       String appId, String password,
                       VoxeetServiceListener listener,
                       final String tokenAccess, final RefreshTokenCallback tokenRefresh,
                       boolean debug) {
        this.sdk = sdk;
        this.serverUrl = serverUrl;
        this.serverPort = serverPort;
        this.appContext = appContext;

        this.debug = debug;

        this.listener = listener;

        //set 1 provider according to the 2 different cases which be made available
        if (null != appId && null != password) {
            Log.d(TAG, "VoxeetHttp: initializing using appId/password");
            tokenResponseProvider = new KeySecretTokenResponseProvider(appId, password, sdk.getVoxeetEnvironmentHolder());
        } else if (null != tokenRefresh) {
            Log.d(TAG, "VoxeetHttp: initializing using access/refresh");

            //in such case, the user's servers are here to manage oauth
            tokenResponseProvider = new OAuthTokenResponseProvider(tokenAccess, tokenRefresh, sdk.getVoxeetEnvironmentHolder());
        }

        initClient();
        initRetrofit();
        initService();
    }

    /**
     * Overrode retrofit client.
     */
    private void initClient() {
        this.cookieJar = new VoxeetCookieJar(getBuiltServerUrl());

        OkHttpClient.Builder builderIdentify = new OkHttpClient.Builder()
                .cookieJar(cookieJar)
                .addNetworkInterceptor(new Interceptor() {
                    @Override
                    public Response intercept(@NonNull Chain chain) throws IOException {
                        Log.d(TAG, "intercept: builderIdentify");
                        return tokenResponseProvider.executeIdentify(chain, "builderIdentify");
                    }
                });

        OkHttpClient.Builder builderSubIdentify = new OkHttpClient.Builder()
                .cookieJar(cookieJar)
                .addNetworkInterceptor(new Interceptor() {
                    @Override
                    public Response intercept(@NonNull Chain chain) throws IOException {
                        Log.d(TAG, "intercept: builderSubIdentify");
                        return tokenResponseProvider.executeIdentify(chain, "builderSubIdentify");
                    }
                });

        OkHttpClient.Builder builder = new OkHttpClient.Builder()
                .cookieJar(cookieJar)
                .addNetworkInterceptor(new Interceptor() {
                    @Override
                    public Response intercept(@NonNull Chain chain) throws IOException {
                        Log.d(TAG, "intercept: builder");
                        return tokenResponseProvider.execute(chain);
                    }
                });

        tokenResponseProvider.configureOkHttpClientBuilder(builder, true);
        tokenResponseProvider.configureOkHttpClientBuilder(builderIdentify, true);
        tokenResponseProvider.configureOkHttpClientBuilder(builderSubIdentify, false);

        if (this.debug) {
            HostnameVerifier hostnameVerifier = new HostnameVerifier() {
                @SuppressLint("BadHostnameVerifier")
                @Override
                public boolean verify(String hostname, SSLSession session) {
                    return true;
                }
            };

            builder = builder.hostnameVerifier(hostnameVerifier);
            builderIdentify = builderIdentify.hostnameVerifier(hostnameVerifier);
            builderSubIdentify = builderSubIdentify.hostnameVerifier(hostnameVerifier);
        }

        client = builder.build();
        clientIdentify = builderIdentify.build();
        clientSubIdentify = builderSubIdentify.build();
    }

    private void initRetrofit() {
        retrofit = new Retrofit.Builder()
                //.addCallAdapterFactory(RxJavaCallAdapterFactory.create())
                .addConverterFactory(JacksonConverterFactory.create(new ObjectMapper().disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES)))
                .baseUrl(getBuiltServerUrl())
                .client(client)
                .build();


        retrofitIdentify = new Retrofit.Builder()
                //.addCallAdapterFactory(RxJavaCallAdapterFactory.create())
                .addConverterFactory(JacksonConverterFactory.create(new ObjectMapper().disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES)))
                .baseUrl(getBuiltServerUrl())
                .client(clientIdentify)
                .build();

        retrofitSubIdentify = new Retrofit.Builder()
                //.addCallAdapterFactory(RxJavaCallAdapterFactory.create())
                .addConverterFactory(JacksonConverterFactory.create(new ObjectMapper().disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES)))
                .baseUrl(getBuiltServerUrl())
                .client(clientSubIdentify)
                .build();
    }

    private void initService() {
        tokenResponseProvider.initService(retrofitIdentify.create(IVoxeetRService.class),
                retrofitSubIdentify.create(IVoxeetRService.class));
    }

    /**
     * Identify. Similar to a login
     *
     * @param userInfo the user info
     */
    void identify(final UserInfo userInfo) {
        identifyChain(userInfo).then(new PromiseExec<Boolean, Object>() {
            @Override
            public void onCall(@Nullable Boolean result, @NonNull Solver<Object> solver) {
                Log.d(TAG, "onCall: identify finished " + result);
            }
        }).error(new ErrorPromise() {
            @Override
            public void onError(@NonNull Throwable error) {
                error.printStackTrace();
            }
        });
    }

    Promise<Boolean> identifyChain(final UserInfo userInfo) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> final_solver) {
                tokenResponseProvider.setUserInfo(userInfo);

                VoxeetPreferences.setExternalUserInfo(userInfo);

                userInfo.setDeviceIdentifier(Settings.Secure.getString(appContext.getContentResolver(), Settings.Secure.ANDROID_ID));

                if (isTokenUploadAllowed()) {
                    Log.d(TAG, "onCall: isTokenUploadAllowed :: true");
                    String token = getUploadToken();

                    if (token != null) {
                        Log.d(TAG, "FirebaseInstanceId.getInstance().getAccessToken(): " + token);
                        userInfo.setDevicePushToken(token);
                    } else {
                        Log.d(TAG, "FirebaseInstanceId.getInstance().getAccessToken() returned an IllegalStateException, you have an issue with your project configuration (google-services.json for instance)");
                    }

                } else
                    Log.d(TAG, "FirebaseApp is not initialized. Make sure to call FirebaseApp.initializeApp(Context) " +
                            "before initializing the VoxeetSDK if you are planning on using FCM.");

                userInfo.setDeviceType(DeviceType.ANDROID);

                tokenResponseProvider.retrieve().then(new PromiseExec<UserTokenResponse, Object>() {
                    @Override
                    public void onCall(@Nullable UserTokenResponse result, @NonNull Solver<Object> solver) {
                        if (null != result) {
                            Log.d(TAG, "Successful login with id " + result.getId());

                            VoxeetPreferences.setId(result.getId());
                            VoxeetPreferences.saveLoginCookie(result.getUserToken());

                            listener.onIdentifySuccess(VoxeetHttp.this);

                            List<Cookie> cookies = cookieJar.getCookies(getBuiltServerUrl());
                            if (cookies != null && cookies.size() > 0)
                                VoxeetPreferences.saveLoginCookie(cookies.get(0).toString());
                            final_solver.resolve(true);
                        } else {
                            final_solver.resolve(false);
                        }
                    }
                }).error(new ErrorPromise() {
                    @Override
                    public void onError(@NonNull Throwable error) {
                        error.printStackTrace();
                        if (error instanceof UnknownHostException) {
                            listener.onNetworkError((UnknownHostException) error);
                        } else {
                            listener.onIdentifyError(error.getMessage());
                        }
                        final_solver.reject(error);
                    }
                });
            }
        });
    }

    /**
     * Gets client.
     *
     * @return the client
     */

    public OkHttpClient getClient() {
        return client;
    }

    Retrofit getRetrofit() {
        return retrofit;
    }

    @Nullable
    String getToken() {
        return tokenResponseProvider.getToken();
    }

    @Nullable
    public String getJwtToken() {
        return tokenResponseProvider.getJwtToken();
    }

    public void resetVoxeetHttp() {
        tokenResponseProvider.resetVoxeetHttp();
    }

    public void cleanUserSession(@NonNull String id) {
        tokenResponseProvider.cleanUserSession(id);
    }

    /**
     * The interface Voxeet service listener.
     */
    public interface VoxeetServiceListener {

        /**
         * On identify success.
         *
         * @param provider the client provider
         */
        void onIdentifySuccess(VoxeetHttp provider);

        //@Deprecated
        //void onIdentifySuccess(UserTokenResponse response, TokenResponse tokenResponse);

        /**
         * On identify error.
         *
         * @param error the error
         */
        void onIdentifyError(String error);

        /**
         * On network error
         *
         * @param error the error
         */
        void onNetworkError(UnknownHostException error);
    }

    String getBuiltServerUrl() {

        String port = serverPort;
        if (!port.startsWith(":")) port = ":" + port;
        return serverUrl + port;
    }

    public final static class Builder {

        private Application _application;
        private VoxeetSdkTemplate _sdk;
        private String _server_url;
        private String _server_port;
        private String _app_id;
        private String _password;
        private String _tokenAccess;
        private RefreshTokenCallback _tokenRefresh;
        private VoxeetServiceListener _listener;
        private boolean _debug;

        Builder() {

        }

        public Builder setApplication(Application application) {
            this._application = application;
            return this;
        }

        public Builder setVoxeetSDK(VoxeetSdkTemplate sdk) {
            this._sdk = sdk;
            return this;
        }

        Builder setServerUrl(String serverUrl) {
            this._server_url = serverUrl;
            return this;
        }

        public Builder setServerPort(String serverPort) {
            this._server_port = serverPort;
            return this;
        }

        public Builder setAppId(String appId) {
            this._app_id = appId;
            return this;
        }

        public Builder setPassword(String password) {
            this._password = password;
            return this;
        }

        public Builder setVoxeetServiceListener(VoxeetServiceListener listener) {
            this._listener = listener;
            return this;
        }

        public Builder setDebug(boolean debug) {
            this._debug = debug;
            return this;
        }

        public Builder setTokenAccess(String tokenAccess) {
            this._tokenAccess = tokenAccess;
            return this;
        }

        public Builder setTokenRefresh(RefreshTokenCallback tokenRefresh) {
            this._tokenRefresh = tokenRefresh;
            return this;
        }

        public VoxeetHttp build() {
            return new VoxeetHttp(this._application,
                    _sdk,
                    _server_url,
                    _server_port,
                    _app_id,
                    _password,
                    _listener,
                    _tokenAccess,
                    _tokenRefresh,
                    _debug);
        }
    }

    private boolean isTokenUploadAllowed() {
        Object firebase = getFirebaseInstance();
        if (null != firebase) {
            try {
                Method isTokenUploadAllowed = firebase.getClass().getDeclaredMethod("isTokenUploadAllowed");
                Boolean result = (Boolean) isTokenUploadAllowed.invoke(firebase);
                return null != result && result;
            } catch (NoSuchMethodException e) {
                e.printStackTrace();
            } catch (IllegalAccessException e) {
                e.printStackTrace();
            } catch (InvocationTargetException e) {
                e.printStackTrace();
            }
        }
        return false;
    }

    @Nullable
    private String getUploadToken() {
        Object firebase = getFirebaseInstance();
        if (null != firebase) {
            try {
                Method getToken = firebase.getClass().getDeclaredMethod("getToken");
                return (String) getToken.invoke(firebase);
            } catch (NoSuchMethodException e) {
                e.printStackTrace();
            } catch (IllegalAccessException e) {
                e.printStackTrace();
            } catch (InvocationTargetException e) {
                e.printStackTrace();
            }
        }
        return null;
    }

    @Nullable
    private Object getFirebaseInstance() {
        try {
            Class klass = Class.forName("com.voxeet.push.firebase.FirebaseController");
            Method getInstance = klass.getDeclaredMethod("getInstance");
            return getInstance.invoke(null);
        } catch (ClassNotFoundException e) {
            e.printStackTrace();
        } catch (NoSuchMethodException e) {
            e.printStackTrace();
        } catch (IllegalAccessException e) {
            e.printStackTrace();
        } catch (InvocationTargetException e) {
            e.printStackTrace();
        }
        return null;
    }
}