package com.voxeet.sdk.utils;

import android.Manifest;
import android.app.Activity;
import android.content.Context;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.os.Build;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.support.v4.app.ActivityCompat;
import android.support.v4.content.ContextCompat;
import android.util.Log;

import java.util.Collection;

public final class Validate {

    private static final String TAG = Validate.class.getName();

    private static final String NO_INTERNET_PERMISSION_REASON =
            "No internet permissions granted for the app, please add " +
                    "<uses-permission android:name=\"android.permission.INTERNET\" /> " +
                    "to your AndroidManifest.xml.";

    private static final String NO_BLUETOOTH_PERMISSION_REASON =
            "No bluetooth permissions granted for the app, please add " +
                    "<uses-permission android:name=\"android.permission.BLUETOOTH\" /> " +
                    "to your AndroidManifest.xml.";

    private static final String NO_CAMERA_PERMISSION_REASON =
            "No Video permission granted for the app, please add " +
                    "<uses-permission android:name=\"android.permission.CAMERA\" /> " +
                    "to your AndroidManifest.xml and request for the permission to use it if sdk > 21 ";

    private static final String NO_RECORD_AUDIO_PERMISSION_REASON =
            "No Camera permission granted for the app, please add " +
                    "<uses-permission android:name=\"android.permission.RECORD_AUDIO\" /> " +
                    "to your AndroidManifest.xml and request for the permission to use it if sdk > 21 ";

    public static void notNull(Object arg, String name) {
        if (arg == null) {
            throw new NullPointerException("Argument '" + name + "' cannot be null");
        }
    }

    public static <T> void notEmpty(Collection<T> container, String name) {
        if (container.isEmpty()) {
            throw new IllegalArgumentException("Container '" + name + "' cannot be empty");
        }
    }

    public static <T> void containsNoNulls(Collection<T> container, String name) {
        Validate.notNull(container, name);
        for (T item : container) {
            if (item == null) {
                throw new NullPointerException("Container '" + name +
                        "' cannot contain null values");
            }
        }
    }

    public static void containsNoNullOrEmpty(Collection<String> container, String name) {
        Validate.notNull(container, name);
        for (String item : container) {
            if (item == null) {
                throw new NullPointerException("Container '" + name +
                        "' cannot contain null values");
            }
            if (item.length() == 0) {
                throw new IllegalArgumentException("Container '" + name +
                        "' cannot contain empty values");
            }
        }
    }

    public static <T> void notEmptyAndContainsNoNulls(Collection<T> container, String name) {
        Validate.containsNoNulls(container, name);
        Validate.notEmpty(container, name);
    }

    public static void runningOnUiThread() {
        if (!Looper.getMainLooper().equals(Looper.myLooper())) {
            throw new IllegalStateException("This method should be called from the UI thread");
        }
    }

    public static void oneOf(Object arg, String name, Object... values) {
        for (Object value : values) {
            if (value != null) {
                if (value.equals(arg)) {
                    return;
                }
            } else {
                if (arg == null) {
                    return;
                }
            }
        }
        throw new IllegalArgumentException("Argument '" + name +
                "' was not one of the allowed values");
    }

    public static void sdkInitialized(boolean initialized) {
        if (!initialized) {
            throw new IllegalStateException(
                    "The SDK has not been initialized, make sure to call " +
                            "VoxeetClientSdk.sdkInitialize() first.");
        }
    }

    /**
     * Request a set of permission with a given result code as result
     *
     * @param activity a non null activity
     * @param permissions the set of permissions
     * @param result_code the result code to manage
     * @return true if the call could have been made, false otherwise
     */
    public static boolean requestMandatoryPermissions(@NonNull Activity activity,
                                                      @NonNull String[] permissions,
                                                      int result_code) {
        if (android.os.Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            ActivityCompat.requestPermissions(activity,
                    permissions,
                    result_code);
            return true;
        }

        return false;
    }

    public static void hasInternetPermissions(@NonNull Context context) {
        Validate.hasInternetPermissions(context, true);
    }

    public static void hasInternetPermissions(@NonNull Context context, boolean shouldThrow) {
        Validate.notNull(context, "context");

        if (deniedPermission(context, Manifest.permission.INTERNET)) {
            if (shouldThrow) {
                throw new IllegalStateException(NO_INTERNET_PERMISSION_REASON);
            } else {
                Log.w(TAG, NO_INTERNET_PERMISSION_REASON);
            }
        }
    }

    public static boolean hasBluetoothPermissions(@NonNull Context context) {
        return hasPermission(context, Manifest.permission.BLUETOOTH, NO_BLUETOOTH_PERMISSION_REASON);
    }

    public static boolean hasCameraPermissions(@NonNull Context context) {
        return hasPermission(context, Manifest.permission.CAMERA, NO_CAMERA_PERMISSION_REASON);
    }

    public static boolean hasMicrophonePermissions(@NonNull Context context) {
        return hasPermission(context, Manifest.permission.RECORD_AUDIO, NO_RECORD_AUDIO_PERMISSION_REASON);
    }

    private static boolean hasPermission(@NonNull Context context,
                                         @NonNull String permission,
                                         @NonNull String reason) {
        Validate.notNull(context, "context");

        if (!hasPermissionInManifest(context, permission)
                || deniedPermission(context, permission)) {
            Log.e(TAG, reason);

            return false;
        } else
            return true;
    }

    private static boolean deniedPermission(Context context, String permission) {
        final int GRANTED = PackageManager.PERMISSION_GRANTED;

        return ContextCompat.checkSelfPermission(context, permission) != GRANTED
                || context.checkCallingOrSelfPermission(permission) != GRANTED;

    }

    public static boolean hasPermissionInManifest(@NonNull Context context,
                                                  @NonNull String permission_to_check) {
        PackageManager pm = context.getPackageManager();
        try {
            PackageInfo packageInfo = pm.getPackageInfo(context.getPackageName(), PackageManager.GET_PERMISSIONS);
            String[] requestedPermissions = null;
            if (packageInfo != null) {
                requestedPermissions = packageInfo.requestedPermissions;
            }


            if (null != requestedPermissions) {
                for (String permission : requestedPermissions) {
                    if (permission.equals(permission_to_check)) {
                        Log.d(TAG, "hasPermissionInManifest: permission "
                                + permission + " is in manifest");
                        return true;
                    }
                }
            }
        } catch (PackageManager.NameNotFoundException e) {
            e.printStackTrace();
        }

        Log.d(TAG, "hasPermissionInManifest: the permission "
                + permission_to_check + " is not in the manifest");
        return false;
    }
}
