package com.voxeet.sdk.core.abs.promises;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.sdk.core.ConferenceSimpleState;
import com.voxeet.sdk.core.abs.AbstractConferenceSdkObservableProvider;
import com.voxeet.sdk.core.abs.AbstractPromiseable;
import com.voxeet.sdk.core.abs.ConferenceService;
import com.voxeet.sdk.core.abs.information.ConferenceInformation;
import com.voxeet.sdk.core.abs.information.ConferenceState;
import com.voxeet.sdk.core.http.HttpCallback;
import com.voxeet.sdk.core.http.HttpHelper;
import com.voxeet.sdk.core.services.MediaService;
import com.voxeet.sdk.events.error.ConferenceCreatedError;
import com.voxeet.sdk.events.error.HttpException;
import com.voxeet.sdk.events.success.ConferenceCreatingEvent;
import com.voxeet.sdk.json.CreateConferenceParams;
import com.voxeet.sdk.json.internal.MetadataHolder;
import com.voxeet.sdk.json.internal.ParamsHolder;
import com.voxeet.sdk.models.ConferenceResponse;

import org.greenrobot.eventbus.EventBus;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.ErrorPromise;
import eu.codlab.simplepromise.solve.PromiseExec;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;
import retrofit2.Call;
import retrofit2.Response;

public class CreateConferencePromiseable extends AbstractPromiseable<ConferenceResponse> {

    private String conferenceAlias;
    private MetadataHolder metadata;
    private ParamsHolder params = null;

    public CreateConferencePromiseable(@NonNull ConferenceService parent, @NonNull MediaService mediaService, @NonNull AbstractConferenceSdkObservableProvider provider, @Nullable ConferenceInformation information, @NonNull EventBus eventBus,
                                       @Nullable String conferenceAlias,
                                       @Nullable MetadataHolder metadata,
                                       @Nullable ParamsHolder params) {
        super(parent, mediaService, provider, information, eventBus);

        this.conferenceAlias = conferenceAlias;
        this.metadata = metadata;
        this.params = params;
    }

    @NonNull
    @Override
    public Promise<ConferenceResponse> createPromise() {
        final Promise<ConferenceResponse> promise = new Promise<>(new PromiseSolver<ConferenceResponse>() {
            @Override
            public void onCall(@NonNull final Solver<ConferenceResponse> solver) {
                //TODO SET STATE FOR CONFERENCE ALIAS

                //setIsInConference(true);

                getEventBus().post(new ConferenceCreatingEvent(conferenceAlias, null));
                log("Attempting to create mConferene alias:=" + conferenceAlias);

                CreateConferenceParams arg = new CreateConferenceParams()
                        .setMetadataHolder(metadata)
                        .setParamsHolder(params);
                //        .setStats(mEnableStats);

                if (null != conferenceAlias) arg.setConferenceAlias(conferenceAlias);

                //now create the conference and will retry in case of issue...
                internalCreate(true, arg, solver);
            }
        });

        if (getVoxeetSDK().isSocketOpen()) {
            return promise;
        } else {
            return new Promise<>(new PromiseSolver<ConferenceResponse>() {
                @Override
                public void onCall(@NonNull final Solver<ConferenceResponse> solver) {
                    getVoxeetSDK().logCurrentlySelectedUserWithChain().then(new PromiseExec<Boolean, Object>() {
                        @Override
                        public void onCall(@Nullable Boolean result, @NonNull Solver internal_solver) {
                            promise.then(new PromiseExec<ConferenceResponse, Object>() {
                                @Override
                                public void onCall(@Nullable ConferenceResponse result, @NonNull Solver<Object> internal_solver) {
                                    solver.resolve(result);
                                }
                            }).error(new ErrorPromise() {
                                @Override
                                public void onError(@NonNull Throwable error) {
                                    solver.reject(error);
                                }
                            });
                        }
                    }).error(new ErrorPromise() {
                        @Override
                        public void onError(@NonNull Throwable error) {
                            solver.reject(error);
                        }
                    });
                }
            });
        }
    }


    /**
     * Create a conference and retry if neede
     *
     * @param retry  retry or not
     * @param params the params to send
     * @param solver the solver to resolve
     */
    private void internalCreate(final boolean retry, final CreateConferenceParams params, final Solver<ConferenceResponse> solver) {
        Call<ConferenceResponse> observable = getProvider().getCreateConferenceObservable(params);
        HttpHelper.enqueue(observable, new HttpCallback<ConferenceResponse>() {
            @Override
            public void onSuccess(@NonNull ConferenceResponse object, @NonNull Response<ConferenceResponse> response) {
                log("onNext: having conference");
                ConferenceInformation information = createOrSetConferenceWithParams(object.getConfId(), object.getConfAlias());

                information.setConferenceState(ConferenceState.CREATED);
                information.setConferenceType(ConferenceSimpleState.CONFERENCE);

                log("internalCreate onNext: join with := " + object.getConfId() + " " + object.getConfAlias());

                //TODO check for issue in here but for now, setting back id...
                information.getConference().setConferenceId(object.getConfId());
                log("now conference is " + information.getConference().getConferenceId());

                solver.resolve(object);
            }

            @Override
            public void onFailure(@NonNull Throwable e, @Nullable Response<ConferenceResponse> response) {
                HttpException.dumpErrorResponse(response);
                e.printStackTrace();

                if (!retry) {
                    log("internalCreate onFailure: conference creation failed ! but no retry... quit...");
                    setIsInConference(false);

                    closeMedia();

                    getEventBus().post(new ConferenceCreatedError(getParent().handleError(e)));
                    solver.reject(e);
                } else {
                    log("internalCreate onFailure: conference creation failed ! but retry... now...");
                    internalCreate(false, params, solver);
                }
            }
        });
    }
}
