package com.voxeet.sdk.core.abs.promises;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.sdk.core.abs.AbstractConferenceSdkObservableProvider;
import com.voxeet.sdk.core.abs.AbstractPromiseable;
import com.voxeet.sdk.core.abs.ConferenceService;
import com.voxeet.sdk.core.abs.information.ConferenceInformation;
import com.voxeet.sdk.core.abs.information.ConferenceState;
import com.voxeet.sdk.core.http.HttpCallback;
import com.voxeet.sdk.core.http.HttpHelper;
import com.voxeet.sdk.core.services.MediaService;
import com.voxeet.sdk.events.error.ConferenceLeftError;
import com.voxeet.sdk.events.error.HttpException;
import com.voxeet.sdk.events.promises.NotInConferenceException;
import com.voxeet.sdk.events.success.ConferenceLeftSuccessEvent;
import com.voxeet.sdk.models.abs.Conference;

import org.greenrobot.eventbus.EventBus;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;
import okhttp3.ResponseBody;
import retrofit2.Call;
import retrofit2.Response;

public class LeavePromise extends AbstractPromiseable<Boolean> {
    private final static String TAG = LeavePromise.class.getSimpleName();

    public LeavePromise(@NonNull ConferenceService parent,
                        @NonNull MediaService mediaService,
                        @NonNull AbstractConferenceSdkObservableProvider provider,
                        @Nullable ConferenceInformation information,
                        @NonNull EventBus eventBus) {
        super(parent, mediaService, provider, information, eventBus);
    }

    @NonNull
    @Override
    public Promise<Boolean> createPromise() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                try {
                    final ConferenceInformation infos = getInformation();
                    if (null == infos) {
                        throw new NotInConferenceException();
                    }
                    final String conferenceId = infos.getConference().getConferenceId();

                    if (null != infos) infos.setConferenceState(ConferenceState.LEAVING);
                    final String conferenceAlias = null != infos ? infos.getConference().getConferenceAlias() : "";

                    log("Attempting to leave mConference with mConference id " + conferenceId);

                    closeMedia();

                    final Call<ResponseBody> user = getProvider().leaveConference(conferenceId);
                    HttpHelper.enqueue(user, new HttpCallback<ResponseBody>() {
                        @Override
                        public void onSuccess(@NonNull ResponseBody object, @NonNull Response<ResponseBody> response) {
                            removeTimeoutCallbacks();

                            Log.d(TAG, "Conference left successfully");

                            Conference conference = infos.getConference();
                            int leftInConference = infos.getConference().getConferenceUsers().size() - 1;

                            //previously this call would have been in direct conflict with possible parallel create/join
                            //closeMedia();

                            if (response.code() == 200 || response.code() == 404) {
                                getEventBus().post(new ConferenceLeftSuccessEvent(conferenceId, conferenceAlias, leftInConference));
                                solver.resolve(true);
                            } else {
                                getEventBus().post(new ConferenceLeftError(conferenceId, conferenceAlias, response.code() + ""));

                                solver.resolve(false);
                            }
                        }

                        @Override
                        public void onFailure(@NonNull Throwable e, @Nullable Response<ResponseBody> response) {
                            HttpException.dumpErrorResponse(response);

                            e.printStackTrace();
                            removeTimeoutCallbacks();

                            Log.d(TAG, "Something went wrong while leaving the mConference");

                            //previously this call would have been in direct conflict with possible parallel create/join
                            //closeMedia();

                            infos.setConferenceState(ConferenceState.LEFT);
                            getEventBus().post(new ConferenceLeftError(conferenceId, conferenceAlias, getParent().handleError(e)));
                            solver.resolve(false);
                        }
                    });
                } catch (NotInConferenceException e) {
                    e.printStackTrace();
                    solver.reject(e);
                }
            }
        });
    }
}
