package com.voxeet.sdk.core.abs.information;

import android.support.annotation.NonNull;
import android.util.Log;

import com.voxeet.sdk.audio.UserPosition;
import com.voxeet.sdk.core.ConferenceSimpleState;
import com.voxeet.sdk.models.abs.Conference;
import com.voxeet.sdk.models.abs.ConferenceUser;
import com.voxeet.sdk.models.impl.DefaultParticipant;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class ConferenceInformation {

    private static final String TAG = ConferenceInformation.class.getSimpleName();

    @NonNull
    private Conference mConference;

    @NonNull
    private String mConferenceId;

    @NonNull
    private Map<String, String> mUserIdsCached;

    private ConferenceSimpleState mConferenceType;

    @NonNull
    private List<ConferenceUser> mLastInvitationReceived;
    private ConferenceState state = ConferenceState.DEFAULT;

    private HashMap<String, UserPosition> mPositions;

    private boolean mOwnScreenShareStarted;
    private boolean mOwnVideoStarted;
    private ConferenceUserType mUserType;

    private ConferenceInformation() {
        mUserType = ConferenceUserType.NORMAL;
        mConferenceId = "unknown";
        mUserIdsCached = new HashMap<>();
        mLastInvitationReceived = new ArrayList<>();
        mConference = new Conference();
        mConferenceType = ConferenceSimpleState.NONE;
        mOwnVideoStarted = false;
        mOwnScreenShareStarted = false;
        mPositions = new HashMap<>();
    }

    public ConferenceInformation(@NonNull String conferenceId) {
        this();

        mConferenceId = conferenceId;
        mConference.setConferenceId(mConferenceId);
    }

    @NonNull
    public UserPosition getPosition(@NonNull ConferenceUser user) {
        return getPosition(user.getUserId());
    }

    @NonNull
    public UserPosition getPosition(@NonNull String userId) {
        UserPosition p = mPositions.get(userId);
        if(null == p) p = new UserPosition(0,0);
        return p;
    }

    public void setPosition(@NonNull ConferenceUser user, @NonNull UserPosition point) {
        setPosition(user.getUserId(), point);
    }

    public void setPosition(@NonNull String userId, @NonNull UserPosition point) {
        mPositions.put(userId, point);
    }

    @NonNull
    public Conference getConference() {
        return mConference;
    }

    public void setConference(@NonNull Conference conference) {
        mConference = conference;
    }

    public void setConferenceType(@NonNull ConferenceSimpleState type) {
        mConferenceType = type;
    }

    @NonNull
    public ConferenceSimpleState getConferenceType() {
        return mConferenceType;
    }

    @NonNull
    public Map<String, String> getUserIdsCached() {
        return mUserIdsCached;
    }

    @NonNull
    public List<ConferenceUser> getLastInvitationReceived() {
        return mLastInvitationReceived;
    }

    public void setConferenceState(@NonNull ConferenceState state) {
        this.state = state;
    }

    @NonNull
    public ConferenceState getState() {
        return state;
    }

    @NonNull
    public ConferenceState getConferenceState() {
        return state;
    }

    /**
     * Set if the user asked for video in this conference state
     * Helpful to deal with new privacy features in Android P+
     *
     * @param ownVideoStarted a boolean indicating the proper state
     */
    public void setOwnVideoStarted(boolean ownVideoStarted) {
        Log.d(TAG, "setOwnVideoStarted: " + ownVideoStarted);
        mOwnVideoStarted = ownVideoStarted;
    }

    /**
     * Return if the user asked for video in this conference state
     * Helpful to deal with new privacy features in Android P+
     *
     * @return a boolean indicating the proper state
     */
    public boolean isOwnVideoStarted() {
        Log.d(TAG, "isOwnVideoStarted: " + mOwnVideoStarted);
        return mOwnVideoStarted;
    }

    public void setConferenceUserType(@NonNull ConferenceUserType userType) {
        mUserType = userType;
    }

    public ConferenceUserType getConferenceUserType() {
        return mUserType;
    }

    public boolean isBroadcaster() {
        return ConferenceUserType.BROADCASTER.equals(mUserType);
    }

    public boolean isListener() {
        return ConferenceUserType.LISTENER.equals(mUserType);
    }

    /**
     * Must be done on init
     * <p>
     * Any call afterward to this won't only be used when the "join" will be reused
     * <p>
     * It is should usefull if a user will switch from any mode to the other with proper leave -> join
     *
     * @param userType the state to use
     */
    public void setLocalUserType(ConferenceUserType userType) {
        mUserType = userType;
    }

    public void setScreenShareOn(boolean enable) {
        mOwnScreenShareStarted = enable;
    }

    public boolean isScreenShareOn() {
        return mOwnScreenShareStarted;
    }


    public void participantsToConferenceUsers(List<DefaultParticipant> participants) {
        if (null != participants) {
            Map<String, String> cache = getUserIdsCached();
            for (DefaultParticipant participant : participants) {
                cache.put(participant.getExternalId(), participant.getExternalId());
            }

            Conference conference = getConference();
            for (DefaultParticipant participant : participants) {
                conference.updateParticipantToConferenceUsers(participant);
            }
        }
    }
}
