package com.voxeet.sdk.core.abs;

import android.content.Context;
import android.content.Intent;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.android.media.MediaEngine;
import com.voxeet.android.media.MediaEngineException;
import com.voxeet.android.media.MediaSDK;
import com.voxeet.android.media.MediaStream;
import com.voxeet.android.media.SdpCandidate;
import com.voxeet.android.media.peer.PendingPeerCallback;
import com.voxeet.android.media.peer.SdpDescription;
import com.voxeet.android.media.peer.SdpMessage;
import com.voxeet.audio.AudioRoute;
import com.voxeet.sdk.audio.UserPosition;
import com.voxeet.sdk.core.ConferenceSimpleState;
import com.voxeet.sdk.core.VoxeetSdkTemplate;
import com.voxeet.sdk.core.abs.information.ConferenceInformation;
import com.voxeet.sdk.core.abs.information.ConferenceInformationHolder;
import com.voxeet.sdk.core.abs.information.ConferenceState;
import com.voxeet.sdk.core.abs.information.ConferenceUserType;
import com.voxeet.sdk.core.abs.promises.CreateConferencePromiseable;
import com.voxeet.sdk.core.abs.promises.DeclinePromise;
import com.voxeet.sdk.core.abs.promises.GetConferenceHistoryPromiseable;
import com.voxeet.sdk.core.abs.promises.GetConferenceStatus;
import com.voxeet.sdk.core.abs.promises.GetConferenceStatusPromiseable;
import com.voxeet.sdk.core.abs.promises.InvitePromise;
import com.voxeet.sdk.core.abs.promises.JoinPromise;
import com.voxeet.sdk.core.abs.promises.LeavePromise;
import com.voxeet.sdk.core.abs.promises.LogoutPromise;
import com.voxeet.sdk.core.abs.promises.ReplayPromise;
import com.voxeet.sdk.core.abs.promises.SendBroadcastMessagePromise;
import com.voxeet.sdk.core.abs.promises.StartRecordingPromiseable;
import com.voxeet.sdk.core.abs.promises.StartScreensharePromise;
import com.voxeet.sdk.core.abs.promises.StartVideoPromise;
import com.voxeet.sdk.core.abs.promises.StopRecordingPromiseable;
import com.voxeet.sdk.core.abs.promises.StopScreenSharePromise;
import com.voxeet.sdk.core.abs.promises.StopVideoPromise;
import com.voxeet.sdk.core.abs.promises.SubscribeConferenceEventPromiseable;
import com.voxeet.sdk.core.abs.promises.SubscribeForCallStartPromiseable;
import com.voxeet.sdk.core.abs.promises.UnsubscribeConferenceEventPromiseable;
import com.voxeet.sdk.core.abs.promises.UnsubscribeForCallStartPromiseable;
import com.voxeet.sdk.core.http.HttpCallback;
import com.voxeet.sdk.core.http.HttpHelper;
import com.voxeet.sdk.core.impl.ConferenceSdkObservableProvider;
import com.voxeet.sdk.core.network.ISdkConferenceRService;
import com.voxeet.sdk.core.preferences.VoxeetPreferences;
import com.voxeet.sdk.core.services.AudioService;
import com.voxeet.sdk.core.services.MediaService;
import com.voxeet.sdk.core.services.TimeoutRunnable;
import com.voxeet.sdk.core.services.builders.ConferenceCreateInformation;
import com.voxeet.sdk.core.services.builders.ConferenceJoinInformation;
import com.voxeet.sdk.core.services.holder.ServiceProviderHolder;
import com.voxeet.sdk.events.error.ConferenceCreatedError;
import com.voxeet.sdk.events.error.ConferenceJoinedError;
import com.voxeet.sdk.events.error.HttpException;
import com.voxeet.sdk.events.error.ParticipantAddedErrorEvent;
import com.voxeet.sdk.events.error.PermissionRefusedEvent;
import com.voxeet.sdk.events.promises.InConferenceException;
import com.voxeet.sdk.events.promises.NotInConferenceException;
import com.voxeet.sdk.events.promises.PromiseConferenceJoinedErrorException;
import com.voxeet.sdk.events.promises.PromiseParticipantAddedErrorEventException;
import com.voxeet.sdk.events.success.ConferenceCreationSuccess;
import com.voxeet.sdk.events.success.ConferenceDestroyedPushEvent;
import com.voxeet.sdk.events.success.ConferenceEndedEvent;
import com.voxeet.sdk.events.success.ConferenceJoinedSuccessEvent;
import com.voxeet.sdk.events.success.ConferencePreJoinedEvent;
import com.voxeet.sdk.events.success.ConferenceRefreshedEvent;
import com.voxeet.sdk.events.success.ConferenceStatsEvent;
import com.voxeet.sdk.events.success.ConferenceUpdatedEvent;
import com.voxeet.sdk.events.success.ConferenceUserAddedEvent;
import com.voxeet.sdk.events.success.ConferenceUserCallDeclinedEvent;
import com.voxeet.sdk.events.success.ConferenceUserJoinedEvent;
import com.voxeet.sdk.events.success.ConferenceUserLeftEvent;
import com.voxeet.sdk.events.success.ConferenceUserQualityUpdatedEvent;
import com.voxeet.sdk.events.success.ConferenceUserUpdatedEvent;
import com.voxeet.sdk.events.success.ConferenceUsersInvitedEvent;
import com.voxeet.sdk.events.success.DeclineConferenceResultEvent;
import com.voxeet.sdk.events.success.GetConferenceHistoryEvent;
import com.voxeet.sdk.events.success.GetConferenceStatusEvent;
import com.voxeet.sdk.events.success.IncomingCallEvent;
import com.voxeet.sdk.events.success.InvitationReceived;
import com.voxeet.sdk.events.success.OfferCreatedEvent;
import com.voxeet.sdk.events.success.OwnConferenceStartedEvent;
import com.voxeet.sdk.events.success.ParticipantUpdatedEvent;
import com.voxeet.sdk.events.success.QualityIndicators;
import com.voxeet.sdk.events.success.QualityUpdatedEvent;
import com.voxeet.sdk.events.success.RecordingStatusUpdate;
import com.voxeet.sdk.events.success.RenegociationEndedEvent;
import com.voxeet.sdk.events.success.RenegociationUpdate;
import com.voxeet.sdk.events.success.ResumeConference;
import com.voxeet.sdk.events.success.ScreenStreamAddedEvent;
import com.voxeet.sdk.events.success.ScreenStreamRemovedEvent;
import com.voxeet.sdk.events.success.SocketConnectEvent;
import com.voxeet.sdk.events.success.StartVideoAnswerEvent;
import com.voxeet.sdk.exceptions.ExceptionManager;
import com.voxeet.sdk.factories.VoxeetIntentFactory;
import com.voxeet.sdk.json.ConferenceStats;
import com.voxeet.sdk.json.ConferenceUserAdded;
import com.voxeet.sdk.json.InvitationReceivedEvent;
import com.voxeet.sdk.json.JoinParameters;
import com.voxeet.sdk.json.OfferCreated;
import com.voxeet.sdk.json.RecordingStatusUpdateEvent;
import com.voxeet.sdk.json.UserInfo;
import com.voxeet.sdk.json.internal.MetadataHolder;
import com.voxeet.sdk.json.internal.ParamsHolder;
import com.voxeet.sdk.models.CandidatesPush;
import com.voxeet.sdk.models.ConferenceQuality;
import com.voxeet.sdk.models.ConferenceResponse;
import com.voxeet.sdk.models.ConferenceType;
import com.voxeet.sdk.models.ConferenceUserStatus;
import com.voxeet.sdk.models.NormalConferenceResponse;
import com.voxeet.sdk.models.OfferCandidate;
import com.voxeet.sdk.models.OfferDescription;
import com.voxeet.sdk.models.RecordingStatus;
import com.voxeet.sdk.models.abs.Conference;
import com.voxeet.sdk.models.abs.ConferenceUser;
import com.voxeet.sdk.models.impl.DefaultInvitation;
import com.voxeet.sdk.networking.DeviceType;
import com.voxeet.sdk.utils.ConferenceUtils;
import com.voxeet.sdk.utils.Validate;

import org.greenrobot.eventbus.EventBus;
import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;
import org.webrtc.CameraVideoCapturer;
import org.webrtc.EglBaseMethods;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.ReentrantLock;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.ErrorPromise;
import eu.codlab.simplepromise.solve.PromiseExec;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;
import okhttp3.ResponseBody;
import retrofit2.Call;
import retrofit2.Callback;
import retrofit2.Response;

/**
 * Conference SDK Service
 * <p>
 * Warning : always implements the error for each promises you may call to start resolve them
 * You can also use the execute method but in case of Exception, you will trigger errors...
 */
public class ConferenceService extends com.voxeet.sdk.core.AbstractVoxeetService<ISdkConferenceRService> {

    private final static String TAG = ConferenceService.class.getSimpleName();
    private final VoxeetSdkTemplate mInstance;
    private VoxeetSdkTemplate mSDK;
    private AbstractConferenceSdkObservableProvider<ISdkConferenceRService> mConferenceObservableProvider;

    private EventBus mEventBus;
    private boolean isRecording = false;
    private boolean mInConference = false;
    private long mTimeOutTimer = -1;
    private HashMap<String, MediaStream> mapOfStreams = new HashMap<>();
    private HashMap<String, MediaStream> mapOfScreenShareStreams = new HashMap<>();
    private ReentrantLock joinLock = new ReentrantLock();

    private boolean isDefaultOnSpeaker;

    private String mConferenceId = null;
    private ConferenceInformationHolder mConferenceInformationHolder = new ConferenceInformationHolder();

    private Context mContext;

    private @Nullable
    TimeoutRunnable timeoutRunnable = null;

    private MediaEngine.StreamListener mediaStreamListener = new MediaEngine.StreamListener() {
        @Override
        public void onStreamAdded(@NonNull final String peer, @NonNull final MediaStream stream) {
            Log.d(TAG, "onStreamAdded: stream for peer " + peer);
            postOnMainThread(new Runnable() {
                @Override
                public void run() {
                    Log.d(TAG, "New mConference user joined with id: " + peer + " checking... (ours is " + VoxeetPreferences.id());

                    mapOfStreams.put(peer, stream);
                    ConferenceUser user = updateConferenceParticipants(peer, ConferenceUserStatus.ON_AIR);


                    if (user != null) {
                        if (!peer.equalsIgnoreCase(VoxeetPreferences.id()) && mTimeOutTimer != -1) {
                            removeTimeoutCallbacks();
                        }

                        //update the current conference state specificly from the users point of view
                        updateConferenceFromUsers();
                        mEventBus.post(new ConferenceUserJoinedEvent(user, getSafeConferenceId(), getConference(), stream));
                        mEventBus.post(new ConferenceUserUpdatedEvent(user, getSafeConferenceId(), getConference(), stream));
                    } else {
                        Log.d(TAG, "run: unknown user in stream added");
                    }
                }
            });
        }

        @Override
        public void onStreamUpdated(@NonNull final String peer, @NonNull final MediaStream stream) {
            Log.d(TAG, "onStreamUpdated: screen for peer " + peer);
            postOnMainThread(new Runnable() {
                @Override
                public void run() {
                    if (stream.isScreenShare()) mapOfScreenShareStreams.put(peer, stream);
                    else mapOfStreams.put(peer, stream);

                    ConferenceUser user = findUserById(peer);
                    if (user != null) {
                        //update the current conference state specificly from the users point of view
                        updateConferenceFromUsers();
                        mEventBus.post(new ConferenceUserUpdatedEvent(user, getSafeConferenceId(), getConference(), stream));
                    } else {
                        Log.d(TAG, "run: unknown user in stream updated");
                    }
                }
            });
        }

        @Override
        public void onStreamRemoved(@NonNull final String peer) {
            Log.d(TAG, "onStreamRemoved: OnStreamRemoved");
            postOnMainThread(new Runnable() {
                @Override
                public void run() {
                    onUserLeft(peer);
                }
            });
        }

        @Override
        public void onScreenStreamAdded(@NonNull final String peer, @NonNull final MediaStream stream) {
            Log.d(TAG, "onScreenStreamAdded: screen for peer " + peer);

            mapOfScreenShareStreams.put(peer, stream);
            mEventBus.post(new ScreenStreamAddedEvent(peer, stream));
        }

        @Override
        public void onScreenStreamRemoved(@NonNull String peer) {
            mapOfScreenShareStreams.remove(peer);
            mEventBus.post(new ScreenStreamRemovedEvent(peer));
        }

        @Override
        public void onShutdown() {

        }

        @Override
        public void onIceCandidateDiscovered(String peer, SdpCandidate[] candidates) {

            if (null == getConferenceId()) return;

            Call<ResponseBody> call = mConferenceObservableProvider.candidates(getConferenceId(),
                    peer,
                    new CandidatesPush(candidates));

            call.enqueue(new Callback<ResponseBody>() {
                @Override
                public void onResponse(Call<ResponseBody> call, Response<ResponseBody> response) {
                    Log.d(TAG, "onResponse: Candidates sent ");
                }

                @Override
                public void onFailure(Call<ResponseBody> call, Throwable t) {
                    t.printStackTrace();
                }
            });
        }
    };

    private CameraVideoCapturer.CameraEventsHandler cameraEventsHandler = new CameraVideoCapturer.CameraEventsHandler() {
        @Override
        public void onCameraError(String s) {
            Log.d(TAG, "onCameraError: error...");

            stopVideo().then(new PromiseExec<Boolean, Object>() {
                @Override
                public void onCall(@Nullable Boolean result, @NonNull Solver<Object> solver) {
                    Log.d(TAG, "onCall: stopped video after camera issue " + result);
                }
            }).error(new ErrorPromise() {
                @Override
                public void onError(@NonNull Throwable error) {
                    error.printStackTrace();
                }
            });
        }

        @Override
        public void onCameraDisconnected() {
            Log.d(TAG, "onCameraDisconnected");
        }

        @Override
        public void onCameraFreezed(String s) {
            Log.d(TAG, "onCameraFreezed: " + s);
        }

        @Override
        public void onCameraOpening(String s) {
            Log.d(TAG, "onCameraOpening: " + s);
        }

        @Override
        public void onFirstFrameAvailable() {
            Log.d(TAG, "onFirstFrameAvailable");
        }

        @Override
        public void onCameraClosed() {
            Log.d(TAG, "onCameraClosed");
        }
    };

    private boolean isDefaultVideo = false;
    private boolean isDefaultMute = false;
    private boolean isICERestartEnabled = false;

    public ConferenceService(VoxeetSdkTemplate instance, long timeout) {
        super(instance, new ServiceProviderHolder.Builder<ISdkConferenceRService>()
                .setRetrofit(instance.getRetrofit())
                .setService(ISdkConferenceRService.class)
                .setEventBus(instance.getEventBus())
                .setClient(instance.getClient())
                .build());

        mInstance = instance;

        mConferenceObservableProvider = new ConferenceSdkObservableProvider();
        mConferenceObservableProvider.setRetrofitInstantiatedProvider(getService());
        mSDK = instance;
        mTimeOutTimer = timeout;
        mEventBus = EventBus.getDefault();
        mContext = instance.getApplicationContext();
        setDefaultBuiltInSpeaker(true);


        register();
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Public management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    /**
     * Mute or unmute the current user
     *
     * @param mute the new unmute/mute state
     * @return if the state can be changed
     * @deprecated Please use {@link #mute(boolean)} ()} instead.
     */
    @Deprecated
    public boolean muteConference(boolean mute) {
        return mute(mute);
    }

    /**
     * Mute or unmute the current user
     *
     * @param mute the new unmute/mute state
     * @return if the state can be changed
     */
    public boolean mute(boolean mute) {
        if (getMediaService().hasMedia()) {
            if (!mute && getMediaService().getMedia().isMuted()) {
                if (!Validate.hasMicrophonePermissions(context)) {
                    getEventBus().post(new PermissionRefusedEvent(PermissionRefusedEvent.Permission.MICROPHONE));
                    return false;
                } else {
                    getMedia().unMute();
                }
            } else if (mute) {
                getMedia().mute();
            }
        }
        return true;
    }

    public boolean isMuted() {
        return getMedia() != null && getMedia().isMuted();
    }

    public boolean isUserMuted(String userId) {
        ConferenceUser user = findUserById(userId);
        return user != null && user.isMuted();
    }

    public void setListenerMode(boolean isListener) {
        if (getMediaService().hasMedia()) {
            MediaSDK media = getMediaService().getMedia();
            if (media.isMuted()) {
                media.unMute();
            } else {
                media.mute();
            }
            //TODO set music?
        }
    }

    public void register() {
        registerEventBus();
    }

    public void unregister() {
        unRegisterEventBus();
    }

    @Deprecated
    public void setDefaultCamera(String cameraName) {
        getMediaService().setDefaultCamera(cameraName);
    }

    public void toggleVideo() {
        Promise<Boolean> promise = isVideoOn() ? stopVideo() : startVideo();
        promise.then(new PromiseExec<Boolean, Object>() {
            @Override
            public void onCall(@Nullable Boolean result, @NonNull Solver<Object> solver) {
                Log.d(TAG, "onSuccess: toggleVideo " + result);
            }
        }).error(ConferenceServiceHelper.manageError());
    }

    public void toggleScreenShare() {
        if (isScreenShareOn()) {
            stopScreenShare().then(new PromiseExec<Boolean, Object>() {
                @Override
                public void onCall(@Nullable Boolean result, @NonNull Solver<Object> solver) {
                    Log.d(TAG, "onSuccess: toggleScreenShare " + result);
                }
            }).error(ConferenceServiceHelper.manageError());
        } else {
            getVoxeetSDK().getScreenShareService().sendRequestStartScreenShare();
        }
    }

    public String getCurrentConferenceId() {
        return mConferenceId;
    }

    public int getConferenceRoomSize() {
        ConferenceInformation information = getCurrentConferenceInformation();
        if (null != information) {
            return information.getConference().getConferenceRoomSize();
        }
        return 0;
    }

    public String currentSpeaker() {
        com.voxeet.sdk.models.abs.Conference conference = getConference();
        if (!hasMedia() || conference == null) {
            return VoxeetPreferences.id();
        } else {
            String currentSpeaker = null;
            for (ConferenceUser user : conference.getConferenceUsers()) {
                if (user.getUserId() != null
                        && !user.getUserId().equals(VoxeetPreferences.id())
                        && ConferenceUserStatus.ON_AIR.equals(user.getConferenceStatus())) {
                    double peerVuMeter = getMedia().getPeerVuMeter(user.getUserId());
                    if (currentSpeaker == null || (peerVuMeter > 0.001 && getMedia().getPeerVuMeter(currentSpeaker) < peerVuMeter))
                        currentSpeaker = user.getUserId();
                }
            }
            return currentSpeaker;
        }
    }

    @NonNull
    private String currentUserOrEmpty() {
        String currentUserId = VoxeetPreferences.id();
        return null != currentUserId ? currentUserId : "";
    }

    public double getSdkPeerVuMeter(@Nullable String peerId) {
        Validate.runningOnUiThread();
        return hasMedia() && null != peerId ? getMedia().getPeerVuMeter(peerId) : 0;
    }

    public ConferenceUser findUserById(final String userId) {
        Conference conference = getConference();
        return null != conference ? conference.findUserById(userId) : null;
    }

    public String getAliasId() {
        Conference conference = getConference();
        return null != conference ? conference.getConferenceAlias() : null;
    }

    @Nullable
    public String getConferenceId() {
        return mConferenceId;
    }

    @NonNull
    private String getSafeConferenceId() {
        String conferenceId = getConferenceId();
        if (null == conferenceId) conferenceId = "";
        return conferenceId;
    }

    @Deprecated
    public List<AudioRoute> getAvailableRoutes() {
        return getAudioService().getAvailableRoutes();
    }

    @Deprecated
    public AudioRoute currentRoute() {
        return getAudioService().currentRoute();
    }

    @Nullable
    public EglBaseMethods.Context getEglContext() {
        return getMediaService().getEglContext();
    }

    public long getTimeout() {
        return mTimeOutTimer;
    }

    public void toggleRecording() {
        Promise<Boolean> promise = isRecording ? stopRecording() : startRecording();
        promise.then(new PromiseExec<Boolean, Object>() {
            @Override
            public void onCall(@Nullable Boolean result, @NonNull Solver<Object> solver) {
                Log.d(TAG, "onSuccess: toggle done " + result);
            }
        }).error(new ErrorPromise() {
            @Override
            public void onError(Throwable error) {
                error.printStackTrace();
            }
        });
    }

    public boolean muteUser(@NonNull String userId, boolean shouldMute) {
        Validate.runningOnUiThread();

        ConferenceUser user = findUserById(userId);
        if (user != null) {
            user.setMuted(shouldMute);
            getMedia().changePeerGain(userId, shouldMute ? ConferenceUtils.MUTE_FACTOR : ConferenceUtils.UNMUTE_FACTOR);
        }
        return user != null;
    }

    @Deprecated
    public boolean isVideoOn() {
        ConferenceInformation information = getCurrentConferenceInformation();
        return null != information && information.isOwnVideoStarted();
    }

    @Deprecated
    public boolean isScreenShareOn() {
        ConferenceInformation information = getCurrentConferenceInformation();
        return null != information && information.isScreenShareOn();
    }

    @Deprecated
    public boolean setDefaultVideo(boolean default_state) {
        //isDefaultVideo = default_state;
        //return true;
        return false;
    }

    public boolean setDefaultMute(boolean default_state) {
        isDefaultMute = default_state;
        return true;
    }

    public boolean setDefaultBuiltInSpeaker(boolean default_state) {
        isDefaultOnSpeaker = default_state;
        VoxeetPreferences.setDefaultBuiltInSpeakerOn(default_state);
        return true;
    }

    public boolean setTimeOut(long timeout) {
        mTimeOutTimer = timeout;
        return true;
    }

    public boolean setUserPosition(final String userId, final double angle, final double distance) {
        if (hasMedia()) {
            //TODO possible improvement here in case of recovery
            ConferenceInformation conference = getCurrentConferenceInformation();
            if (null != conference) {
                UserPosition position = conference.getPosition(userId);
                position.angle = angle;
                position.distance = distance;
                conference.setPosition(userId, position);
            }
            getMedia().changePeerPosition(userId, angle, distance);
        }
        return hasMedia();
    }

    /**
     * Get the current conference type
     *
     * @return a ConferenceType, ConferenceType.NONE is the default value
     */
    @NonNull
    public ConferenceSimpleState getConferenceType() {
        ConferenceInformation information = getCurrentConferenceInformation();
        if (null == information || null == mConferenceId) return ConferenceSimpleState.NONE;
        return information.getConferenceType();
    }

    /**
     * Get the current mConference object
     *
     * @return a nullable object
     */
    @Nullable
    public Conference getConference() {
        ConferenceInformation information = getCurrentConferenceInformation();
        if (null == information) return null;

        return information.getConference();
    }

    @NonNull
    private Conference getOrSetConference(@NonNull String conferenceId) {
        ConferenceInformation information = getConferenceInformation(conferenceId);
        if (null != information) return information.getConference();
        Log.d(TAG, "getOrSetConference: INVALID CALL FOR GET OR SET CONFERENCE WITH " + conferenceId);
        return new Conference();
    }

    /**
     * Get the current state of joining
     * in conjunction of the getConference() it is easy to check the current state :
     * <p>
     * mInConference() + getConference() non null = mConference joined and on air
     * mInConference() + getConference() null = joining mConference with the answer from server for now
     * !mInConference() = not currently joining, joined mConference (or left)
     *
     * @return if in mConference or "attempting" to join
     */
    public boolean isInConference() {
        return mInConference;
    }

    protected void setIsInConference(boolean status) {
        mInConference = status;
    }

    public double getPeerVuMeter(@Nullable String peerId) {
        Validate.runningOnUiThread();

        if (hasMedia() && null != peerId) return getMedia().getPeerVuMeter(peerId);
        return 0.;
    }

    @NonNull
    public List<ConferenceUser> getConferenceUsers() {
        Conference conference = getConference();
        if (null == conference) {
            Log.d(TAG, "getConferenceUsers: returning a new instance :: NOT IN A CONFERENCE");
            return new ArrayList<>();
        }

        return conference.getConferenceUsers();
    }

    @NonNull
    public List<ConferenceUser> getLastInvitationUsers() {
        ConferenceInformation information = getCurrentConferenceInformation();
        if (null == information) return new ArrayList<>();
        return information.getLastInvitationReceived();
    }

    public boolean isLive() {
        return null != getConference();
    }

    public boolean isListenerMode() {
        ConferenceInformation information = getCurrentConferenceInformation();
        return null != information && information.isListener();
    }

    @Nullable
    public ConferenceUser getUser(final String userId) {
        com.voxeet.sdk.models.abs.Conference conference = getConference();
        if (null != conference) {
            return ConferenceUtils.findUserById(userId, conference.getConferenceUsers());
        }
        return null;
    }

    /**
     * Deprecated since it will possibly be removed in the near future
     *
     * @return the map of streams available
     */
    @Deprecated
    @NonNull
    public HashMap<String, MediaStream> getMapOfStreams() {
        return mapOfStreams;
    }

    /**
     * Deprecated since it will possibly be removed in the near future
     *
     * @return the map of screenshare streams available
     */
    @Deprecated
    @NonNull
    public HashMap<String, MediaStream> getMapOfScreenShareStreams() {
        return mapOfScreenShareStreams;
    }

    /**
     * Give the ability to cancel the timeout in some cases
     */
    public ConferenceService cancelTimeout() {
        removeTimeoutCallbacks();

        return this;
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Public Promises management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    /**
     * resolve -> the result event
     * reject -> network error
     *
     * @param conferenceId the conference id to decline
     * @return a promise to resolve
     */
    public Promise<DeclineConferenceResultEvent> decline(final String conferenceId) {
        return new DeclinePromise(this,
                getMediaService(),
                mConferenceObservableProvider,
                getConferenceInformation(conferenceId),
                getEventBus(),
                conferenceId).createPromise();
    }

    public void sendRequestStartScreenShare() {
        getVoxeetSDK().getScreenShareService().sendRequestStartScreenShare();
    }

    public void onUserCanceledScreenShare() {
        //call before any specific code so stopScreenCapturer() should do nothing /!\
        if (hasMedia()) {
            getMedia().stopScreenCapturer();
        }
        ConferenceInformation information = getCurrentConferenceInformation();
        if (null != information) {
            information.setScreenShareOn(false);
        }
    }

    public Promise<Boolean> startScreenShare(@NonNull final Intent intent,
                                             final int width,
                                             final int height) {
        return new StartScreensharePromise(this,
                getMediaService(),
                mConferenceObservableProvider,
                getCurrentConferenceInformation(),
                getEventBus(),
                intent,
                width,
                height).createPromise();
    }

    /**
     * resolve -> StopVideoAnswerEvent
     * reject -> MediaException
     * reject -> NotInConferenceException
     */
    public Promise<Boolean> stopScreenShare() {
        return new StopScreenSharePromise(this,
                getMediaService(),
                mConferenceObservableProvider,
                getCurrentConferenceInformation(),
                getEventBus()).createPromise();
    }

    /**
     * Start the video
     * resolve -> StartVideoAnswerEvent
     * reject -> PromisePermissionRefusedEventException
     * reject -> MediaException
     * <p>
     * If the app does not have the permission, will fire a PermissionRefusedEvent
     */
    public Promise<Boolean> startVideo() {
        return new StartVideoPromise(this,
                getMediaService(),
                mConferenceObservableProvider,
                getCurrentConferenceInformation(),
                getEventBus()).createPromise();
    }

    /**
     * resolve -> StopVideoAnswerEvent
     * reject -> MediaException
     * reject -> NotInConferenceException
     */
    public Promise<Boolean> stopVideo() {
        return new StopVideoPromise(this,
                getMediaService(),
                mConferenceObservableProvider,
                getCurrentConferenceInformation(),
                getEventBus()).createPromise();
    }

    /**
     * Create a video answer and post it
     * <p>
     * resolve -> true -> everything went fine
     * resolve -> false -> on error occured
     * reject -> media exception
     *
     * @param userId
     * @param offerDescription
     * @param offerCandidates
     * @return
     */
    protected Promise<Boolean> createVideoAnswer(@NonNull final String userId, @Nullable final OfferDescription offerDescription, final List<OfferCandidate> offerCandidates) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                String sdp = "";
                String type = "";
                if (null != offerDescription) {
                    sdp = offerDescription.sdp;
                    type = offerDescription.type;
                }

                try {
                    if (!hasMedia()) {
                        throw new MediaEngineException("media is null");
                    }
                } catch (MediaEngineException exception) {
                    solver.reject(exception);
                }

                if (null == offerDescription) {
                    stopVideo().execute();
                    Log.d(TAG, "onCall: unable to start video, are you a listener?");
                    solver.resolve(false);
                    return;
                }


                SdpDescription description = new SdpDescription(type, sdp);

                List<SdpCandidate> candidates = new ArrayList<>();
                if (offerCandidates != null) {
                    for (OfferCandidate candidate : offerCandidates) {
                        candidates.add(new SdpCandidate(candidate.getMid(), Integer.parseInt(candidate.getmLine()), candidate.getSdp()));
                    }
                }

                try {
                    getMedia().createAnswerForPeer(userId,
                            description.ssrc,
                            description,
                            candidates,
                            VoxeetPreferences.id().equalsIgnoreCase(userId),
                            new PendingPeerCallback() {
                                @Override
                                public void onMessage(@Nullable SdpMessage message) {
                                    Log.d(TAG, "onMessage: having a message " + message);
                                    answer(userId, message)
                                            .then(new PromiseExec<Integer, Object>() {
                                                @Override
                                                public void onCall(@Nullable Integer result, @NonNull Solver<Object> internal_solver) {
                                                    Log.d(TAG, "onSuccess: " + result);

                                                    //resolve true value : all clear
                                                    solver.resolve(true);
                                                }
                                            })
                                            .error(new ErrorPromise() {
                                                @Override
                                                public void onError(Throwable error) {
                                                    if (error instanceof PromiseParticipantAddedErrorEventException)
                                                        mEventBus.post(((PromiseParticipantAddedErrorEventException) error).getEvent());
                                                    else
                                                        error.printStackTrace();

                                                    //resolve false value : something happened
                                                    solver.resolve(false);
                                                }
                                            });
                                }
                            });
                } catch (MediaEngineException e) {
                    e.printStackTrace();
                    solver.reject(e);
                }
            }
        });
    }

    /**
     * resolve -> true
     * resolve -> false
     * reject -> exception
     *
     * @return a promise to resolve
     * @deprecated use {@link #create(ConferenceCreateInformation)} instead.
     */
    @Deprecated
    public Promise<ConferenceResponse> create() {
        return create(null, null, null);
    }

    /**
     * resolve -> true
     * resolve -> false
     * reject -> exception
     *
     * @param metadata the MetadataHolder
     * @return a promise to resolve
     * @deprecated use {@link #create(ConferenceCreateInformation)} instead.
     */
    @Deprecated
    public Promise<ConferenceResponse> create(@Nullable MetadataHolder metadata,
                                              @Nullable ParamsHolder params) {
        return create(null, metadata, params);
    }


    /**
     * resolve -> true
     * resolve -> false
     * reject -> exception
     *
     * @param conferenceAlias create a given conference then join it
     * @return a promise to resolve
     */
    public Promise<ConferenceResponse> create(@Nullable final String conferenceAlias) {
        return create(new ConferenceCreateInformation.Builder()
                .setConferenceAlias(conferenceAlias).build()
        );
    }

    /**
     * resolve -> true
     * resolve -> false
     * reject -> exception
     *
     * @param conferenceAlias create a given conference then join it
     * @param metadata
     * @param paramsholder
     * @return a promise to resolve
     * @deprecated use {@link #create(ConferenceCreateInformation)} instead.
     */
    @Deprecated
    public Promise<ConferenceResponse> create(@Nullable String conferenceAlias,
                                              @Nullable MetadataHolder metadata,
                                              @Nullable ParamsHolder paramsholder) {
        return create(new ConferenceCreateInformation.Builder()
                .setConferenceAlias(conferenceAlias)
                .setMetadataHolder(metadata)
                .setParamsHolder(paramsholder)
                .build()
        );
    }

    /**
     * Create a conference based on information from the built holder
     *
     * @param conferenceCreateInformation the information holder where id, params and metadata can be passed into
     * @return a conference creation information holder
     */
    public Promise<ConferenceResponse> create(@NonNull ConferenceCreateInformation conferenceCreateInformation) {
        return new CreateConferencePromiseable(this,
                getMediaService(),
                mConferenceObservableProvider,
                getCurrentConferenceInformation(),
                getEventBus(),
                conferenceCreateInformation.getConferenceAlias(),
                conferenceCreateInformation.getMetadataHolder(),
                conferenceCreateInformation.getParamsHolder()
        ).createPromise();
    }

    /**
     * Join a conference with a conference
     * <p>
     * Note that the conferenceId needs to be a Voxeet conferenceId,
     *
     * @param conferenceId id of the conference to join as a listener
     * @return a promise to resolve
     */
    public Promise<Boolean> listenConference(@NonNull String conferenceId) {
        return join(new ConferenceJoinInformation.Builder(conferenceId)
                .setConferenceUserType(ConferenceUserType.LISTENER)
                .build()
        );
    }

    /**
     * Join a conference with a conference
     * <p>
     * Note that the conferenceId needs to be a Voxeet conferenceId,
     *
     * @param conferenceId id of the conference to join as a broadcaster
     * @return a promise to resolve
     */
    public Promise<Boolean> broadcastConference(@NonNull String conferenceId) {
        return join(new ConferenceJoinInformation.Builder(conferenceId)
                .setConferenceUserType(ConferenceUserType.BROADCASTER)
                .build()
        );
    }

    /**
     * resolve -> true
     * resolve -> false
     * reject -> exception
     *
     * @param conferenceId the existing conference to join
     * @return a promise to resolve
     */
    public Promise<Boolean> join(@NonNull final String conferenceId) {
        return join(new ConferenceJoinInformation.Builder(conferenceId)
                .setConferenceUserType(ConferenceUserType.NORMAL)
                .build()
        );
    }

    /**
     * resolve -> true
     * resolve -> false
     * reject -> exception
     *
     * @param conferenceJoinInformation the holder of the information to join
     * @return a promise to resolve
     */
    private Promise<Boolean> join(@NonNull ConferenceJoinInformation conferenceJoinInformation) {
        return new JoinPromise(this,
                getMediaService(),
                mConferenceObservableProvider,
                getConferenceInformation(conferenceJoinInformation.getConferenceId()),
                getEventBus(),
                mInstance,
                conferenceJoinInformation.getConferenceId(),
                conferenceJoinInformation.getConferenceUserType()).createPromise();
    }

    protected Promise<Boolean> onConferenceCreated(@NonNull final String conferenceId,
                                                   @NonNull final String conferenceAlias,
                                                   @Nullable final NormalConferenceResponse other) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                createOrSetConferenceWithParams(conferenceId, conferenceAlias);
                getConferenceInformation(conferenceId).setConferenceState(ConferenceState.CREATED);

                mEventBus.post(new ConferenceCreationSuccess(conferenceId, conferenceAlias));

                joinVoxeetConference(conferenceId).then(new PromiseExec<Boolean, Object>() {
                    @Override
                    public void onCall(@Nullable Boolean result, @NonNull Solver<Object> internal_solver) {
                        solver.resolve(result);
                    }
                }).error(new ErrorPromise() {
                    @Override
                    public void onError(Throwable error) {
                        error.printStackTrace();
                        solver.resolve(false);
                    }
                });
            }
        });
    }

    /**
     * resole -> true
     * reject -> network error or on created error
     *
     * @return a promise with a boolean indicating the demo conference is (or not) successfully joined
     */
    public Promise<Boolean> demo() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                final Call<ConferenceResponse> user = mConferenceObservableProvider.getCreateDemoObservable();
                HttpHelper.enqueue(user, new HttpCallback<ConferenceResponse>() {
                    @Override
                    public void onSuccess(@NonNull final ConferenceResponse object, @NonNull Response<ConferenceResponse> response) {
                        solver.resolve(listenConference(response.body().getConfId()));
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<ConferenceResponse> response) {
                        HttpException.dumpErrorResponse(response);

                        mEventBus.post(new ConferenceCreatedError(handleError(e)));
                        solver.reject(e);
                    }
                });
            }
        });
    }

    /**
     * resolve -> list of ConferenceRefreshedUsers
     * reject -> network error
     *
     * @param userInfos a non null list of non null strings
     * @return a non null promise
     */
    @NonNull
    public Promise<List<ConferenceRefreshedEvent>> inviteUserInfos(final String conferenceId, final List<UserInfo> userInfos) {
        return new Promise<>(new PromiseSolver<List<ConferenceRefreshedEvent>>() {
            @Override
            public void onCall(@NonNull Solver<List<ConferenceRefreshedEvent>> solver) {
                //Warning = getConferenceUsers is returning a new non retained if not in a conference
                List<ConferenceUser> users = getConferenceUsers();
                List<String> strings = new ArrayList<>();

                if (null != userInfos) {
                    for (UserInfo info : userInfos) {
                        if (null != info && info.getExternalId() != null) {
                            int i = 0;
                            while (i < users.size()) {
                                ConferenceUser user = users.get(i);
                                if (null != user && null != user.getUserInfo()
                                        && info.getExternalId().equalsIgnoreCase(user.getUserInfo().getExternalId())) {
                                    user.updateIfNeeded(info.getName(), info.getAvatarUrl());
                                }
                                i++;
                            }
                            strings.add(info.getExternalId());
                        }
                    }
                }

                solver.resolve(invite(conferenceId, strings));
            }
        });
    }

    /**
     * resolve -> list of ConferenceRefreshedUsers
     * reject -> network error
     *
     * @param conferenceId
     * @param ids          a non null list of non null strings
     * @return a non null promise
     */
    @NonNull
    @Deprecated
    public Promise<List<ConferenceRefreshedEvent>> invite(final String conferenceId, final List<String> ids) {
        return new InvitePromise(this,
                getMediaService(),
                mConferenceObservableProvider,
                getCurrentConferenceInformation(),
                getEventBus(),
                conferenceId,
                ids).createPromise();
    }

    /**
     * resolve -> true
     * resolve -> false response from server is an error
     * reject -> network error
     *
     * @return
     */
    public Promise<Boolean> logout() {
        return new LogoutPromise(this,
                getMediaService(),
                mConferenceObservableProvider,
                null,
                getEventBus()).createPromise();
    }

    /**
     * resolve -> true
     * resolve -> false in case of error
     * reject -> dangerous issue
     *
     * @param conferenceId the mConference id
     * @param offset
     * @return
     */
    public Promise<Boolean> replay(final String conferenceId, final long offset) {
        return new ReplayPromise(this,
                getMediaService(),
                mConferenceObservableProvider,
                getConferenceInformation(conferenceId),
                getEventBus(),
                conferenceId,
                offset).createPromise();
    }

    /**
     * resolve -> true
     * resolve -> false in case of non 200 response
     * reject -> network error
     *
     * @return
     */
    public Promise<Boolean> startRecording() {
        return new StartRecordingPromiseable(this,
                getMediaService(),
                mConferenceObservableProvider,
                getCurrentConferenceInformation(),
                getEventBus()).createPromise();
    }

    /**
     * resolve -> true
     * resolve -> false not a 200 response
     * reject -> network error
     *
     * @return
     */
    public Promise<Boolean> stopRecording() {
        return new StopRecordingPromiseable(this,
                getMediaService(),
                mConferenceObservableProvider,
                getCurrentConferenceInformation(),
                getEventBus()).createPromise();
    }

    /**
     * Get the current mConference status
     * send events but also managed by the promise
     * <p>
     * resolve -> GetConferenceStatusEvent
     * reject -> error
     *
     * @param conferenceId the mConference id
     * @return
     */
    public Promise<GetConferenceStatusEvent> getConferenceStatus(final String conferenceId) {
        return new GetConferenceStatusPromiseable(this,
                getMediaService(),
                mConferenceObservableProvider,
                getConferenceInformation(conferenceId),
                getEventBus()).createPromise();
    }

    /**
     * resolve -> event
     * reject -> network error
     *
     * @param conferenceId the mConference id
     * @return
     */
    public Promise<GetConferenceHistoryEvent> conferenceHistory(final String conferenceId) {
        return new GetConferenceHistoryPromiseable(this,
                getMediaService(),
                mConferenceObservableProvider,
                getCurrentConferenceInformation(),
                getEventBus()).createPromise();
    }

    /**
     * resolve -> true
     * resolve -> false
     * reject -> none
     *
     * @return
     */
    @Deprecated
    public Promise<Boolean> switchCamera() {
        return getMediaService().switchCamera();
    }

    /**
     * resolve -> true
     * reject -> network error
     *
     * @param conferenceId the mConference id
     * @return
     */
    public Promise<Boolean> subscribe(@NonNull final String conferenceId) {
        return new SubscribeConferenceEventPromiseable(this,
                getMediaService(),
                mConferenceObservableProvider,
                getConferenceInformation(conferenceId),
                getEventBus()).createPromise();
    }

    /**
     * resolve -> true
     * reject -> network error
     *
     * @return a valid promise
     */
    public Promise<Boolean> unSubscribe(@NonNull final String conferenceId) {
        return new UnsubscribeConferenceEventPromiseable(this,
                getMediaService(),
                mConferenceObservableProvider,
                getConferenceInformation(conferenceId),
                getEventBus()).createPromise();
    }

    /**
     * resolve -> true
     * reject -> network error
     *
     * @param conferenceId
     * @return
     */
    public Promise<Boolean> subscribeForCall(final String conferenceId) {
        return new SubscribeForCallStartPromiseable(this,
                getMediaService(),
                mConferenceObservableProvider,
                getConferenceInformation(conferenceId),
                getEventBus()).createPromise();
    }

    /**
     * resolve -> true
     * reject -> network error
     *
     * @param conferenceId
     * @return
     */
    public Promise<Boolean> unSubscribeFromCall(final String conferenceId) {
        return new UnsubscribeForCallStartPromiseable(this,
                getMediaService(),
                mConferenceObservableProvider,
                getConferenceInformation(conferenceId),
                getEventBus()).createPromise();
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(@NonNull SocketConnectEvent event) {
        String conferenceId = getConferenceId();
        ConferenceInformation information = getCurrentConferenceInformation();

        if (null != conferenceId && null != information && ConferenceState.JOINED.equals(information.getConferenceState())) {
            if (isICERestartEnabled()) {
                Log.d(TAG, "onEvent: SocketConnectEvent Joined <3");
                final Call<ResponseBody> user = mConferenceObservableProvider.iceRestart(conferenceId);
                HttpHelper.enqueue(user, new HttpCallback<ResponseBody>() {
                    @Override
                    public void onSuccess(@NonNull ResponseBody object, @NonNull Response<ResponseBody> response) {
                        Log.d(TAG, "onEvent: SocketConnectEvent Joined responded <3");
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<ResponseBody> response) {
                        Log.d(TAG, "onEvent: SocketConnectEvent Joined error </3");
                        HttpException.dumpErrorResponse(response);
                    }
                });
            } else {
                Log.d(TAG, "onEvent: socket state opened while in conference but no isICERestartEnabled() = true. A reconnect may be longer");
            }
        } else {
            ConferenceState state = ConferenceState.DEFAULT;
            if (null != information) {
                state = information.getConferenceState();
            }
            Log.d(TAG, "onEvent: SocketConnectEvent not rejoined </3 " + state + " " + conferenceId);
        }
    }

    /**
     * resolve -> true
     * reject -> InConferenceException must be considered false
     * reject -> PromiseConferenceJoinedErrorException must be considered false
     *
     * @param conferenceId the mConference id
     * @return a promise to catch
     */
    @NonNull
    private Promise<Boolean> joinVoxeetConference(@NonNull final String conferenceId) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                String existingConferenceId = getConferenceId();

                Log.d(TAG, "onCall: " + mInConference + " " + existingConferenceId + " " + conferenceId);
                if (mInConference && (null == existingConferenceId || !existingConferenceId.equals(conferenceId))) {
                    try {
                        throw new InConferenceException();
                    } catch (InConferenceException exception) {
                        solver.reject(exception);
                        return;
                    }
                }

                //remove the timeout, a new one should be started when user will be invited
                removeTimeoutCallbacks();

                Log.d(TAG, "joining " + conferenceId);

                setIsInConference(true);
                mConferenceId = conferenceId;
                final ConferenceInformation information = getCurrentConferenceInformation();

                if (null != information) {
                    solver.resolve(joinConferenceInternalPackage(information));
                } else {
                    Log.d(TAG, "onCall: WARNING :: IMPROPER RESULT !!");
                    solver.resolve(false);
                }

            }
        });
    }


    protected void setCurrentConferenceIfNotInPreviousConference(@NonNull ConferenceInformation conference) {
        String conferenceId = conference.getConference().getConferenceId();

        boolean invalidId = null == conferenceId;
        boolean inConference = isInConference();
        boolean sameConference = null == mConferenceId || mConferenceId.equals(conferenceId);
        boolean canOverride = !inConference || sameConference;

        if (!invalidId && canOverride) {
            mConferenceId = conferenceId;
        }

        Log.d(TAG, "setCurrentConferenceIfNotInPreviousConference: "
                + " invalidId = " + invalidId
                + " inConference = " + inConference
                + " sameConference = " + sameConference + " (" + mConferenceId + " " + conferenceId + ") "
                + " canOverride = " + canOverride
                + " mConferenceId = " + mConferenceId);
    }

    /**
     * resolve -> true
     * reject -> InConferenceException must be considered false
     * reject -> PromiseConferenceJoinedErrorException must be considered false
     * <p>
     * Warning : if multiple attempt to join the same conference are made, it is not managed...
     * May be fix in a future version but currently not in the flow
     *
     * @param conference the conference to join
     * @return a promise to catch
     */
    @NonNull
    protected Promise<Boolean> joinConferenceInternalPackage(@NonNull final ConferenceInformation conference) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                final String conferenceId = conference.getConference().getConferenceId();

                boolean invalidId = null == conferenceId;
                boolean inConference = isInConference();
                boolean sameConference = null == mConferenceId || mConferenceId.equals(conferenceId);

                Log.d(TAG, "onCall: join conference " + conference
                        + " invalidId:=" + invalidId
                        + " inConference:=" + inConference
                        + " sameConference:=" + sameConference
                        + " canOverride:=" + (inConference && !sameConference)
                        + " mConferenceId:=" + mConferenceId
                        + " conferenceId:=" + conferenceId);
                Log.d(TAG, "onCall: Not checking for conference joined. Method internally used -- in case of regression please contact support for this specific");
                if (invalidId/* || (inConference && !sameConference)*/) {
                    try {
                        throw new InConferenceException();
                    } catch (InConferenceException exception) {
                        solver.reject(exception);
                        return;
                    }
                }

                //remove the timeout, a new one should be started when user will be invited
                removeTimeoutCallbacks();

                Log.d(TAG, "joining " + conferenceId);

                setIsInConference(true);
                mConferenceId = conferenceId;

                conference.setConferenceState(ConferenceState.JOINING);

                AudioService service = mInstance.getAudioService();
                if (null != service) {
                    service.enableAec(true);
                    service.enableNoiseSuppressor(true);
                }

                boolean isSuccess = initMedia(conference.isListener());

                if (!isSuccess) {
                    Log.d(TAG, "onCall: InitMedia failed... new state = left");
                    try {
                        setIsInConference(false);
                        conference.setConferenceState(ConferenceState.LEFT);
                        closeMedia();
                        ConferenceJoinedError error = new ConferenceJoinedError(handleError(null));
                        mEventBus.post(error);
                        throw new PromiseConferenceJoinedErrorException(error, null);
                    } catch (PromiseConferenceJoinedErrorException exception) {
                        solver.reject(exception);
                    }
                    return;
                }

                mEventBus.post(new ConferencePreJoinedEvent(getConferenceId(), getAliasId()));

                final Call<ResumeConference> user = mConferenceObservableProvider.joinConference(conferenceId, new JoinParameters(DeviceType.ANDROID, conference.isListener()));

                HttpHelper.enqueue(user, new HttpCallback<ResumeConference>() {
                    @Override
                    public void onSuccess(@NonNull ResumeConference object, @NonNull Response<ResumeConference> response) {
                        createOrSetConferenceWithParams(object.getConferenceId(),
                                object.getConferenceAlias());
                        initMedia(conference.isListener());

                        ConferenceInformation information = getCurrentConferenceInformation();
                        if (null != information) {
                            information.setConferenceState(ConferenceState.JOINED);
                        }

                        onConferenceResumedInternal(object, solver);
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<ResumeConference> response) {
                        HttpException.dumpErrorResponse(response);

                        try {
                            e.printStackTrace();

                            setIsInConference(false);

                            //ConferenceInformation information = getCurrentConferenceInformation();
                            conference.setConferenceState(ConferenceState.LEFT);
                            closeMedia();
                            ConferenceJoinedError error = new ConferenceJoinedError(handleError(e));
                            mEventBus.post(error);
                            throw new PromiseConferenceJoinedErrorException(error, e);
                        } catch (PromiseConferenceJoinedErrorException exception) {
                            solver.reject(exception);
                        }
                    }
                });
            }
        });
    }

    protected void onConferenceResumedInternal(@NonNull ResumeConference response,
                                               @NonNull final Solver<Boolean> solver) {
        mConferenceId = response.getConferenceId();

        ConferenceInformation information = getCurrentConferenceInformation();

        //information should not be null in this case

        Conference conference = information.getConference();

        conference.setConferenceId(response.getConferenceId());
        conference.setConferenceAlias(response.getConferenceId());

        if (response.getConferenceAlias() != null) {
            conference.setConferenceAlias(response.getConferenceAlias());
        }

        information.participantsToConferenceUsers(response.getParticipants());

        setIsInConference(true);

        getAudioService().setAudioRoute(AudioRoute.ROUTE_PHONE);

        mEventBus.post(new ConferenceJoinedSuccessEvent(getConferenceId(), getAliasId()));

        if (null == response.getCandidates() && null == response.getDescription()) {
            Log.d(TAG, "onConferenceResumedInternal: candidates + description");
            solver.resolve(true);
            return;
        }

        Log.d(TAG, "onConferenceResumedInternal: having candidates and description");

        UserInfo info = mInstance.getUserInfo();
        String name = "";
        String externalId = "";
        String avatarUrl = "";
        String userId = "";
        if (null != info) {
            userId = VoxeetPreferences.id();
            name = info.getName();
            externalId = info.getExternalId();
            avatarUrl = info.getAvatarUrl();
        }

        handleAnswer(response.getConferenceId(),
                userId,
                externalId,
                name,
                avatarUrl,
                "ANDROID",
                true,
                response.getDescription(),
                response.getCandidates())
                .then(new PromiseExec<Boolean, Object>() {
                    @Override
                    public void onCall(@Nullable Boolean aBoolean, @NonNull Solver<Object> s) {
                        Log.d(TAG, "onCall: answer called, result is " + aBoolean + " " + VoxeetPreferences.id());
                        solver.resolve(true);
                    }
                })
                .error(new ErrorPromise() {
                    @Override
                    public void onError(@NonNull Throwable error) {
                        error.printStackTrace();

                        ExceptionManager.sendException(error);
                        solver.reject(error);
                    }
                });
    }

    /**
     * resolve -> the event with the list
     * reject -> any error
     *
     * @return
     */
    public Promise<ConferenceUsersInvitedEvent> getInvitedUsers(@NonNull String conferenceId) {
        return new GetConferenceStatus(this,
                getMediaService(),
                mConferenceObservableProvider,
                getConferenceInformation(conferenceId),
                getEventBus()).createPromise();
    }

    public Promise<Boolean> sendMessage(@NonNull String conferenceId, @NonNull final String message) {
        return new SendBroadcastMessagePromise(this,
                getMediaService(),
                mConferenceObservableProvider,
                getConferenceInformation(conferenceId),
                getEventBus(),
                conferenceId,
                message).createPromise();
    }

    /**
     * This method attempts to leave the current mConference
     * <p>
     * The previous version used a boolean, the new method fingerprint is now void
     */
    public Promise<Boolean> leave() {
        return new LeavePromise(this,
                getMediaService(),
                mConferenceObservableProvider,
                getCurrentConferenceInformation(),
                getEventBus())
                .createPromise();
    }


    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Public Event management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceStatsEvent event) {
        ConferenceStats stats = event.getEvent();
        if (stats.getConference_id() != null && stats.getConference_id().equals(getConferenceId())) {
            float mos = stats.getScore(VoxeetPreferences.id());

            mEventBus.post(new QualityIndicators(mos));
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceUpdatedEvent event) {
        //TODO maybe manage users here too ?
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final InvitationReceived invitation) {
        try {
            onEvent(invitation.getEvent());
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final InvitationReceivedEvent invitation) {

        String conferenceId = invitation.getConferenceId();

        if (null == conferenceId && null != invitation.getConference()) {
            conferenceId = invitation.getConference().getConferenceId();
        }

        Log.d(TAG, "onEvent: current mConferenceId " + mConferenceId + " vs " + invitation.getConferenceId() + " vs " + conferenceId);

        if (null != mConferenceId && mConferenceId.equals(conferenceId)) {
            Log.d(TAG, "onEvent: receiving invitation for our conference, we prevent this");
            return;
        }

        //we update the information for this conference only
        final ConferenceInformation information = getConferenceInformation(conferenceId);

        Conference conference = invitation.getConference();
        boolean is_own_user = invitation.getUserId().equals(VoxeetPreferences.id());
        ConferenceType type = ConferenceType.fromId(conference.getConferenceType());

        if (isLive() && null != getConferenceId() && getConferenceId().equals(conferenceId)) {
            Log.d(TAG, "onEvent: receiving invitation for the same conference, invalidate the call");
            return;
        }

        if (!is_own_user || ConferenceType.SCHEDULED.equals(type)) {
            InvitationReceivedEvent.UserInviter inviter = invitation.getInviter();
            List<DefaultInvitation> invitations = invitation.getInvitations();

            if (null == information) {
                Log.d(TAG, "onEvent: INVALID INFORMATION FOR THE CONFERENCE");
            }

            if (null != inviter && null != inviter.externalId && null != inviter.userId) {
                Log.d(TAG, "onEvent: Invitation with inviter " + inviter.userId + " " + inviter.externalId);
                UserInfo info = createUserInfo(inviter.nickName, inviter.externalId, inviter.externalAvatarUrl);

                ConferenceUser inviter_user = new ConferenceUser(inviter.userId,
                        "", info);
                if (null != information) {
                    information.getLastInvitationReceived().add(inviter_user);
                    information.getUserIdsCached().put(inviter.userId, inviter.externalId);
                }
            } else {
                Log.d(TAG, "onEvent: Invitation with invalid inviter");
            }

            final String finalConferenceId = conferenceId;
            getConferenceStatus(conference.getConferenceId())
                    .then(new PromiseExec<GetConferenceStatusEvent, Object>() {
                        @Override
                        public void onCall(@Nullable GetConferenceStatusEvent result, @NonNull Solver<Object> internal_solver) {
                            Log.d(TAG, "onSuccess: " + result);

                            //removed the check for live Event since now it is able to have multiple

                            if (result.getConferenceUsers().size() > 0) {
                                Log.d(TAG, "onEvent: users " + result.getConferenceUsers().size());

                                ConferenceUser foundUser = null;
                                String foundExternalId = null;

                                List<ConferenceUser> merged_list = new ArrayList<>();
                                //add every user from the mConference
                                merged_list.addAll(result.getConferenceUsers());
                                //add every use from the invitation
                                merged_list.addAll(getLastInvitationUsers());

                                List<ConferenceUser> list = ConferenceUtils.findUsersMatching(currentUserOrEmpty(), merged_list);

                                for (ConferenceUser conferenceUser : list) {
                                    String userId = conferenceUser.getUserId();
                                    if (null == foundExternalId && null != information) {
                                        String externalId = conferenceUser.getUserInfo().getExternalId();
                                        String cachedExternalId = information.getUserIdsCached().get(userId);
                                        Log.d(TAG, "onEvent: " + userId + " " + externalId + " " + cachedExternalId);

                                        foundUser = conferenceUser;
                                        if (null != userId && (null == externalId || userId.equals(externalId))) {
                                            //should be different than null
                                            externalId = cachedExternalId;
                                            if (null != externalId) {
                                                foundExternalId = cachedExternalId;
                                            }
                                        }
                                    }
                                }

                                if (foundUser != null && null == foundExternalId) {
                                    foundExternalId = foundUser.getUserId();
                                    Log.d(TAG, "externalId is null, setting it to userId");
                                }

                                Map<String, String> infos = new HashMap<>();

                                if (null != foundUser) {

                                }
                                infos.put(VoxeetIntentFactory.INVITER_ID, null != foundUser ? foundUser.getUserId() : "");
                                infos.put(VoxeetIntentFactory.INVITER_NAME, null != foundUser ? foundUser.getUserInfo().getName() : "");
                                infos.put(VoxeetIntentFactory.NOTIF_TYPE, null != result.getType() ? result.getType() : "");
                                infos.put(VoxeetIntentFactory.INVITER_EXTERNAL_ID, null != foundUser ? foundExternalId : "");
                                infos.put(VoxeetIntentFactory.INVITER_URL, null != foundUser ? foundUser.getUserInfo().getAvatarUrl() : "");
                                infos.put(VoxeetIntentFactory.CONF_ID, null != result.getConferenceId() ? result.getConferenceId() : "");

                                Intent intent = VoxeetIntentFactory.buildFrom(mContext, VoxeetPreferences.getDefaultActivity(), infos);
                                if (intent != null)
                                    mContext.startActivity(intent);

                                mEventBus.post(new IncomingCallEvent());
                            }
                        }
                    })
                    .error(new ErrorPromise() {
                        @Override
                        public void onError(@NonNull Throwable error) {
                            error.printStackTrace();
                        }
                    });
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(OwnConferenceStartedEvent ownConferenceStartedEvent) {
        if (null != ownConferenceStartedEvent.event().getConferenceInfos()) {
            mConferenceId = ownConferenceStartedEvent.event().getConferenceInfos().getConferenceId();
        }

        Conference conference = getConference();

        if (null != conference) {
            conference.setConferenceInfos(ownConferenceStartedEvent.event().getConferenceInfos());
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final StartVideoAnswerEvent event) {
        Log.d(TAG, "onEvent: start video event " + event.isSuccess() + " " + event.message());
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(RenegociationUpdate event) {
        RenegociationEndedEvent renego = event.getEvent();

        Log.d(TAG, "onEvent: Renegociation " + renego.getConferenceId()
                + " " + renego.getType() + " " + renego.getAnswerReceived()
                + " " + renego.getOfferSent());
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final OfferCreatedEvent event) {
        Log.d(TAG, "onEvent: OfferCreatedEvent " + event.message());
        OfferCreated offer = event.offer();
        handleAnswer(offer.getConferenceId(),
                offer.getUserId(),
                offer.getExternalId(),
                offer.getName(),
                offer.getAvatarUrl(),
                offer.getDevice(),
                offer.isMaster(),
                offer.getDescription(),
                offer.getCandidates())
                .then(new PromiseExec<Boolean, Object>() {
                    @Override
                    public void onCall(@Nullable Boolean aBoolean, @NonNull Solver<Object> solver) {
                        Log.d(TAG, "onCall: answer called, result is " + aBoolean + " " + event.offer().getUserId() + " " + VoxeetPreferences.id());
                    }
                })
                .error(new ErrorPromise() {
                    @Override
                    public void onError(@NonNull Throwable error) {
                        error.printStackTrace();
                        ExceptionManager.sendException(error);
                    }
                });
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(ConferenceUserAddedEvent event) {
        if (!mConferenceId.equals(event.getEvent().getConferenceId())) {
            Log.d(TAG, "onEvent: USER ADDED FOR ANOTHER CONFERENCE");
        }
        String conferenceId = event.getEvent().getConferenceId();
        if (null == conferenceId) conferenceId = "";

        Conference conference = getOrSetConference(conferenceId);
        ConferenceUserAdded content = event.getEvent();

        if (null != content) {
            Log.d(TAG, "onEvent: ParticipantAdded " + content + " " + content.getUserId() + " " + conference.getConferenceUsers().size());

            ConferenceUser user = conference.findUserById(content.getUserId());

            if (null == user) {
                user = new ConferenceUser(content.getUserId(), null);
                user.setIsOwner(false);
                user.setUserInfo(new UserInfo(content.getName(),
                        content.getExternalId(),
                        content.getAvatarUrl()));

                conference.getConferenceUsers().add(user);
            }

            if (null == user.getUserInfo()) {
                Log.d(TAG, "onEvent: existing user info are invalid, refreshing");
                user.setUserInfo(new UserInfo(content.getName(),
                        content.getExternalId(),
                        content.getAvatarUrl()));
            }

            user.updateIfNeeded(content.getName(), content.getAvatarUrl());

            ConferenceUserStatus status = ConferenceUserStatus.fromString(content.getStatus());
            if (null != status) {
                user.setConferenceStatus(status);
                user.setStatus(content.getStatus());
            }

            MediaStream stream = null;
            if (hasMedia()) {
                HashMap<String, MediaStream> streams = getMapOfStreams();
                if (streams.containsKey(user.getUserId())) {
                    stream = streams.get(user.getUserId());
                }
            }

            //update the current conference state specificly from the users point of view
            updateConferenceFromUsers();
            mEventBus.post(new ConferenceUserJoinedEvent(user, conferenceId, conference, stream));
        }

        //in a previous version in the v1.0, the createOffer was forced right here
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final ParticipantUpdatedEvent event) {
        String conferenceId = event.getConfId();
        if (null == conferenceId) conferenceId = "";

        com.voxeet.sdk.models.abs.Conference conference = null;
        ConferenceUser user = findUserById(event.getUserId());
        ConferenceInformation information = getConferenceInformation(conferenceId);
        if (null != information) {
            conference = information.getConference();
            user = ((Conference) conference).findUserById(event.getUserId());
        }

        final ConferenceUserStatus status = ConferenceUserStatus.valueOf(event.getStatus());

        if (user != null) {
            user.setStatus(event.getStatus());
        }

        //update the current conference state specificly from the users point of view
        updateConferenceFromUsers();

        switch (status) {
            case CONNECTING:
                if (null != user && !user.getUserId().equals(VoxeetPreferences.id())) {
                    Log.d(TAG, "Cancelling timeout timer from user connecting");
                    removeTimeoutCallbacks();
                }
                break;
            case LEFT:
            case DECLINE:
                if (hasMedia() && user != null) {
                    Log.d(TAG, "In mConference user with id: " + event.getUserId() + " status updated to " + event.getStatus());
                    getMedia().removePeer(event.getUserId());
                }
                break;
            default:
                Log.d(TAG, "status not managed updated to " + event.getStatus());

        }

        //check if the timeout runnable was not triggered
        boolean timeout_managed = timeoutRunnable != null && timeoutRunnable.isTriggered();

        if (!timeout_managed) {
            switch (status) {
                case DECLINE:
                    if (!event.getUserId().equals(VoxeetPreferences.id())) {
                        Log.d(TAG, "Conference user with id: " + event.getUserId() + " declined the call");

                        mEventBus.post(new ConferenceUserCallDeclinedEvent(event.getConfId(), event.getUserId(), event.getStatus()));
                    }
                    break;
                case LEFT:
                    onUserLeft(event.getUserId());
                    break;
                default:
                    MediaStream stream = null;
                    if (hasMedia() && null != user) {
                        HashMap<String, MediaStream> streams = getMapOfStreams();
                        if (streams.containsKey(user.getUserId())) {
                            stream = streams.get(user.getUserId());
                        }
                    }

                    if (null != user) {
                        mEventBus.post(new ConferenceUserUpdatedEvent(user, conferenceId, conference, stream));
                    }
            }
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final ConferenceDestroyedPushEvent event) {
        if (null == mConferenceId || mConferenceId.equals(event.getPush().getConferenceId())) {
            closeMedia();

            mEventBus.post(event.getPush());
        } else {
            Log.d(TAG, "onEvent: another conference has ended");
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final ConferenceEndedEvent event) {
        if (null == mConferenceId || mConferenceId.equals(event.getEvent().getConferenceId())) {
            closeMedia();

            mEventBus.post(event.getEvent());
        } else {
            Log.d(TAG, "onEvent: another conference has ended");
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final QualityUpdatedEvent event) {
        for (ConferenceUser conferenceUser : event.getEvent().getUser()) {
            ConferenceUser user = findUserById(conferenceUser.getUserId());
            if (user != null) {
                user.setQuality(conferenceUser.getQuality());

                mEventBus.post(new ConferenceUserQualityUpdatedEvent(user));
            }
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(final RecordingStatusUpdate event) {
        String conferenceId = event.getEvent().getConferenceId();
        Log.d(TAG, "onEvent: eevnt for " + conferenceId);

        Conference conference = getConferenceInformation(conferenceId).getConference();
        RecordingStatusUpdateEvent recordingStatusUpdateEvent = event.getEvent();
        ConferenceUser user = findUserById(recordingStatusUpdateEvent.getUserId());
        if (user != null) {
            RecordingStatus status = RecordingStatus.valueOf(recordingStatusUpdateEvent.getRecordingStatus());
            if (status == RecordingStatus.RECORDING) {
                isRecording = true;
                conference.setStartRecordTimestamp(new Date(recordingStatusUpdateEvent.getTimeStamp()));
                conference.setRecordingStatus(RecordingStatus.RECORDING);
                conference.setRecordingUser(recordingStatusUpdateEvent.getUserId());
                user.setIsRecordingOwner(true);
            } else {
                isRecording = false;
                conference.setStartRecordTimestamp(null);
                conference.setRecordingStatus(RecordingStatus.NOT_RECORDING);
                conference.setRecordingUser(null);
                user.setIsRecordingOwner(false);
            }
        }

        mEventBus.post(recordingStatusUpdateEvent);
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Protected management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    protected ConferenceUser createConferenceUser(String userId, String device, UserInfo userInfo) {
        return new ConferenceUser(userId, device, userInfo);
    }

    protected UserInfo createUserInfo(String userName, String externalId, String avatarUrl) {
        return new UserInfo(userName, externalId, avatarUrl);
    }

    protected void setConference(@NonNull Conference conference) {
        mConferenceId = conference.getConferenceId();
        ConferenceInformation information = getCurrentConferenceInformation();
        information.setConference(conference);
    }

    protected void setConferenceAlias(String alias) {
        Conference conference = getConference();
        conference.setConferenceAlias(alias);
        //null ? throw error B)
    }

    public EventBus getEventBus() {
        return mEventBus;
    }

    protected MediaSDK getMedia() {
        return getMediaService().getMedia();
    }

    /**
     * Return true if the current implementation is SDK
     * false otherwise
     *
     * @return the state of this implementation
     */
    protected boolean isSDK() {
        return true;
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Private Promises management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    protected Promise<Integer> answer(final String peer, final SdpMessage message) {
        return new Promise<>(new PromiseSolver<Integer>() {
            @Override
            public void onCall(@NonNull final Solver<Integer> solver) {
                try {
                    if (null == mConferenceId) {
                        throw new NotInConferenceException();
                    }
                } catch (NotInConferenceException exception) {
                    solver.reject(exception);
                    return;
                }

                Log.d("SDKMEDIA", "answer: peer := " + peer + " message := " + message);

                final Call<ResponseBody> user = mConferenceObservableProvider.answerConference(mConferenceId, peer, message.getDescription());
                HttpHelper.enqueue(user, new HttpCallback<ResponseBody>() {
                    @Override
                    public void onSuccess(@NonNull ResponseBody object, @NonNull Response<ResponseBody> response) {
                        if (response.code() != 200) {
                            try {
                                ParticipantAddedErrorEvent event = new ParticipantAddedErrorEvent(response.code() + "");
                                throw new PromiseParticipantAddedErrorEventException(event);
                            } catch (PromiseParticipantAddedErrorEventException exception) {
                                solver.reject(exception);
                            }
                        } else {
                            solver.resolve(response.code());
                        }
                    }

                    @Override
                    public void onFailure(@NonNull Throwable e, @Nullable Response<ResponseBody> response) {
                        HttpException.dumpErrorResponse(response);
                        try {
                            ParticipantAddedErrorEvent event = new ParticipantAddedErrorEvent(handleError(e));
                            throw new PromiseParticipantAddedErrorEventException(event);
                        } catch (PromiseParticipantAddedErrorEventException exception) {
                            solver.reject(exception);
                        }
                    }
                });
            }
        });
    }


    /**
     * resolve -> true everything went fine
     * resolve -> false an error occured
     * reject -> media exception
     *
     * @return a promise with the result of the call
     */
    protected Promise<Boolean> handleAnswer(@NonNull final String conferenceId,
                                            @NonNull final String userId,
                                            @NonNull final String externalId,
                                            @NonNull final String userName,
                                            @NonNull final String avatarUrl,
                                            @NonNull final String device,
                                            final boolean isMaster,
                                            @NonNull final OfferDescription offerDescription,
                                            @NonNull final List<OfferCandidate> offerCandidates) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                if (!hasMedia()) {
                    try {
                        throw new MediaEngineException("handleAnswer media is null");
                    } catch (MediaEngineException exception) {
                        solver.reject(exception);
                    }
                }

                if (!conferenceId.equals(mConferenceId)) {
                    Log.d(TAG, "onCall: CONFERENCE IS NOT THE SAME ! ANSWER SHALL BE DISCARDED");
                }

                SdpDescription description = new SdpDescription(offerDescription.type, offerDescription.sdp);

                List<SdpCandidate> candidates = new ArrayList<>();
                for (OfferCandidate candidate : offerCandidates) {
                    Log.d(TAG, "onCall: sdpcandidate " + candidate.getMid() + " " + candidate.getmLine());
                    candidates.add(new SdpCandidate(candidate.getMid(), Integer.parseInt(candidate.getmLine()), candidate.getSdp()));
                }

                //String userId = offer.getUserId();//isSDK() ? offer.getExternalId() : offer.getUserId();
                Log.d(TAG, "handleAnswer: offer := " + userId + " " + externalId);
                Log.d("SDKMEDIA", "handleAnswer: " + userId + " " + externalId + " " + description.ssrc);


                try {
                    //TODO in the callback, check for the conferenceId being the same !
                    getMedia().createAnswerForPeer(userId,
                            description.ssrc,
                            description,
                            candidates,
                            isMaster,
                            new PendingPeerCallback() {
                                @Override
                                public void onMessage(@Nullable SdpMessage message) {
                                    try {
                                        if (null == mConferenceId) {
                                            Log.d(TAG, "onMessage: INVALID CONFERENCE ID WHEN OFFER IS RECEIVED");
                                            mConferenceId = conferenceId;
                                        }
                                        Conference conference = getConference();

                                        ConferenceUser user = null;

                                        for (ConferenceUser in_conf : conference.getConferenceUsers()) {
                                            if (in_conf.getUserId() != null && in_conf.getUserId().equals(userId)) {
                                                user = in_conf;
                                            }
                                        }

                                        UserInfo infos;
                                        if (userId.contains("11111"))
                                            infos = createUserInfo("Julie", externalId, "https://raw.githubusercontent.com/romainbenmansour/JCenter/master/user_1.png");
                                        else if (userId.contains("22222"))
                                            infos = createUserInfo("Sophie", externalId, "https://raw.githubusercontent.com/romainbenmansour/JCenter/master/user_2.png");
                                        else if (userId.contains("33333"))
                                            infos = createUserInfo("Mike", externalId, "https://raw.githubusercontent.com/romainbenmansour/JCenter/master/user_3.png");
                                        else
                                            infos = createUserInfo(userName, externalId, avatarUrl);

                                        if (user == null) {
                                            user = createConferenceUser(userId, device, infos);
                                            conference.getConferenceUsers().add(user);
                                        } else {
                                            user.setUserInfo(infos);
                                        }


                                        setUserPosition(userId, 0, 0.5);

                                        answer(userId, message)
                                                .then(new PromiseExec<Integer, Object>() {
                                                    @Override
                                                    public void onCall(@Nullable Integer result, @NonNull Solver<Object> internal_solver) {
                                                        Log.d(TAG, "onSuccess: " + result);

                                                        solver.resolve(true);
                                                    }
                                                })
                                                .error(new ErrorPromise() {
                                                    @Override
                                                    public void onError(Throwable error) {
                                                        if (error instanceof PromiseParticipantAddedErrorEventException)
                                                            mEventBus.post(((PromiseParticipantAddedErrorEventException) error).getEvent());
                                                        else
                                                            error.printStackTrace();

                                                        solver.resolve(false);
                                                    }
                                                });
                                    } catch (Exception e) {
                                        Log.d(TAG, "onMessage: unlockPeerOperation" + e.getMessage());
                                        solver.resolve(false);
                                    }
                                }
                            });
                } catch (MediaEngineException e) {
                    e.printStackTrace();
                    solver.reject(e);
                }
            }
        });
    }
    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * Private management
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    protected ConferenceUser updateConferenceParticipants(String userId, ConferenceUserStatus status) {
        ConferenceUser user = findUserById(userId);
        Conference conference = getConference();

        if (null != user && null != conference) {
            user.setIsOwner(conference.getOwnerProfile() == null || conference.getOwnerProfile().getUserId().equals(user.getProfile().getUserId()))
                    .setConferenceStatus(status)
                    .setQuality(ConferenceQuality.ULTRA_HIGH.StringValue());

            return user;
        }
        return null;
    }

    /**
     * @return false if permission refused -> cancel ! true otherwise
     */
    protected boolean initMedia(boolean listenerMode) {
        Validate.notNull(mContext, "mContext");
        Validate.notNull(VoxeetPreferences.id(), "user id");

        if (!hasMedia()) {
            try {
                if (!listenerMode && !Validate.hasMicrophonePermissions(context)) {
                    Log.d(TAG, "the app does not seem to have mic permission, disabling mic");
                    mute(true);
                    getEventBus().post(new PermissionRefusedEvent(PermissionRefusedEvent.Permission.MICROPHONE));
                    return false;
                }

                getAudioService().enable();
                getMediaService().createMedia(context, VoxeetPreferences.id(), mediaStreamListener, cameraEventsHandler,
                        isDefaultVideo || isVideoOn(), !listenerMode && Validate.hasMicrophonePermissions(context));
                getAudioService().setSpeakerMode(isDefaultOnSpeaker);
                getAudioService().requestAudioFocus();

                if (isDefaultMute) {
                    getMedia().mute();
                } else {
                    getMedia().unMute();
                }
            } catch (MediaEngineException e) {
                e.printStackTrace();
                return false;
            }
        }

        return true;
    }

    protected void closeMedia() {
        if (hasMedia()) {
            MediaSDK media = getMedia();
            com.voxeet.sdk.models.abs.Conference conference = getConference();
            ConferenceInformation information = getCurrentConferenceInformation();

            if (null != information) {
                information.setOwnVideoStarted(false);
                information.setScreenShareOn(false);
            }

            try {
                if (null != conference) {
                    for (ConferenceUser user : conference.getConferenceUsers())
                        if (user != null && user.getUserId() != null)
                            media.removePeer(user.getUserId());
                }
            } catch (Exception ex) {
                Log.e(TAG, "Error", ex);
            }

            getAudioService().abandonAudioFocusRequest();
            getAudioService().disable();

            getMediaService().releaseMedia();

            media.unsetStreamListener();
        }

        setIsInConference(false);
        mConferenceId = null;
    }

    /**
     * Remove any previous timeout callback
     * <p>
     * To be used when it is mandatory to remove such behaviour
     * for instance, when joining a new mConference
     * <p>
     * It happens since we do not want to manage every edge case :
     * - decline
     * - network error
     * - etc...
     */
    protected void removeTimeoutCallbacks() {
        if (timeoutRunnable != null) {
            timeoutRunnable.setCanceled(true);
            handler.removeCallbacks(timeoutRunnable);
        }
    }

    protected void sendTimeoutCallbacks() {
        if (mTimeOutTimer != -1) {

            timeoutRunnable = new TimeoutRunnable(this,
                    mEventBus,
                    mTimeOutTimer);

            handler.postDelayed(timeoutRunnable, mTimeOutTimer);
        }
    }

    private void onUserLeft(@NonNull String peer) {
        ConferenceUser user = findUserById(peer);
        if (user != null) {
            user.setConferenceStatus(ConferenceUserStatus.LEFT);

            getConferenceUsers().remove(user);

            mapOfStreams.remove(user.getUserId());
            mapOfScreenShareStreams.remove(user.getUserId());

            //refresh the conference state users
            updateConferenceFromUsers();
            mEventBus.post(new ConferenceUserLeftEvent(user));
        } else {
            Log.d(TAG, "run: unknown user in stream left");
        }
    }

    protected VoxeetSdkTemplate getVoxeetSDK() {
        return mSDK;
    }

    private AudioService getAudioService() {
        return getVoxeetSDK().getAudioService();
    }

    @NonNull
    protected ConferenceInformation createOrSetConferenceWithParams(@NonNull String conferenceId, @Nullable String conferenceAlias) {
        Log.d(TAG, "createOrSetConferenceWithParams: set conference id := " + conferenceId);
        ConferenceInformation information = getConferenceInformation(conferenceId);

        Conference conference = information.getConference();

        conference.setConferenceId(conferenceId);
        if (null != conferenceAlias || null == conference.getConferenceAlias()) {
            conference.setConferenceAlias(conferenceAlias);
        }

        //the information is always valid
        return information;
    }

    @Nullable
    public ConferenceInformation getCurrentConferenceInformation() {
        return getConferenceInformation(mConferenceId);
    }

    @Nullable
    protected ConferenceInformation getConferenceInformation(@Nullable String conferenceId) {
        if (null == conferenceId) {
            return null;
        }
        return mConferenceInformationHolder.getInformation(conferenceId);
    }

    private void updateConferenceFromUsers() {
        ConferenceInformation information = getCurrentConferenceInformation();
        if (null != information) {
            ConferenceState state = information.getState();
            switch (state) {
                case JOINED:
                    if (hasParticipants())
                        information.setConferenceState(ConferenceState.FIRST_PARTICIPANT);
                    break;
                case FIRST_PARTICIPANT:
                    if (!hasParticipants())
                        information.setConferenceState(ConferenceState.NO_MORE_PARTICIPANT);
                    break;
                case NO_MORE_PARTICIPANT:
                    if (hasParticipants())
                        information.setConferenceState(ConferenceState.FIRST_PARTICIPANT);
                    break;
                default:
                    //nothing
            }
            state = information.getState();
        }
    }

    public boolean hasParticipants() {
        List<ConferenceUser> users = getConferenceUsers();

        for (ConferenceUser user : users) {
            ConferenceUserStatus status = user.getConferenceStatus();
            if (null != user.getUserId() && !user.getUserId().equals(VoxeetPreferences.id())
                    && ConferenceUserStatus.ON_AIR.equals(status)) {
                return true;
            }
        }
        return false;
    }

    public void setICERestartEnabled(boolean new_state) {
        isICERestartEnabled = new_state;
    }

    public boolean isICERestartEnabled() {
        return isICERestartEnabled;
    }

    @NonNull
    public Context getContext() {
        return mContext;
    }

    @Deprecated
    @Nullable
    public String getDefaultCamera() {
        return getMediaService().getDefaultCamera();
    }

    private boolean hasMedia() {
        return null != getMedia();
    }

    @NonNull
    private MediaService getMediaService() {
        return mInstance.getMediaService();
    }

    void joinLock() {
        try {
            joinLock.lock();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void joinUnlock() {
        try {
            if (joinLock.isLocked())
                joinLock.unlock();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

}
