package com.voxeet.sdk.core;

import android.app.Application;
import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.android.media.MediaSDK;
import com.voxeet.sdk.core.abs.ConferenceService;
import com.voxeet.sdk.core.preferences.VoxeetPreferences;
import com.voxeet.sdk.core.services.AudioService;
import com.voxeet.sdk.core.services.LocalStatsService;
import com.voxeet.sdk.core.services.MediaService;
import com.voxeet.sdk.core.services.ScreenShareService;
import com.voxeet.sdk.core.services.UserService;
import com.voxeet.sdk.core.services.authenticate.WebSocketState;
import com.voxeet.sdk.core.services.authenticate.token.RefreshTokenCallback;
import com.voxeet.sdk.events.error.SocketConnectErrorEvent;
import com.voxeet.sdk.events.success.SocketConnectEvent;
import com.voxeet.sdk.events.success.SocketStateChangeEvent;
import com.voxeet.sdk.exceptions.ExceptionListener;
import com.voxeet.sdk.exceptions.ExceptionManager;
import com.voxeet.sdk.exceptions.VoxeetSentry;
import com.voxeet.sdk.factories.EventsFactory;
import com.voxeet.sdk.json.Event;
import com.voxeet.sdk.json.UserInfo;
import com.voxeet.sdk.utils.AbstractVoxeetEnvironmentHolder;

import org.greenrobot.eventbus.EventBus;
import org.greenrobot.eventbus.EventBusException;
import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.locks.ReentrantLock;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.ErrorPromise;
import eu.codlab.simplepromise.solve.PromiseExec;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;
import okhttp3.OkHttpClient;
import retrofit2.Retrofit;

public abstract class VoxeetSdkTemplate {
    static final long DEFAULT_TIMEOUT_MS = 60 * 1000;
    private static final String TAG = "VoxeetSdk";
    private UserInfo _user_info;
    private VoxeetWebSocket _voxeet_websocket;
    private VoxeetHttp _voxeet_http;

    protected Context _application_context;
    protected EventBus _event_bus;

    private AbstractVoxeetEnvironmentHolder _voxeet_environment_holder;

    private HashMap<Class<? extends AbstractVoxeetService>, AbstractVoxeetService> _services;
    private List<Solver<Boolean>> mWaitForLogInSocket;
    private ReentrantLock lockConnectAttempt = new ReentrantLock();
    private VoxeetSentry voxeetSentry;

    private VoxeetSdkTemplate() {

    }

    protected VoxeetSdkTemplate(@NonNull Application application_context,
                                @NonNull String appId,
                                @NonNull String password,
                                @NonNull UserInfo userInfo,
                                @NonNull AbstractVoxeetEnvironmentHolder holder,
                                boolean debug) {
        this(application_context, appId, password, userInfo, null, null, holder, debug);
    }

    protected VoxeetSdkTemplate(@NonNull Application application_context,
                                @NonNull String tokenAccess,
                                @Nullable RefreshTokenCallback tokenRefresh,
                                @NonNull UserInfo userInfo,
                                @NonNull AbstractVoxeetEnvironmentHolder holder,
                                boolean debug) {
        this(application_context, null, null, userInfo, tokenAccess, tokenRefresh, holder, debug);
    }

    private VoxeetSdkTemplate(@NonNull Application application_context,
                              @Nullable String appId,
                              @Nullable String password,
                              @NonNull UserInfo userInfo,
                              @Nullable String tokenAccess,
                              @Nullable RefreshTokenCallback tokenRefresh,
                              @NonNull AbstractVoxeetEnvironmentHolder holder,
                              boolean debug) {
        this();

        initExceptionIfOk(application_context);

        mWaitForLogInSocket = new ArrayList<>();
        _application_context = application_context;
        _event_bus = EventBus.getDefault();

        //make sure the looper is the main looper for promise execution
        Promise.setHandler(new Handler(Looper.getMainLooper()));

        _user_info = userInfo;

        Log.d("VoxeetSDK", "VoxeetSdkTemplate: version := " + holder.getVersionName());

        _voxeet_environment_holder = holder;

        _voxeet_websocket = new VoxeetWebSocket(this, _voxeet_environment_holder.getSocketUrl());

        VoxeetHttp.Builder builder = new VoxeetHttp.Builder()
                .setApplication(application_context)
                .setVoxeetSDK(this)
                .setServerUrl(_voxeet_environment_holder.getServerUrl())
                .setServerPort(_voxeet_environment_holder.getServerPort())
                .setAppId(appId)
                .setPassword(password)
                .setTokenAccess(tokenAccess)
                .setTokenRefresh(tokenRefresh)
                .setVoxeetServiceListener(new VoxeetHttp.VoxeetServiceListener() {

                    @Override
                    public void onIdentifySuccess(VoxeetHttp provider) {
                        _voxeet_websocket.connect(provider);
                    }

                    @Override
                    public void onIdentifyError(String error) {
                        if (null == error) error = "identfify error";
                        Log.e(getTag(), error);
                        _event_bus.post(new SocketStateChangeEvent(WebSocketState.CLOSED.name()));
                    }

                    @Override
                    public void onNetworkError(UnknownHostException error) {
                        _event_bus.post(new SocketStateChangeEvent(WebSocketState.CLOSED.name()));
                    }
                })
                .setDebug(debug);

        _voxeet_http = builder.build();


        _services = new HashMap<>();

        if (!_event_bus.isRegistered(this)) {
            _event_bus.register(this);
        }

        VoxeetPreferences.init(getApplicationContext(), getVoxeetEnvironmentHolder());
        initServices();
    }

    private void initExceptionIfOk(@NonNull Application application_context) {
        Log.d(TAG, "initExceptionIfOk: checking for exception to log internally");
        voxeetSentry = new VoxeetSentry(application_context);
        ExceptionManager.register(voxeetSentry);
        Log.d(TAG, "initExceptionIfOk: finished to try implementing error management");
    }

    public void resetVoxeetHttp() {
        _voxeet_http.resetVoxeetHttp();
    }

    @Deprecated
    public void logUser(@NonNull UserInfo userInfo) {
        _user_info = userInfo;

        //openSdkSession();
        logUserWithChain(userInfo).then(new PromiseExec<Boolean, Object>() {
            @Override
            public void onCall(@Nullable Boolean result, @NonNull Solver<Object> solver) {
                Log.d(TAG, "onCall: deprecated logUser method finished");
            }
        }).error(new ErrorPromise() {
            @Override
            public void onError(@NonNull Throwable error) {
                error.printStackTrace();
            }
        });
    }

    public Promise<Boolean> logUserWithChain(@NonNull UserInfo userInfo) {
        _user_info = userInfo;
        return logCurrentlySelectedUserWithChain();
    }

    public Promise<Boolean> logCurrentlySelectedUserWithChain() {
        if (isSocketOpen()) {
            return new Promise<Boolean>(new PromiseSolver<Boolean>() {
                @Override
                public void onCall(@NonNull Solver<Boolean> solver) {
                    Log.d(TAG, "onCall: socket opened");
                    lockConnectAttempt();
                    resolveLoginSockets();
                    unlockConnectAttempt();

                    solver.resolve(true);
                }
            });
        }

        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull Solver<Boolean> solver) {
                if (null != _voxeet_websocket && isSocketOpen()) {
                    _voxeet_websocket.close(true);
                }

                Log.d(TAG, "onCall: start login in promise");
                lockConnectAttempt();

                Log.d(TAG, "onCall: having the list with elements := " + mWaitForLogInSocket);
                // the current list if not empty, we make sure that we are not calling it more than one 1 time
                if (mWaitForLogInSocket.size() == 0) {
                    mWaitForLogInSocket.add(solver);

                    unlockConnectAttempt(); //unlock here not before
                    openSdkSessionChain().then(new PromiseExec<Boolean, Object>() {
                        @Override
                        public void onCall(@Nullable Boolean result, @NonNull Solver<Object> solver) {
                            //nothing, will be in the events
                            Log.d(TAG, "onCall: first login part done");
                        }
                    }).error(new ErrorPromise() {
                        @Override
                        public void onError(@NonNull Throwable error) {
                            Log.d(TAG, "onError: login error " + error.getMessage());
                            //solver.reject(error);

                            lockConnectAttempt();
                            rejectLoginSockets(error);
                            clearLoginSockets();
                            unlockConnectAttempt(); //unlock here not before
                            Log.d(TAG, "onError: login error managed");
                        }
                    });
                } else {
                    Log.d(TAG, "onCall: already have a login attempt in progress");
                    try {
                        throw new IllegalStateException("Can not open a session while an other one is trying to be started");
                    } catch (Exception e) {
                        solver.reject(e);
                    }
                    unlockConnectAttempt(); //unlock here not before

                    //nothing to do, it will be resolved later
                }
                Log.d(TAG, "onCall: start login done... can take a time");
            }
        });
    }

    protected abstract void initServices();

    protected HashMap<Class<? extends AbstractVoxeetService>, AbstractVoxeetService> getServices() {
        return _services;
    }

    public ConferenceService getConferenceService() {
        return getServiceForKlass(ConferenceService.class);
    }

    public ScreenShareService getScreenShareService() {
        return getServiceForKlass(ScreenShareService.class);
    }

    public AudioService getAudioService() {
        return getServiceForKlass(AudioService.class);
    }

    <T extends AbstractVoxeetService> T getServiceForKlass(Class<T> klass) {
        T service = null;
        Iterator<Class<? extends AbstractVoxeetService>> iterator = _services.keySet().iterator();
        while (iterator.hasNext()) {
            Class<? extends AbstractVoxeetService> next = iterator.next();
            AbstractVoxeetService item = _services.get(next);


            if (klass.isInstance(item)) {
                service = (T) item;
            }
        }

        if (service == null) {
            Log.d(TAG, klass.getSimpleName() + " not found in the list of services");
        }
        return service;
    }

    /**
     * Gets upload token. Won't be able to upload files and such without it
     */

    public void getUploadToken() {
        UserService service = getServiceForKlass(UserService.class);
        if (service != null) {
            service.getUploadToken();
        }
    }

    public Event decode(String message) {
        return EventsFactory.decode(message);
    }

    public Retrofit getRetrofit() {
        return _voxeet_http.getRetrofit();
    }

    public OkHttpClient getClient() {
        return _voxeet_http.getClient();
    }

    public Context getApplicationContext() {
        return _application_context;
    }

    AbstractVoxeetEnvironmentHolder getVoxeetEnvironmentHolder() {
        return _voxeet_environment_holder;
    }

    /**
     * Reset services.
     */
    public void resetServices() {
        Iterator<Class<? extends AbstractVoxeetService>> iterator = _services.keySet().iterator();
        while (iterator.hasNext()) {
            Class<? extends AbstractVoxeetService> next = iterator.next();
            AbstractVoxeetService service = _services.get(next);
            if (null != service) service.resetService();
        }
    }

    public EventBus getEventBus() {
        return _event_bus;
    }

    protected VoxeetWebSocket getSocket() {
        return _voxeet_websocket;
    }

    VoxeetHttp getVoxeetHttp() {
        return _voxeet_http;
    }

    public void closeSocket() {
        _voxeet_websocket.close(true);
    }

    public void cleanUserSession(@NonNull String id) {
        _voxeet_http.cleanUserSession(id);
    }

    public boolean isSocketOpen() {
        return _voxeet_websocket.isOpen();
    }

    public void openSdkSession() {
        _voxeet_http.identify(_user_info);
    }

    private Promise<Boolean> openSdkSessionChain() {
        return _voxeet_http.identifyChain(_user_info);
    }

    public UserInfo getUserInfo() {
        return _user_info;
    }

    protected abstract String getTag();

    /**
     * Register Media. To be called before creating/joining any conference.
     *
     * @param context the context
     */
    public boolean register(@NonNull Context context) {
        return register(context, null);
    }

    /**
     * Register Media. To be called before creating/joining any conference.
     *
     * @param context    the context
     * @param subscriber the subscriber
     */
    public boolean register(@NonNull Context context, @Nullable Object subscriber) {
        try {
            MediaSDK.setContext(context);

            if (null != subscriber && !getEventBus().isRegistered(subscriber))
                getEventBus().register(subscriber);
        } catch (EventBusException error) {
            //silent catch
        } catch (UnsatisfiedLinkError error) {
            return false;
        }

        return true;
    }

    /**
     * Unregister Media. To be called in the on destroy of your activty/fragment
     *
     * @param subscriber the subscriber
     */
    public void unregister(@NonNull Object subscriber) {

        if (getEventBus().isRegistered(subscriber))
            getEventBus().unregister(subscriber);
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(SocketConnectEvent event) {
        Log.d(TAG, "onEvent: ");
        lockConnectAttempt();
        resolveLoginSockets();
        unlockConnectAttempt(); //unlock here not before
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(SocketConnectErrorEvent event) {
        lockConnectAttempt();
        int i = 0;
        while (i < mWaitForLogInSocket.size()) {
            Log.d(TAG, "onEvent: calling resolved false");
            Solver<Boolean> solver = mWaitForLogInSocket.get(i);
            try {
                solver.resolve(false);
            } catch (Exception e) {
                e.printStackTrace();
            }
            i++;
        }

        clearLoginSockets();
        unlockConnectAttempt(); //unlock here not before
    }

    @NonNull
    public MediaService getMediaService() {
        return getServiceForKlass(MediaService.class);
    }

    @NonNull
    public LocalStatsService getLocalStatsService() {
        return getServiceForKlass(LocalStatsService.class);
    }

    void lockConnectAttempt() {
        try {
            lockConnectAttempt.lock();
        } catch (Exception e) {

        }
    }

    void unlockConnectAttempt() {
        try {
            if (lockConnectAttempt.isLocked()) {
                lockConnectAttempt.unlock();
            }
        } catch (Exception e) {

        }
    }

    void rejectLoginSockets(@Nullable Throwable error) {
        try {
            int i = 0;
            while (i < mWaitForLogInSocket.size()) {
                Log.d(TAG, "onError: calling reject");
                Solver<Boolean> solver = mWaitForLogInSocket.get(i);
                try {
                    solver.reject(error);
                } catch (Exception e) {
                    e.printStackTrace();
                }
                i++;
            }
        } catch (Exception err) {
            err.printStackTrace();
        }
    }

    private void resolveLoginSockets() {
        try {
            int i = 0;
            while (i < mWaitForLogInSocket.size()) {
                Log.d(TAG, "onEvent: calling resolved true");
                Solver<Boolean> solver = mWaitForLogInSocket.get(i);
                try {
                    solver.resolve(true);
                } catch (Exception e) {
                    e.printStackTrace();
                }
                i++;
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

        clearLoginSockets();
    }

    void clearLoginSockets() {
        try {
            mWaitForLogInSocket.clear();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}
