package com.voxeet.sdk.audio;

import android.content.Context;
import android.media.Ringtone;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.android.media.MediaPowerManager;
import com.voxeet.audio.AudioRoute;
import com.voxeet.audio.AudioStackManager;
import com.voxeet.audio.utils.Constants;
import com.voxeet.sdk.utils.AudioType;
import com.voxeet.sdk.utils.SoundPool;
import com.voxeet.sdk.utils.Validate;

import java.util.HashMap;
import java.util.List;

public class SoundManager {

    private AudioStackManager mAudioManager;
    private MediaPowerManager mMediaPowerManager;
    private Context mContext;
    private SoundPool _sound_pool;
    private HashMap<AudioType, String> mSounds;

    private boolean mEnable;

    private SoundManager() {
        enable();
    }

    public SoundManager(@NonNull Context context) {
        this();

        mAudioManager = new AudioStackManager(context);
        mAudioManager.configureVolumeStream(Constants.STREAM_VOICE_CALL,
                Constants.STREAM_MUSIC);
        mMediaPowerManager = new MediaPowerManager(context,
                currentRoute());

        mContext = context;

        mSounds = new HashMap<>();
        configure();
    }

    /**
     * Get the current available routes
     *
     * @return a non null route
     */
    @NonNull
    public List<AudioRoute> getAvailableRoutes() {
        return mAudioManager.availableRoutes();
    }

    public AudioRoute currentRoute() {
        return mAudioManager.outputRoute();
    }

    public boolean setAudioRoute(@NonNull AudioRoute route) {
        Validate.runningOnUiThread();
        return mAudioManager.setOutputRoute(route);
    }

    @Nullable
    public Ringtone getSystemRingtone() {
        return mAudioManager.getSystemRingtone();
    }

    public boolean isBluetoothHeadsetConnected() {
        return mAudioManager.isBluetoothHeadsetConnected();
    }

    public boolean isWiredHeadsetOn() {
        return mAudioManager.isWiredHeadsetOn();
    }

    public SoundManager setSpeakerMode(boolean isSpeaker) {
        mAudioManager.setSpeakerMode(isSpeaker);
        return this;
    }

    public boolean isSpeakerOn() {
        return AudioRoute.ROUTE_SPEAKER.equals(mAudioManager.outputRoute());
    }

    public SoundManager acquireLocks() {
        mMediaPowerManager.acquire();
        return this;
    }

    public SoundManager releaseLocks() {
        mMediaPowerManager.release();
        return this;
    }

    public SoundManager resetDefaultSoundType() {
        mAudioManager.resetDefaultSoundType();
        return this;
    }

    private int getUiSoundsStreamType() {
        return mAudioManager.getDefaultSoundStreamType();
    }

    private SoundManager forceVolumeControlStream(int type) {
        mAudioManager.forceVolumeControlStream(type);
        return this;
    }

    public SoundManager abandonAudioFocusRequest() {
        mAudioManager.abandonAudioFocus();
        return this;
    }

    public SoundManager requestAudioFocus() {
        mAudioManager.requestAudioFocus();
        return this;
    }

    public SoundManager checkOutputRoute() {
        mAudioManager.checkOutputRoute();
        return this;
    }

    private SoundPool getSoundPool() {
        if (_sound_pool == null) {
            _sound_pool = new SoundPool(mContext, Constants.STREAM_VOICE_CALL);
        }
        return _sound_pool;
    }

    private void configure() {
        //default enable the audio manager
        mAudioManager.enable();

        mSounds.put(AudioType.RING, "out.mp3");
        mSounds.put(AudioType.HANGUP, "leave.mp3");

        setSound(AudioType.RING, mSounds.get(AudioType.RING));
        setSound(AudioType.HANGUP, mSounds.get(AudioType.HANGUP));
    }

    public boolean setSound(@NonNull AudioType type, @NonNull String assetName) {
        return getSoundPool().release(type).setShortResource(type, assetName);
    }

    public SoundManager playSoundType(@NonNull AudioType type) {
        if (isEnabled()) {
            getSoundPool().playShortResource(type, mSounds.get(type));
        }
        return this;
    }

    public SoundManager playSoundTypeForce(@NonNull AudioType type) {
        getSoundPool().playShortResource(type, mSounds.get(type));
        return this;
    }

    public SoundManager stopSoundType(@NonNull AudioType audioType) {
        getSoundPool().stop(audioType);
        return this;
    }


    public SoundManager stop() {
        getSoundPool().stop();
        return this;
    }

    public SoundManager onConferencePreJoinedEvent() {
        mAudioManager.checkOutputRoute();
        return this;
    }

    public SoundManager onConferenceDestroyedPush() {
        abandonAudioFocusRequest();
        return this;
    }

    public SoundManager enable() {
        mEnable = true;
        if (null != mAudioManager) {
            mAudioManager.enable();
        }
        return this;
    }

    public SoundManager disable() {
        mEnable = false;
        if (null != mAudioManager) {
            mAudioManager.disable();
        }
        return this;
    }

    private boolean isEnabled() {
        return mEnable;
    }
}
