package voxeet.com.sdk.core;

import android.app.Application;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.ErrorPromise;
import eu.codlab.simplepromise.solve.PromiseExec;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;
import voxeet.com.sdk.core.impl.ConferenceSdkService;
import voxeet.com.sdk.core.services.AudioService;
import voxeet.com.sdk.core.services.LocalStatsService;
import voxeet.com.sdk.core.services.MediaService;
import voxeet.com.sdk.core.services.SDKFilePresentationService;
import voxeet.com.sdk.core.services.SDKVideoPresentationService;
import voxeet.com.sdk.core.services.ScreenShareService;
import voxeet.com.sdk.core.services.UserService;
import voxeet.com.sdk.core.services.authenticate.token.RefreshTokenCallback;
import voxeet.com.sdk.json.UserInfo;

/**
 * Created by kevinleperf on 14/11/2017.
 */

public final class VoxeetSdk extends VoxeetSdkTemplate<ConferenceSdkService> {

    private static VoxeetSdk CurrentInstance;
    private final static String TAG = VoxeetSdk.class.getSimpleName();

    protected static void setInstance(@NonNull VoxeetSdk sdk) {
        CurrentInstance = sdk;
    }

    public static VoxeetSdk getInstance() {
        return CurrentInstance;
    }

    /**
     * This function initializes the Voxeet SDK, the behavior of Voxeet SDK functions are
     * undetermined if this function is not called. It should be called as early as possible.
     *
     * @param application The application context
     * @param appId       the app id
     * @param password    the password
     * @param userInfo    the user info
     */
    public static synchronized void initialize(@NonNull final Application application,
                                               @NonNull String appId,
                                               @NonNull String password,
                                               @NonNull final UserInfo userInfo) {
        if (null == CurrentInstance) {
            VoxeetSdk sdk = new VoxeetSdk(application,
                    appId,
                    password,
                    userInfo);

            VoxeetSdk.setInstance(sdk);
            VoxeetSdkFactory.validate(sdk);
        } else {
            Log.d(TAG, "initialize: Instance already started !");
        }
    }

    /**
     * This function initializes the Voxeet SDK, the behavior of Voxeet SDK functions are
     * undetermined if this function is not called. It should be called as early as possible.
     * <p>
     * This method will invoke the SDK with the thirdparty authentication feature
     *
     * @param application  The application context
     * @param accessToken  the default accessToken to use
     * @param refreshToken the refreshToken callback
     * @param userInfo     the user info
     */
    public static synchronized void initialize(@NonNull final Application application,
                                               @NonNull String accessToken,
                                               @NonNull RefreshTokenCallback refreshToken,
                                               @NonNull final UserInfo userInfo) {
        if (null == CurrentInstance) {
            VoxeetSdk sdk = new VoxeetSdk(application,
                    accessToken,
                    refreshToken,
                    userInfo);

            VoxeetSdk.setInstance(sdk);
            VoxeetSdkFactory.validate(sdk);
        } else {
            Log.d(TAG, "initialize: Instance already started !");
        }
    }


    private VoxeetSdk(@NonNull Application application_context, @NonNull String appId, @NonNull String password, @NonNull UserInfo userInfo) {
        super(application_context, appId, password, userInfo, false);
    }

    private VoxeetSdk(@NonNull Application application_context, @NonNull String accessToken, @NonNull RefreshTokenCallback refreshToken, @NonNull UserInfo userInfo) {
        super(application_context, accessToken, refreshToken, userInfo, false);
    }

    @Override
    public Promise<Boolean> logout() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                Log.d(TAG, "onCall: logout called");
                onLogoutCleanLoginSockets();
                Log.d(TAG, "onCall: clean awaiting sockets done");
                getConferenceService().logout().then(new PromiseExec<Boolean, Object>() {
                    @Override
                    public void onCall(@Nullable Boolean result, @NonNull Solver<Object> s) {
                        Log.d(TAG, "onCall: logout result := " + result+" ... propagating...");
                        solver.resolve(result);
                    }
                }).error(new ErrorPromise() {
                    @Override
                    public void onError(@NonNull Throwable error) {
                        solver.reject(error);
                    }
                });
            }
        });
    }

    @Override
    public void onSdkPreinit() {
        //nothing here
    }

    @Override
    protected void initServices() {
        getServices().put(ConferenceSdkService.class, new ConferenceSdkService(this, DEFAULT_TIMEOUT_MS));
        getServices().put(UserService.class, new UserService(this));
        getServices().put(SDKFilePresentationService.class, new SDKFilePresentationService(this));
        getServices().put(SDKVideoPresentationService.class, new SDKVideoPresentationService(this));
        getServices().put(AudioService.class, new AudioService(this));
        getServices().put(ScreenShareService.class, new ScreenShareService(this));
        getServices().put(MediaService.class, new MediaService(this));
        getServices().put(LocalStatsService.class, new LocalStatsService(this));
    }

    public UserService getUserService() {
        return getServiceForKlass(UserService.class);
    }

    public SDKFilePresentationService getFilePresentationService() {
        return getServiceForKlass(SDKFilePresentationService.class);
    }

    public SDKVideoPresentationService getVideoPresentationService() {
        return getServiceForKlass(SDKVideoPresentationService.class);
    }

    @Override
    protected String getTag() {
        return VoxeetSdk.class.getSimpleName();
    }
}
