package com.voxeet.android.media;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import java.util.ArrayList;
import java.util.List;

/**
 * A Simple media stream for Voxeet
 *
 * @author tgourgues
 */
public class MediaStream {
    /**
     * The peer id of the stream owner
     */
    private String peerId;

    /**
     * The underlying stream label
     */
    private String label;

    /**
     * A list of AudioTracks
     */
    @NonNull
    private List<AudioTrack> audioTrackList = new ArrayList<>();

    /**
     * A list of VideoTracks
     */
    @NonNull
    private List<VideoTrack> videoTrackList = new ArrayList<>();

    /**
     * Flag describing a screen share stream
     */
    private boolean isScreenShare;

    /**
     * MediaStream Constructor
     * @param peerId - the peer identifier, valid
     * @param nativeMediaStream - the native pointer to the MediaStreamInterface for initialization (Will never be retained)
     * @param isScreenShare boolean indicating wether this stream is a screenshare
     */
    public MediaStream(@NonNull String peerId, long nativeMediaStream, boolean isScreenShare) {
        this.peerId = peerId;
        this.nativeInitMediaStream(nativeMediaStream);

        this.isScreenShare = isScreenShare;
    }

    /**
     * MediaStream Constructor
     * @param peerId - the peer identifier, valid
     * @param nativeMediaStream - the native pointer to the MediaStreamInterface for initialization (Will never be retained)
     */
    public MediaStream(String peerId, long nativeMediaStream) {
        this(peerId, nativeMediaStream, false);
    }

    private void nativeInitMediaStream(long nativeMediaStream) {
        long[] audioTracks = this.nativeGetAudioTracks(nativeMediaStream);
        long[] videoTracks = this.nativeGetVideoTracks(nativeMediaStream);

        if(null != audioTracks) {
            for (long ptr : audioTracks) {
                audioTrackList.add(new AudioTrack(ptr));
            }
        }

        if(null != videoTracks) {
            for (long ptr : videoTracks) {
                videoTrackList.add(new VideoTrack(ptr));
            }
        }
    }

    /**
     * Retrieve the hard list of AudioTrack
     * @return the list of audio tracks
     */
    @NonNull
    public List<AudioTrack> audioTracks() {
        return this.audioTrackList;
    }

    /**
     * Retrieve the hard list of VideoTrack
     * @return the list of video tracks
     */
    @NonNull
    public List<VideoTrack> videoTracks() {
        return this.videoTrackList;
    }

    /**
     * Get the peer identifier
     * @return a valid string
     */
    public String peerId() {
        return this.peerId;
    }

    /**
     * Get the label
     * @return the label written
     */
    public String label() {
        return this.label;
    }

    /**
     * Set the current MediaStream label
     *
     * Internal use only
     * @param label the new label of this MediaStream
     */
    public void label(@NonNull String label) {
        this.label = label;
    }

    /**
     * Is the current MediaStream a screen share
     * @return a boolean indicating the status, read only
     */
    public Boolean isScreenShare() {
        return isScreenShare;
    }

    /**
     *
     * @param nativeMediaStream pointer to the native mediaStream
     * @return list of audio tracks ids
     */
    @Nullable
    private native long[] nativeGetAudioTracks(long nativeMediaStream);

    /**
     *
     * @param nativeMediaStream pointer to the native mediaStream
     * @return list of video tracks ids
     */
    @Nullable
    private native long[] nativeGetVideoTracks(long nativeMediaStream);
}
