package com.voxeet.android.media;

import android.content.Context;
import android.content.Intent;
import android.media.projection.MediaProjection;
import android.os.Build;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.android.media.stats.LocalStats;
import com.voxeet.android.media.video.Camera1Enumerator;
import com.voxeet.android.media.video.Camera2Enumerator;
import com.voxeet.android.media.video.CameraEnumeratorInterface;
import com.voxeet.android.media.video.VideoCapturerConstraintsHolder;

import org.webrtc.CameraVideoCapturer;
import org.webrtc.EglBase;
import org.webrtc.ScreenCapturerAndroid;
import org.webrtc.SurfaceTextureHelper;
import org.webrtc.VideoCapturer;
import org.webrtc.VideoRenderer;
import org.webrtc.VideoStreamRenderer;
import org.webrtc.VideoTrackSourceObserver;
import org.webrtc.voiceengine.WebRtcAudioManager;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * Wrapper around CoreMedia C++
 *
 * @author Thomas Gourgues
 */
public abstract class MediaEngine {

    public static final VideoCapturerConstraintsHolder CAPTURER_CONSTRAINTS_HOLDER = new VideoCapturerConstraintsHolder();
    private static HandlerThread ThreadHandler;
    private CameraVideoCapturer.CameraEventsHandler mCameraListener;
    private Handler mHandler;

    /**
     * The WebRTC Stream listener
     */
    public interface StreamListener {
        void onStreamAdded(@NonNull String peer, @NonNull MediaStream stream);

        void onStreamUpdated(@NonNull String peer, @NonNull MediaStream stream);

        void onStreamRemoved(@NonNull String peer);

        void onScreenStreamAdded(@NonNull String peer, @NonNull MediaStream stream);

        void onScreenStreamRemoved(@NonNull String peer);

        void onShutdown();

        void onIceCandidateDiscovered(String peer, SdpCandidate[] candidates);
    }

    public static final String TAG = MediaEngine.class.getSimpleName();

    private static final String VIDEO_CAPTURER_THREAD_NAME = "VideoCapturerThread";

    public static Context Context;

    private static AtomicBoolean Registered = new AtomicBoolean(false);


    @NonNull
    protected StreamListener mStreamListener;

    private List<VideoStreamRenderer> mRenderers;
    private List<String> mScreenShareIdList = new ArrayList<>();


    private boolean isMuted = false;

    //Egl Context used by the various UI elements
    @NonNull
    private EglBase eglBase = EglBase.createEgl10(EglBase.CONFIG_PIXEL_BUFFER);

    private CameraVideoCapturer videoCapturer = null;
    private long capturerNativeSource = 0;

    private ScreenCapturerAndroid screenCapturer = null;
    private long screenNativeSource = 0;

    private CameraEnumeratorInterface cameraEnumerator;

    static {
        /** Loading dynamic jni lib */
        //System.loadLibrary("jingle_peerconnection_so");
        MediaEngine.loadLibrary("c++_shared");
        MediaEngine.loadLibrary("MediaEngineJni");
    }

    private static void loadLibrary(@NonNull String path) {
        try {
            System.loadLibrary(path);
        } catch (Exception e) {
            Log.e(TAG, e.getMessage());
        } catch (UnsatisfiedLinkError e) {
            Log.d(TAG, "unsupported device architecture");
            Log.e(TAG, e.getMessage());
        }
    }

    private MediaEngine() {
        mRenderers = new ArrayList<>();

        resetEglContext();


        CAPTURER_CONSTRAINTS_HOLDER.setWidth(640)
                .setHeight(480)
                .setFrameRate(25);
    }

    public MediaEngine(@NonNull final Context context,
                       @NonNull final String peer,
                       @NonNull StreamListener streamListener,
                       @NonNull CameraVideoCapturer.CameraEventsHandler cameraListener,
                       boolean startWithVideo,
                       boolean microphone) throws MediaEngineException {
        this();
        MediaEngine.Context = context;
        WebRtcAudioManager.setStereoOutput(true);
        WebRtcAudioManager.setBlacklistDeviceForOpenSLESUsage(true);

        MediaEngine.Context = context;
        mStreamListener = streamListener;
        mCameraListener = cameraListener;

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            this.cameraEnumerator = new Camera2Enumerator(context);
        } else {
            this.cameraEnumerator = new Camera1Enumerator(context);
        }

        if (startWithVideo) {
            String name = this.cameraEnumerator.getNameOfFrontFacingDevice();
            //fallback to the back camera in case of any issue in specific devices
            if (null == name) name = this.cameraEnumerator.getNameOfBackFacingDevice();

            videoCapturer = createVideoCapturer(name, mCameraListener);
        }

        if (null == ThreadHandler) {
            ThreadHandler = new HandlerThread("SDK");
            ThreadHandler.start();
        }

        mHandler = new Handler(ThreadHandler.getLooper());

        CountDownLatch latch = new CountDownLatch(1);

        mHandler.post(new Runnable() {
            @Override
            public void run() {
                if (!Init(context, peer, eglBase.getEglBaseContext(), videoCapturer)) {
                    try {
                        throw new MediaEngineException("An error occurred during init");
                    } catch (MediaEngineException e) {
                        e.printStackTrace();
                    }
                }
                latch.countDown();
            }
        });

        waitFor(latch);
    }

    private void resetEglContext() {
        Log.d(TAG, "resetEglContext: resetting");
        try {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2) {
                eglBase = EglBase.createEgl14(EglBase.CONFIG_PIXEL_BUFFER);
            } else {
                eglBase = EglBase.createEgl10(EglBase.CONFIG_PIXEL_BUFFER);
            }
        } catch (Exception e) {
            Log.d(TAG, "MediaEngine: error while initializing EGL14. Should not happen, please report");
            e.printStackTrace();
        }

        onEglBaseRecreated(eglBase);
    }

    protected abstract void onEglBaseRecreated(@NonNull EglBase base);

    public void stop() {
        CountDownLatch latch = new CountDownLatch(1);
        mStreamListener.onShutdown();

        stopScreenCapturer();
        stopVideo();

        mHandler.post(new Runnable() {
            @Override
            public void run() {
                Release();

                latch.countDown();
            }
        });

        waitFor(latch);
    }

    public void startVideo() {
        String name = this.cameraEnumerator.getNameOfFrontFacingDevice();
        //fallback to the back camera in case of any issue in specific devices
        if (null == name) this.cameraEnumerator.getNameOfBackFacingDevice();

        if (null != name) {
            startVideo(name);
        }
    }

    public void startScreenCapturer(Intent mediaProjectionPermissionResultData, int width, int height) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP && null == screenCapturer) {
            screenCapturer = new ScreenCapturerAndroid(mediaProjectionPermissionResultData,
                    new MediaProjection.Callback() {
                        @Override
                        public void onStop() {
                            super.onStop();

                            stopScreenCapturer();
                        }
                    });

            initVideoCapturer(screenCapturer, true, width, height);
            StartScreenShare(screenNativeSource);
        }
    }


    public void stopScreenCapturer() {
        if (null != screenCapturer) {
            try {
                screenCapturer.stopCapture();
            } catch (Exception e) {
                e.printStackTrace();
            }

            StopScreenShare();
            screenCapturer.dispose();
            screenCapturer = null;
            screenNativeSource = 0;
        }
    }


    /**
     * Taken from the PeerConnectionFactory implementation
     *
     * @param capturer a valid capture
     */
    private void initVideoCapturer(@NonNull VideoCapturer capturer, boolean isScreenShare, int width, int height) {
        final EglBase.Context eglContext = eglBase.getEglBaseContext();

        //Android Video Track Source
        VideoCapturer.CapturerObserver capturerObserver;
        SurfaceTextureHelper surfaceTextureHelper = SurfaceTextureHelper.create(VIDEO_CAPTURER_THREAD_NAME, eglContext);

        if(null == surfaceTextureHelper) {
            resetEglContext();
            surfaceTextureHelper = SurfaceTextureHelper.create(VIDEO_CAPTURER_THREAD_NAME, eglContext);
        }

        if (!isScreenShare) {
            this.capturerNativeSource = CreateVideoSource(surfaceTextureHelper, false);
            capturerObserver = new VideoTrackSourceObserver(this.capturerNativeSource);
        } else {
            this.screenNativeSource = CreateVideoSource(surfaceTextureHelper, false);
            capturerObserver = new VideoTrackSourceObserver(this.screenNativeSource);
        }

        capturer.initialize(surfaceTextureHelper, MediaEngine.Context, capturerObserver);
        capturer.startCapture(width, height, CAPTURER_CONSTRAINTS_HOLDER.getFrameRate());
    }

    private CameraVideoCapturer createVideoCapturer(@NonNull String cameraName, CameraVideoCapturer.CameraEventsHandler cameraListener) {
        CameraVideoCapturer capturer = this.cameraEnumerator.createCapturer(cameraName, cameraListener);
        initVideoCapturer(capturer, false, CAPTURER_CONSTRAINTS_HOLDER.getWidth(), CAPTURER_CONSTRAINTS_HOLDER.getHeight());
        return capturer;
    }

    /**
     * Start Video using the available camera
     * <p>
     * If the requested camera does not exists, it will try to open the other ones
     *
     * @param cameraName the name, should be given by the system
     */
    public void startVideo(@Nullable String cameraName) {
        if (videoCapturer == null) {

            if (null == cameraName) {
                cameraName = this.cameraEnumerator.getNameOfFrontFacingDevice();
                //fallback to the back camera in case of any issue in specific devices
                if (null == cameraName)
                    cameraName = this.cameraEnumerator.getNameOfBackFacingDevice();
            }

            videoCapturer = createVideoCapturer(cameraName, mCameraListener);

            if (capturerNativeSource != 0) {
                StartVideo(capturerNativeSource);
            }
        }
    }

    public void stopVideo() {
        if (videoCapturer != null) {
            try {
                videoCapturer.stopCapture();
            } catch (InterruptedException e) {
                Log.e(TAG, e.getMessage());
            }

            StopVideo();
            videoCapturer.dispose();
            videoCapturer = null;
            capturerNativeSource = 0;
        }
    }

    public void switchCamera(@Nullable CameraVideoCapturer.CameraSwitchHandler handler) {
        if (videoCapturer != null) {
            videoCapturer.switchCamera(handler);
        }
    }

    public void attachMediaStream(VideoRenderer.Callbacks callbacks, MediaStream stream) {
        VideoStreamRenderer renderer = null;
        for (VideoStreamRenderer r : mRenderers) {
            if (r.callbacks() == callbacks) {
                renderer = r;
                break;
            }
        }

        if (renderer == null) {
            renderer = new VideoStreamRenderer(callbacks);
            mRenderers.add(renderer);
        }

        AttachMediaStream(stream.peerId(), stream.label(), renderer.nativeVideoRenderer());
    }

    public void unattachMediaStream(VideoRenderer.Callbacks callbacks, MediaStream stream) {
        VideoStreamRenderer renderer = null;
        for (VideoStreamRenderer r : mRenderers) {
            if (r.callbacks() == callbacks) {
                renderer = r;
                break;
            }
        }

        if (renderer != null) {
            if (stream != null) {
                UnAttachMediaStream(stream.peerId(), stream.label(), renderer.nativeVideoRenderer());
                //TODO: Check wether we should do renderer.dispose here (maybe better at the end of the media)
            }
        }
    }

    public void removePeer(@Nullable final String peer) {
        Log.d(TAG, "removePeer: trying to remove peer " + peer +
                Looper.myLooper() + " " + Looper.getMainLooper());
        if (null != peer && !ClosePeerConnection(peer)) {
            Log.w(TAG, String.format("Unable to close peer connection for peer: %s", peer));
        }
    }

    public void changePeerPosition(String peer, double angle, double distance) {
        if (!SetPeerPosition(peer, angle, distance)) {
            Log.w(TAG, String.format("Unable to change position for peer: %s", peer));
        }
    }

    public void changePeerPosition(String peer, double angle, double distance, float gain) {
      /*  if (!SetPeerPositionGain(peer, angle, distance, gain)) {
            Log.w(TAG, String.format("Unable to change position for peer: %s", peer));
        }
        */
    }

    public void changePeerGain(String peer, float gain) {
        /*if (SetPeerGain(peer, gain)) {
            Log.w(TAG, String.format("Unable to change gain for peer: {0}", peer));
        }*/
    }

    /**
     * Mute the current audio recording
     */
    public void mute() {
        isMuted = true;
        SetMute(true);
    }

    /**
     * Unmute the current audio recording
     */
    public void unMute() {
        isMuted = false;
        SetMute(false);
    }

    public boolean isMuted() {
        return isMuted;
    }

    public int getLocalVuMeter() {
        return GetLocalVuMeter();
    }

    public double getPeerVuMeter(@NonNull String peer) {
        return GetPeerVuMeter(peer);
    }

    @NonNull
    public LocalStats getLocalStats(@NonNull String peer) {
        return GetLocalStats(peer);
    }

    public boolean createPeerConnection(final String peer, final boolean master) throws MediaEngineException {
        return CreatePeerConnection(peer, master);
    }

    public boolean createAnswer(@NonNull String peer) {
        return CreateAnswer(peer);
    }

    public boolean setPeerDescription(@NonNull String peer,
                                      long ssrc,
                                      @NonNull String type,
                                      @NonNull String sdp) {
        return SetPeerDescription(peer, ssrc, type, sdp);
    }

    public boolean setPeerCandidate(@NonNull String peer,
                                    @NonNull String sdpMid,
                                    int sdpMLineIndex,
                                    @NonNull String sdp) {
        return SetPeerCandidate(peer, sdpMid, sdpMLineIndex, sdp);
    }

    //-------------------------------------------------------------------------------------------------
    // Callbacks
    //-------------------------------------------------------------------------------------------------

    public abstract void onSessionCreated(String peer, String type, String sdp);

    public abstract void onIceCandidateDiscovered(String peer, SdpCandidate[] candidates);

    public abstract void onIceGatheringComplete(String peer);

    public void onStreamAdded(String peer, long nativeStream) {
        Log.d(TAG, "Java stream added");
        MediaStream stream = new MediaStream(peer, nativeStream);
        mStreamListener.onStreamAdded(peer, stream);
    }

    public void onStreamUpdated(String peer, long nativeStream) {
        Log.d(TAG, "Java stream updated");
        MediaStream stream = new MediaStream(peer, nativeStream);

        mStreamListener.onStreamUpdated(peer, stream);
    }

    public void onStreamRemoved(String peer, long nativeStream) {
        Log.d(TAG, "Java stream removed");

        MediaStream stream = new MediaStream(peer, nativeStream);
        mStreamListener.onStreamRemoved(peer);
    }

    public void onScreenStreamAdded(String peer, long nativeStream) {
        Log.d(TAG, "Java stream added corresponding to screen");

        MediaStream stream = new MediaStream(peer, nativeStream, true);
        mStreamListener.onScreenStreamAdded(peer, stream);
    }

    public void onScreenStreamRemoved(String peer, long nativeStream) {
        Log.d(TAG, "Java stream removed corresponding to screen");

        MediaStream stream = new MediaStream(peer, nativeStream, true);
        mStreamListener.onScreenStreamRemoved(peer);
    }

    @NonNull
    public EglBase getEglBase() {
        return eglBase;
    }

    /**
     * Default method to wait for a given latch to finish
     * <p>
     * It will for 5s maximum
     * Methods used in the SDK
     *
     * @param latch a valid latch
     */
    private void waitFor(@NonNull CountDownLatch latch) {
        try {
            latch.await(5, TimeUnit.SECONDS);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
    }

    //-------------------------------------------------------------------------------------------------
    // JNI Native methods
    //-------------------------------------------------------------------------------------------------

    //public static native void Register(Context context);
    //public static native void UnRegister();

    private native boolean Init(@NonNull Context context, @NonNull String peer, @NonNull EglBase.Context eglContext, @NonNull VideoCapturer capturer);

    private native void Release();

    private native void StartVideo(long nativeSource);

    private native void StopVideo();

    private native void StartScreenShare(long nativeSource);

    private native void StopScreenShare();

    private native long CreateVideoSource(@NonNull SurfaceTextureHelper helper, boolean isScreencast);

    private native void AttachMediaStream(@NonNull String peer, @NonNull String stream, long nativeRenderer);

    private native void UnAttachMediaStream(@NonNull String peer, @NonNull String stream, long nativeRenderer);

    private native boolean CreatePeerConnection(@NonNull String peer, boolean master);

    private native boolean ClosePeerConnection(@NonNull String peer);

    private native boolean CreateAnswer(@NonNull String peer);

    private native boolean SetPeerDescription(@NonNull String peer, long ssrc, @NonNull String type, @NonNull String sdp);

    private native boolean SetPeerCandidate(@NonNull String peer, @NonNull String sdpMid, int sdpMLineIndex, @NonNull String sdp);

    private native boolean SetPeerPosition(@NonNull String peer, double angle, double position);

    private native void SetMute(boolean mute);

    private native int GetLocalVuMeter();

    private native double GetPeerVuMeter(@NonNull String peer);

    @NonNull
    private native LocalStats GetLocalStats(@NonNull String peer);
}
