package com.voxeet.android.media.audio;

import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothHeadset;
import android.bluetooth.BluetoothProfile;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.support.annotation.NonNull;
import android.util.Log;

import java.util.ArrayList;
import java.util.List;

/**
 * Control and Manager the Audio states of the media
 */

public class AudioManager {
    public static final String MEDIA_OUPUT_ROUTE_INTENT = "media-output-route-intent";
    private static final String TAG = AudioManager.class.getSimpleName();
    private Context mContext;
    private boolean isBluetoothScoStarted;

    private android.media.AudioManager mServiceAudioManager;

    private HeadsetStateReceiver mHeadsetStateReceiver;
    private BluetoothHeadset mCurrentBluetoothHeadset;

    private BluetoothHeadsetListener mBluetoothHeadsetListener;
    private BluetoothAdapter mBluetoothAdapter;

    private AudioRoute mOutputRoute = AudioRoute.ROUTE_PHONE;
    private List<IMediaStateListener> mMediaStateListeners;


    private class BluetoothHeadsetListener implements BluetoothHeadset.ServiceListener {
        @Override
        public void onServiceConnected(int i, BluetoothProfile bluetoothProfile) {
            mCurrentBluetoothHeadset = (BluetoothHeadset) bluetoothProfile;
            for (IMediaStateListener listeners : mMediaStateListeners) {
                listeners.onBluetoothHeadsetStateChange(true);
            }

            checkOutputRoute();
            //checkProximitySensor(); // Optional here
        }

        @Override
        public void onServiceDisconnected(int i) {
            mCurrentBluetoothHeadset = null;

            for (IMediaStateListener listeners : mMediaStateListeners) {
                listeners.onBluetoothHeadsetStateChange(false);
            }
            mServiceAudioManager.stopBluetoothSco();
            isBluetoothScoStarted = false;

            checkOutputRoute();
            //checkProximitySensor();
        }
    }

    private class HeadsetStateReceiver extends BroadcastReceiver {
        @Override
        public void onReceive(Context context, Intent intent) {
            if (Intent.ACTION_HEADSET_PLUG.equals(intent.getAction())) {
                int state = intent.getIntExtra("state", -1);
                int has_mic = intent.getIntExtra("microphone", -1);

                boolean isPlugged = state == 1 && has_mic == 1;

                if (isPlugged) {
                    for (IMediaStateListener listeners : mMediaStateListeners) {
                        listeners.onHeadsetStateChange(true);
                    }
                    setSpeakerMode(false);
                } else {
                    for (IMediaStateListener listeners : mMediaStateListeners) {
                        listeners.onHeadsetStateChange(false);
                    }
                    setSpeakerMode(true);
                }
            }
        }
    }

    private AudioManager() {

    }

    public AudioManager(Context context) {
        this();

        mContext = context;
        mMediaStateListeners = new ArrayList<>();

        mHeadsetStateReceiver = new HeadsetStateReceiver();
        mBluetoothHeadsetListener = new BluetoothHeadsetListener();

        mCurrentBluetoothHeadset = null;
        isBluetoothScoStarted = false;

        mBluetoothAdapter = BluetoothAdapter.getDefaultAdapter();
        if (mBluetoothAdapter != null) {
            mBluetoothAdapter.getProfileProxy(context, mBluetoothHeadsetListener, BluetoothProfile.HEADSET);
        }
        this.mServiceAudioManager = (android.media.AudioManager) context.getSystemService(Context.AUDIO_SERVICE);

        context.registerReceiver(mHeadsetStateReceiver, new IntentFilter(Intent.ACTION_HEADSET_PLUG));
    }

    public void stop() {
        setBluetooth(false);

        mContext.unregisterReceiver(mHeadsetStateReceiver);

        if (mCurrentBluetoothHeadset != null && mBluetoothAdapter != null) {
            mBluetoothAdapter.closeProfileProxy(BluetoothProfile.HEADSET, mCurrentBluetoothHeadset);
        }
    }

    public boolean isBluetoothHeadsetOn() {
        return mCurrentBluetoothHeadset != null;
    }

    public boolean isWiredHeadsetOn() {
        return mServiceAudioManager.isWiredHeadsetOn();
    }

    /**
     * Get the current available routes
     *
     * @return a non null route
     */
    @NonNull
    public List<AudioRoute> availableRoutes() {
        List<AudioRoute> routes = new ArrayList<>();

        routes.add(AudioRoute.ROUTE_SPEAKER);

        if (isWiredHeadsetOn()) {
            routes.add(AudioRoute.ROUTE_HEADSET);
        } else {
            routes.add(AudioRoute.ROUTE_PHONE);
        }

        if (isBluetoothHeadsetOn()) {
            routes.add(AudioRoute.ROUTE_BLUETOOTH);
        }

        return routes;
    }

    /**
     * Retrieve the current audio route defined in this manager
     *
     * @return a non null audio route
     */
    @NonNull
    public AudioRoute outputRoute() {
        return mOutputRoute;
    }


    /**
     * Set the current route for the manager
     *
     * @param route
     */
    public void setOutputRoute(@NonNull AudioRoute route) {
        if (null == route) return;

        switch (route) {
            case ROUTE_HEADSET:
            case ROUTE_PHONE:
                setSpeakerMode(false);
                break;
            case ROUTE_SPEAKER:
                setSpeakerMode(true);
                break;
            case ROUTE_BLUETOOTH:
                setBluetooth(true);
                break;
            default:
                break;
        }
    }

    public void setSpeakerPhoneMode(boolean mode) {
        mServiceAudioManager.setSpeakerphoneOn(mode);

        if (mode) {
            setBluetooth(false);
        }

        mServiceAudioManager.setSpeakerphoneOn(mode);
        mServiceAudioManager.setMode(android.media.AudioManager.MODE_NORMAL);

        checkOutputRoute();
        //checkProximitySensor();
    }

    public void setSpeakerMode(boolean speakerMode) {
        setBluetooth(false); // Ensure that bluetooth is off

        //ResetAudioDevice();
        if ("samsung".equalsIgnoreCase(android.os.Build.BRAND)) {
            int requiredSpeakerMode = android.media.AudioManager.MODE_IN_COMMUNICATION;

            if (speakerMode) {
                // route audio to back speaker
                mServiceAudioManager.setSpeakerphoneOn(true);
                mServiceAudioManager.setMode(android.media.AudioManager.MODE_CURRENT);
            } else {
                // route audio to earpiece
                mServiceAudioManager.setSpeakerphoneOn(speakerMode);
                if (mServiceAudioManager.isWiredHeadsetOn()) {
                    mServiceAudioManager.setMode(android.media.AudioManager.MODE_CURRENT);
                } else {
                    mServiceAudioManager.setMode(requiredSpeakerMode);
                }
            }
        } else {
            // Non-Samsung devices
            mServiceAudioManager.setSpeakerphoneOn(speakerMode);
        }


        for (IMediaStateListener listeners : mMediaStateListeners) {
            listeners.onSpeakerChanged(speakerMode);
        }

        checkOutputRoute();
        //checkProximitySensor();
    }

    /**
     * Start the audio manager in bluetooth mode
     * <p>
     * Can lead to a non-bluetooth state if a crash occured internally (android 5.0)
     *
     * @param isEnabled true if it should start
     */
    public void setBluetooth(boolean isEnabled) {
        try {
            if (isEnabled) {
                if (!isBluetoothScoStarted) {
                    mServiceAudioManager.startBluetoothSco();
                    isBluetoothScoStarted = true;
                }
            } else {
                if (isBluetoothScoStarted) {
                    mServiceAudioManager.stopBluetoothSco();
                    isBluetoothScoStarted = false;
                }
            }
        } catch (NullPointerException e) { // Workaround for lollipop 5.0
            Log.d(TAG, "No bluetooth headset connected");
        }

        checkOutputRoute();
        //checkProximitySensor();
    }

    /**
     * Register a valid listener to this manager
     *
     * @param listener a persisted listener
     */
    public void registerMediaState(@NonNull IMediaStateListener listener) {
        if (!mMediaStateListeners.contains(listener)) {
            mMediaStateListeners.add(listener);
        }
    }

    /**
     * Remove a listener from the list of listeners
     *
     * @param listener a valid listener, if not currently listening, no crash reported
     */
    public void unregisterMediaState(@NonNull IMediaStateListener listener) {
        if (mMediaStateListeners.contains(listener)) {
            mMediaStateListeners.remove(listener);
        }
    }

    protected void checkOutputRoute() {
        if (isBluetoothScoStarted) {
            mOutputRoute = AudioRoute.ROUTE_BLUETOOTH;
        } else if (mServiceAudioManager.isSpeakerphoneOn()) {
            mOutputRoute = AudioRoute.ROUTE_SPEAKER;
        } else {
            if (mServiceAudioManager.isWiredHeadsetOn()) {
                mOutputRoute = AudioRoute.ROUTE_HEADSET;
            } else {
                mOutputRoute = AudioRoute.ROUTE_PHONE;
            }
        }

        Intent intent = new Intent(MEDIA_OUPUT_ROUTE_INTENT);
        intent.putExtra("route", mOutputRoute.value());

        mContext.sendBroadcast(intent);
    }
}
