package com.voxeet.android.media;

import android.content.Context;
import android.content.Intent;
import android.media.projection.MediaProjection;
import android.os.Build;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.android.media.video.CameraEnumerator;

import org.webrtc.CameraVideoCapturer;
import org.webrtc.EglBase;
import org.webrtc.ScreenCapturerAndroid;
import org.webrtc.SurfaceTextureHelper;
import org.webrtc.VideoCapturer;
import org.webrtc.VideoRenderer;
import org.webrtc.VideoStreamRenderer;
import org.webrtc.VideoTrackSourceObserver;
import org.webrtc.voiceengine.WebRtcAudioManager;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * Wrapper around CoreMedia C++
 *
 * @author Thomas Gourgues
 */
public abstract class MediaEngine {

    /**
     * The WebRTC Stream listener
     */
    public interface StreamListener {
        void onStreamAdded(@NonNull String peer, @NonNull MediaStream stream);
        void onStreamUpdated(@NonNull String peer, @NonNull MediaStream stream);
        void onStreamRemoved(@NonNull String peer);
        void onScreenStreamAdded(@NonNull String peer, @NonNull MediaStream stream);
        void onScreenStreamRemoved(@NonNull String peer);
        void onShutdown();
    }

    public static final String TAG = MediaEngine.class.getSimpleName();

    private static final String VIDEO_CAPTURER_THREAD_NAME = "VideoCapturerThread";

    public static Context Context;

    private static AtomicBoolean Registered = new AtomicBoolean(false);


    @NonNull
    private StreamListener mStreamListener;

    private List<VideoStreamRenderer> mRenderers;
    private VideoRenderer screenShareVideoRenderer = null;


    private boolean isMuted = false;

    // Camera variables
    @NonNull
    private EglBase eglBase = EglBase.create();//was eglbase14 but more options

    private CameraVideoCapturer videoCapturer = null;
    private long capturerNativeSource = 0;

    private ScreenCapturerAndroid screenCapturer = null;
    private long screenNativeSource = 0;

    private CameraEnumerator cameraEnumerator;

    static {
        /** Loading dynamic jni lib */
        try {
            //System.loadLibrary("jingle_peerconnection_so");
            System.loadLibrary("MediaEngineJni");
        } catch (Exception e) {
            Log.e(TAG, e.getMessage());
        } catch (UnsatisfiedLinkError e) {
            Log.d(TAG, "unsupported device architecture");
            Log.e(TAG, e.getMessage());
        }
    }

    private MediaEngine() {
        mRenderers = new ArrayList<>();
    }

    public MediaEngine(@NonNull final Context context,
                       @NonNull final String peer,
                       @NonNull StreamListener streamListener,
                       boolean startWithVideo,
                       boolean microphone) throws MediaEngineException {
        this();
        MediaEngine.Context = context;
        WebRtcAudioManager.setStereoOutput(true);
        WebRtcAudioManager.setBlacklistDeviceForOpenSLESUsage(true);

        MediaEngine.Context = context;
        mStreamListener = streamListener;

        this.cameraEnumerator = new CameraEnumerator(context);

        if (startWithVideo) {
            String name = this.cameraEnumerator.getNameOfFrontFacingDevice();
            //fallback to the back camera in case of any issue in specific devices
            if (null == name) name = this.cameraEnumerator.getNameOfBackFacingDevice();

            videoCapturer = createVideoCapturer(name);
        }

        if (!Init(context, peer, eglBase.getEglBaseContext(), videoCapturer)) {
            throw new MediaEngineException("An error occurred during init");
        }
    }

    public void stop() {
        mStreamListener.onShutdown();

        stopScreenCapturer();
        stopVideo();

        Release();
    }

    public void startVideo() {
        String name = this.cameraEnumerator.getNameOfFrontFacingDevice();
        //fallback to the back camera in case of any issue in specific devices
        if (null == name) this.cameraEnumerator.getNameOfBackFacingDevice();

        if (null != name) {
            startVideo(name);
        }
    }

        public void startScreenCapturer(Intent mediaProjectionPermissionResultData) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP && null == screenCapturer) {
            screenCapturer = new ScreenCapturerAndroid(mediaProjectionPermissionResultData,
                    new MediaProjection.Callback() {
                        @Override
                        public void onStop() {
                            super.onStop();


                            stopScreenCapturer();
                        }
                    });

            initVideoCapturer(screenCapturer, true);
            StartScreenShare(screenNativeSource);
        }
    }


    public void stopScreenCapturer() {
        if (null != screenCapturer) {
            try {
                screenCapturer.stopCapture();
            } catch (Exception e) {
                e.printStackTrace();
            }

            StopScreenShare();
            screenCapturer.dispose();
            screenCapturer = null;
            screenNativeSource = 0;
        }
    }


    /**
     * Taken from the PeerConnectionFactory implementation
     *
     * @param capturer a valid capture
     */
    private void initVideoCapturer(@NonNull VideoCapturer capturer, boolean isScreenShare) {
        final EglBase.Context eglContext = eglBase.getEglBaseContext();

        final SurfaceTextureHelper surfaceTextureHelper = SurfaceTextureHelper.create(VIDEO_CAPTURER_THREAD_NAME, eglContext);
        //Android Video Track Source
        VideoCapturer.CapturerObserver capturerObserver;

        if (!isScreenShare) {
            this.capturerNativeSource = CreateVideoSource(surfaceTextureHelper, false);
            capturerObserver = new VideoTrackSourceObserver(this.capturerNativeSource);
        } else {
            this.screenNativeSource = CreateVideoSource(surfaceTextureHelper, false);
            capturerObserver = new VideoTrackSourceObserver(this.screenNativeSource);
        }

        capturer.initialize(surfaceTextureHelper, MediaEngine.Context, capturerObserver);
        capturer.startCapture(640, 480, 25);
    }

    private CameraVideoCapturer createVideoCapturer(@NonNull String cameraName) {
        CameraVideoCapturer capturer = this.cameraEnumerator.createCapturer(cameraName, null);
        initVideoCapturer(capturer, false);
        return capturer;
    }


    public void startVideo(@NonNull String cameraName) {
        if (videoCapturer == null) {
            videoCapturer = createVideoCapturer(cameraName);

            if (capturerNativeSource != 0) {
                StartVideo(capturerNativeSource);
            }
        }
    }

    public void stopVideo() {
        if (videoCapturer != null) {
            try {
                videoCapturer.stopCapture();
            } catch (InterruptedException e) {
                Log.e(TAG, e.getMessage());
            }

            StopVideo();
            videoCapturer.dispose();
            videoCapturer = null;
            capturerNativeSource = 0;
        }
    }

    public void switchCamera(@Nullable CameraVideoCapturer.CameraSwitchHandler handler) {
        if (videoCapturer != null) {
            videoCapturer.switchCamera(handler);
        }
    }

    public void attachMediaStream(VideoRenderer.Callbacks callbacks, MediaStream stream) {

        VideoStreamRenderer renderer = null;
        for (VideoStreamRenderer r : mRenderers) {
            if (r.callbacks() == callbacks) {
                renderer = r;
                break;
            }
        }

        if (renderer == null) {
            renderer = new VideoStreamRenderer(callbacks);
            mRenderers.add(renderer);
        }

        AttachMediaStream(stream.peerId(), stream.label(), renderer.nativeVideoRenderer());
    }

    public void unattachMediaStream(VideoRenderer.Callbacks callbacks, MediaStream stream) {
        VideoStreamRenderer renderer = null;
        for (VideoStreamRenderer r : mRenderers) {
            if (r.callbacks() == callbacks) {
                renderer = r;
                break;
            }
        }

        if (renderer != null) {
            if (stream != null) {
                UnAttachMediaStream(stream.peerId(), stream.label(), renderer.nativeVideoRenderer());
                //TODO: Check wether we should do renderer.dispose here (maybe better at the end of the media)
            }
        }
    }

    public void removePeer(final String peer) {
        if (!ClosePeerConnection(peer)) {
            Log.w(TAG, String.format("Unable to close peer connection for peer: %s", peer));
        }
    }

    public void changePeerPosition(String peer, double angle, double distance) {
        if (!SetPeerPosition(peer, angle, distance)) {
            Log.w(TAG, String.format("Unable to change position for peer: %s", peer));
        }
    }

    public void changePeerPosition(String peer, double angle, double distance, float gain) {
      /*  if (!SetPeerPositionGain(peer, angle, distance, gain)) {
            Log.w(TAG, String.format("Unable to change position for peer: %s", peer));
        }
        */
    }

    public void changePeerGain(String peer, float gain) {
        /*if (SetPeerGain(peer, gain)) {
            Log.w(TAG, String.format("Unable to change gain for peer: {0}", peer));
        }*/
    }

    /**
     * Mute the current audio recording
     */
    public void mute() {
        isMuted = true;
        SetMute(true);
    }

    /**
     * Unmute the current audio recording
     */
    public void unMute() {
        isMuted = false;
        SetMute(false);
    }

    public boolean isMuted() {
        return isMuted;
    }

    public int getLocalVuMeter() {
        return GetLocalVuMeter();
    }

    public int getPeerVuMeter(String peer) {
        return GetPeerVuMeter(peer);
    }

    public boolean createPeerConnection(final String peer, final boolean master) throws MediaEngineException {
        return CreatePeerConnection(peer, master);
    }

    public boolean createAnswer(@NonNull String peer) {
        return CreateAnswer(peer);
    }

    public boolean setPeerDescription(@NonNull String peer,
                                      long ssrc,
                                      @NonNull String type,
                                      @NonNull String sdp) {
        return SetPeerDescription(peer, ssrc, type, sdp);
    }

    public boolean setPeerCandidate(@NonNull String peer,
                                    @NonNull String sdpMid,
                                    int sdpMLineIndex,
                                    @NonNull String sdp) {
        return SetPeerCandidate(peer, sdpMid, sdpMLineIndex, sdp);
    }

    //-------------------------------------------------------------------------------------------------
    // Callbacks
    //-------------------------------------------------------------------------------------------------

    public abstract void onSessionCreated(String peer, String type, String sdp);

    public abstract void onIceCandidateDiscovered(String peer, String sdpMid, int sdpMLineIndex, String sdp);

    public abstract void onIceGatheringComplete(String peer);

    public void onStreamAdded(String peer, long nativeStream) {
        Log.d(TAG, "Java stream added");
        MediaStream stream = new MediaStream(peer, nativeStream);
        mStreamListener.onStreamAdded(peer, stream);
    }

    public void onStreamUpdated(String peer, long nativeStream) {
        Log.d(TAG, "Java stream updated");
        MediaStream stream = new MediaStream(peer, nativeStream);
        mStreamListener.onStreamUpdated(peer, stream);
    }

    public void onStreamRemoved(String peer, long nativeStream) {
        Log.d(TAG, "Java stream removed");

        MediaStream stream = new MediaStream(peer, nativeStream);
        mStreamListener.onStreamRemoved(peer);
    }

    public void onScreenStreamAdded(String peer, long nativeStream) {
        Log.d(TAG, "Java stream added corresponding to screen");

        MediaStream stream = new MediaStream(peer, nativeStream);
        mStreamListener.onScreenStreamAdded(peer, stream);
    }

    public void onScreenStreamRemoved(String peer, long nativeStream) {
        Log.d(TAG, "Java stream removed");

        MediaStream stream = new MediaStream(peer, nativeStream);
        mStreamListener.onScreenStreamRemoved(peer);
    }

    //-------------------------------------------------------------------------------------------------
    // JNI Native methods
    //-------------------------------------------------------------------------------------------------

    //public static native void Register(Context context);
    //public static native void UnRegister();

    private native boolean Init(Context context, String peer, EglBase.Context eglContext, VideoCapturer capturer);

    private native void Release();

    private native void StartVideo(long nativeSource);
    private native void StopVideo();

    private native void StartScreenShare(long nativeSource);
    private native void StopScreenShare();

    private native long CreateVideoSource(SurfaceTextureHelper helper, boolean isScreencast);

    private native void AttachMediaStream(String peer, String stream, long nativeRenderer);

    private native void UnAttachMediaStream(String peer, String stream, long nativeRenderer);

    private native boolean CreatePeerConnection(String peer, boolean master);
    private native boolean ClosePeerConnection(String peer);
    private native boolean CreateAnswer(String peer);
    private native boolean SetPeerDescription(String peer, long ssrc, String type, String sdp);
    private native boolean SetPeerCandidate(String peer, String sdpMid, int sdpMLineIndex, String sdp);
    private native boolean SetPeerPosition(String peer, double angle, double position);

    private native void SetMute(boolean mute);

    private native int GetLocalVuMeter();
    private native int GetPeerVuMeter(String peer);
}
