package com.voxeet.android.media;

import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothHeadset;
import android.bluetooth.BluetoothProfile;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.media.AudioManager;
import android.os.Build;
import android.os.Handler;
import android.os.HandlerThread;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.android.media.peer.PendingPeerCallback;
import com.voxeet.android.media.peer.PendingPeerOperation;
import com.voxeet.android.media.peer.SdpCandidate;
import com.voxeet.android.media.peer.SdpCandidates;
import com.voxeet.android.media.peer.SdpDescription;
import com.voxeet.android.media.peer.SdpMessage;
import com.voxeet.android.media.peer.Type;
import com.voxeet.android.media.settings.AudioSettings;
import com.voxeet.android.media.video.CameraEnumerationAndroid;
import com.voxeet.android.media.video.EglBase;
import com.voxeet.android.media.video.EglBase14;
import com.voxeet.android.media.video.VideoCapturerAndroid;

import org.webrtc.voiceengine.WebRtcAudioUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * Wrapper around CoreMedia C++
 *
 * @author Thomas Gourgues
 */
public class Media {
    public static final String TAG = Media.class.getSimpleName();

    public static final String MEDIA_OUPUT_ROUTE_INTENT = "media-output-route-intent";

    private static AtomicBoolean registered = new AtomicBoolean(false);
    private Handler mHandler;
    private HandlerThread mHandlerThread;
    private VideoCapturerAndroid.CameraSwitchHandler mDefaultCameraSwitchHandler = new VideoCapturerAndroid.CameraSwitchHandler() {
        @Override
        public void onCameraSwitchDone(boolean isFrontCamera) {
            log("onCameraSwitchDone: " + isFrontCamera);
        }

        @Override
        public void onCameraSwitchError(String errorDescription) {
            log("onCameraSwitchError: " + errorDescription);
        }
    };

    private com.voxeet.android.media.video.VideoCapturerAndroid.CameraEventsHandler mDefaultCameraHandler = new VideoCapturerAndroid.CameraEventsHandler() {
        @Override
        public void onCameraError(String errorDescription) {
            log("onCameraError: " + errorDescription);
        }

        @Override
        public void onCameraFreezed(String errorDescription) {
            log("onCameraFreezed: " + errorDescription);
        }

        @Override
        public void onCameraOpening(int cameraId) {
            log("onCameraOpening: " + cameraId);
        }

        @Override
        public void onFirstFrameAvailable() {
            log("onFirstFrameAvailable: ");
        }

        @Override
        public void onCameraClosed() {
            log("onCameraClosed: ");
        }
    };
    private HashMap<String, MediaStream> streams;
    private boolean mStopped = false;

    public enum AudioRoute {
        ROUTE_HEADSET(0),
        ROUTE_PHONE(1),
        ROUTE_SPEAKER(2),
        ROUTE_BLUETOOTH(3);

        private int value;

        private AudioRoute(int value) {
            this.value = value;
        }

        public int value() {
            return this.value;
        }

        public static AudioRoute valueOf(int value) {
            switch (value) {
                case 0:
                    return ROUTE_HEADSET;
                case 1:
                    return ROUTE_PHONE;
                case 2:
                    return ROUTE_SPEAKER;
                case 3:
                    return ROUTE_BLUETOOTH;
                default:
                    return ROUTE_SPEAKER;
            }
        }
    }

    private Object operationLock = new Object();

    private Object candidateLock = new Object();

    private static Context context;

    private AudioManager audioManager;

    private String peer;

    private AudioSettings settings;

    private SdpCandidates peerCandidates;

    private List<PendingPeerOperation> peerOperations;

    private static long sDescriptionTimeout = 2000;

    //private long candidatesTimeout = 5000;

    private MediaPowerManager powerManager;

    private MediaStateListener mediaStateListener;

    private MediaStreamListener mediaStreamListener;

    private HeadsetStateReceiver headsetStateReceiver;

    private BluetoothHeadsetListener bluetoothHeadsetListener;

    private BluetoothAdapter bluetoothAdapter;

    private BluetoothHeadset bluetoothHeadset;

    private AudioRoute outputRoute;

    private List<VideoRenderer> renderers;
    private VideoRenderer screenShareVideoRenderer = null;

    private boolean bluetoothScoStarted;

    private boolean isMuted = false;

    private boolean video = false;
    private boolean microphone = false;

    // Camera variables
    private EglBase eglBase = EglBase14.create();
    private VideoCapturerAndroid videoCapturer = null;

    public interface MediaStateListener {
        void OnSpeakerChanged(boolean isEnabled);

        void onHeadsetStateChange(boolean isPlugged);

        void onBluetoothHeadsetStateChange(boolean isPlugged);
    }

    public interface MediaStreamListener {
        void onStreamAdded(String peer, MediaStream stream);

        void onStreamUpdated(String peer, MediaStream stream);

        void onStreamRemoved(String peer);

        void onScreenStreamAdded(String peer, MediaStream stream);

        void onScreenStreamRemoved(String peer);
    }

    private class HeadsetStateReceiver extends BroadcastReceiver {
        @Override
        public void onReceive(Context context, Intent intent) {
            if (Intent.ACTION_HEADSET_PLUG.equals(intent.getAction())) {
                int state = intent.getIntExtra("state", -1);
                switch (state) {
                    case 0:
                        if (mediaStateListener != null) {
                            mediaStateListener.onHeadsetStateChange(false);
                        }
                        setSpeakerMode(true);
                        break;
                    case 1:
                        if (mediaStateListener != null) {
                            mediaStateListener.onHeadsetStateChange(true);
                        }
                        setSpeakerMode(false);
                        break;
                    default:
                        break;
                }
            }
        }
    }

    private class BluetoothStateReceiver extends BroadcastReceiver {
        @Override
        public void onReceive(Context context, Intent intent) {
            if (BluetoothHeadset.ACTION_CONNECTION_STATE_CHANGED.equals(intent.getAction()) && mediaStateListener != null) {
                int state = intent.getIntExtra(BluetoothHeadset.EXTRA_STATE, -1);
                switch (state) {
                    case 0:
                        mediaStateListener.onBluetoothHeadsetStateChange(false);
                        break;
                    case 2:
                        mediaStateListener.onBluetoothHeadsetStateChange(true);
                        break;
                    default:
                        break;
                }
            }
        }
    }

    private class BluetoothHeadsetListener implements BluetoothHeadset.ServiceListener {
        @Override
        public void onServiceConnected(int i, BluetoothProfile bluetoothProfile) {
            bluetoothHeadset = (BluetoothHeadset) bluetoothProfile;
            if (mediaStateListener != null) {
                mediaStateListener.onBluetoothHeadsetStateChange(true);
            }

            //checkProximitySensor(); // Optional here
        }

        @Override
        public void onServiceDisconnected(int i) {
            bluetoothHeadset = null;
            if (mediaStateListener != null) {
                mediaStateListener.onBluetoothHeadsetStateChange(false);
                audioManager.stopBluetoothSco();
                bluetoothScoStarted = false;
            }

            checkOutputRoute();
//            checkProximitySensor();
        }
    }

    static {
        /** Loading dynamic jni lib*/
        try {
            System.loadLibrary("media-jni");
        } catch (Exception e) {
            Log.e(TAG, e.getMessage());
        } catch (UnsatisfiedLinkError e) {
            log("unsupported device architecture");
            Log.e(TAG, e.getMessage());
        }
    }

    public static void setDescriptionTimeout(long descriptionTimeout) {
        Media.sDescriptionTimeout = descriptionTimeout;
    }

    public static Context context() {
        return Media.context;
    }

    public Media(Context context) {
        Media.context = context;
    }

    public Media(final String peer, Context context, boolean video, boolean microphone, final AudioSettings settings) throws MediaException {
        this.streams = new HashMap<>();
        this.peer = peer;
        this.video = video;
        this.microphone = microphone;

        this.settings = settings;
        this.mediaStateListener = null;
        this.mediaStreamListener = null;

        Media.context = context;

        powerManager = new MediaPowerManager(context, this);
        headsetStateReceiver = new HeadsetStateReceiver();

        bluetoothHeadsetListener = new BluetoothHeadsetListener();
        bluetoothAdapter = BluetoothAdapter.getDefaultAdapter();
        bluetoothHeadset = null;
        bluetoothScoStarted = false;

        powerManager.acquire();

        this.peerOperations = new ArrayList<>();
        this.peerCandidates = new SdpCandidates();
        this.renderers = new ArrayList<>();

        this.audioManager = (AudioManager) context.getSystemService(Context.AUDIO_SERVICE);

        context.registerReceiver(headsetStateReceiver, new IntentFilter(Intent.ACTION_HEADSET_PLUG));

        mHandlerThread = new HandlerThread(getClass().getSimpleName());
        mHandlerThread.start();
        mHandler = new Handler(mHandlerThread.getLooper());

        //this.audioManager.setBluetoothScoOn(true);

        bluetoothAdapter.getProfileProxy(context, bluetoothHeadsetListener, BluetoothProfile.HEADSET);

        //int result = audioManager.requestAudioFocus(new AudioManager.OnAudioFocusChangeListener() {
        //    @Override
        //    public void onAudioFocusChange(int i) {
        //        Log.e("Media.java", "Audio focus change: " + i);
        //    }
        //}, WebRtcAudioManager.IsAndroidStreamVoice() ? AudioManager.STREAM_VOICE_CALL : AudioManager.STREAM_MUSIC,  AudioManager.AUDIOFOCUS_GAIN);
        //

        //Log.e("Media.java", "Request audio focus result code: " + result);

        // Delegates init
        // ...

        WebRtcAudioUtils.setDefaultSampleRateHz(16000);

        if (video) {
            videoCapturer = VideoCapturerAndroid.create(CameraEnumerationAndroid.getNameOfFrontFacingDevice(), new VideoCapturerAndroid.CameraEventsHandler() {
                @Override
                public void onCameraError(String errorDescription) {
                    log("onCameraError: " + errorDescription);
                }

                @Override
                public void onCameraFreezed(String errorDescription) {
                    log("onCameraFreezed: " + errorDescription);
                }

                @Override
                public void onCameraOpening(int cameraId) {
                    log("onCameraOpening: " + cameraId);
                }

                @Override
                public void onFirstFrameAvailable() {
                    log("onFirstFrameAvailable: ");
                }

                @Override
                public void onCameraClosed() {
                    log("onCameraClosed: ");
                }
            });
        }

        if (!Init(peer, eglBase.getEglBaseContext(), videoCapturer, settings)) {
            throw new MediaException("An error occurred during init");
        }

        if (audioManager.isWiredHeadsetOn()) {
            setSpeakerMode(false);
        } else {
            setSpeakerMode(true);
        }

        //this.audioManager.setSpeakerphoneOn(false);		// Setting device

        // ... (useless on Android for the moment)
    }

    public void stop() {
        mStopped = true;
        setBluetooth(false);

        context.unregisterReceiver(headsetStateReceiver);

        if (bluetoothHeadset != null) {
            bluetoothAdapter.closeProfileProxy(BluetoothProfile.HEADSET, bluetoothHeadset);
        }

        powerManager.release();
//        powerManager.releaseProximity();

        if (videoCapturer != null) {
            try {
                videoCapturer.stopCapture();
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
            videoCapturer.dispose();
            videoCapturer = null;
        }

        Release();

        this.audioManager.setSpeakerphoneOn(true);
        this.audioManager.setMode(AudioManager.MODE_NORMAL);

        try {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2) {
                mHandlerThread.quitSafely();
            } else {
                mHandlerThread.quit();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void startVideo() {
        startVideo(CameraEnumerationAndroid.getNameOfFrontFacingDevice());
    }

    public void startVideo(String cameraName) {
        if (videoCapturer == null) {
            videoCapturer = VideoCapturerAndroid.create(cameraName, mDefaultCameraHandler);
            StartVideo(eglBase.getEglBaseContext(), videoCapturer);
        }
    }

    public void stopVideo() {
        if (videoCapturer != null) {
            StopVideo();
            videoCapturer = null;
        }
    }

    public void switchCamera(VideoCapturerAndroid.CameraSwitchHandler handler) {
        if (videoCapturer != null) {
            videoCapturer.switchCamera(handler);
        }
    }

    public void registerListener(MediaStateListener listener) {
        this.mediaStateListener = listener;
    }

    public void unregisterListener() { // Only one at a time
        this.mediaStateListener = null;
    }

    public void setMediaStreamListener(MediaStreamListener listener) {
        this.mediaStreamListener = listener;
    }

    public AudioRoute outputRoute() {
        return this.outputRoute;
    }

    public boolean isBluetoothHeadsetOn() {
        return bluetoothHeadset != null;
    }

    public boolean isWiredHeadsetOn() {
        return audioManager.isWiredHeadsetOn();
    }

    public void setBluetooth(boolean isEnabled) {
        try {
            if (isEnabled) {
                if (!bluetoothScoStarted) {
                    audioManager.startBluetoothSco();
                    bluetoothScoStarted = true;
                }
            } else {
                if (bluetoothScoStarted) {
                    audioManager.stopBluetoothSco();
                    bluetoothScoStarted = false;
                }
            }
        } catch (NullPointerException e) { // Workaround for lollipop 5.0
            log("No bluetooth headset connected");
        }

        checkOutputRoute();
//        checkProximitySensor();
    }

    public void setSpeakerMode(boolean speakerMode) {
        setBluetooth(false); // Ensure that bluetooth is off
        //ResetAudioDevice();
        if ((android.os.Build.BRAND.equals("Samsung") || android.os.Build.BRAND.equals("samsung"))) {
            int requiredSpeakerMode = AudioManager.MODE_IN_COMMUNICATION;

            if (speakerMode) {
                // route audio to back speaker
                audioManager.setSpeakerphoneOn(true);
                audioManager.setMode(AudioManager.MODE_CURRENT);
            } else {
                // route audio to earpiece
                this.audioManager.setSpeakerphoneOn(speakerMode);
                if (audioManager.isWiredHeadsetOn()) {
                    audioManager.setMode(AudioManager.MODE_CURRENT);
                } else {
                    this.audioManager.setMode(requiredSpeakerMode);
                }
            }
        } else {
            // Non-Samsung devices
            this.audioManager.setSpeakerphoneOn(speakerMode);
        }

        if (mediaStateListener != null) {
            mediaStateListener.OnSpeakerChanged(speakerMode);
        }

        checkOutputRoute();

//        checkProximitySensor();
    }

//    protected void checkProximitySensor() {
//        if (outputRoute == AudioRoute.ROUTE_PHONE) {
//            powerManager.acquireProximity();
//        } else {
//            powerManager.releaseProximity();
//        }
//    }

    protected void checkOutputRoute() {
        if (bluetoothScoStarted) {
            outputRoute = AudioRoute.ROUTE_BLUETOOTH;
        } else if (audioManager.isSpeakerphoneOn()) {
            outputRoute = AudioRoute.ROUTE_SPEAKER;
        } else {
            if (audioManager.isWiredHeadsetOn()) {
                outputRoute = AudioRoute.ROUTE_HEADSET;
            } else {
                outputRoute = AudioRoute.ROUTE_PHONE;
            }
        }

        Intent intent = new Intent(MEDIA_OUPUT_ROUTE_INTENT);
        intent.putExtra("route", outputRoute.value());

        context.sendBroadcast(intent);
    }

    public void setOutputRoute(AudioRoute route) {
        switch (route) {
            case ROUTE_HEADSET:
            case ROUTE_PHONE:
                setSpeakerMode(false);
                break;
            case ROUTE_SPEAKER:
                setSpeakerMode(true);
                break;
            case ROUTE_BLUETOOTH:
                setBluetooth(true);
                break;
            default:
                break;
        }
    }

    public HashMap<String, MediaStream> getMapOfStreams() {
        log("getMapOfStreams");
        HashMap<String, MediaStream> streams_copy = new HashMap<>();
        synchronized (streams) {
            for (String key : streams.keySet()) {
                streams_copy.put(key, streams.get(key));
            }
        }
        return streams_copy;
    }

    public List<AudioRoute> availableRoutes() {
        List<AudioRoute> routes = new ArrayList<>();

        routes.add(AudioRoute.ROUTE_SPEAKER);

        if (isWiredHeadsetOn()) {
            routes.add(AudioRoute.ROUTE_HEADSET);
        } else {
            routes.add(AudioRoute.ROUTE_PHONE);
        }

        if (isBluetoothHeadsetOn()) {
            routes.add(AudioRoute.ROUTE_BLUETOOTH);
        }

        return routes;
    }

    public int calibrate(String sound) {
        int delay = Calibrate(context, sound);
        return delay;
    }

    public void attachMediaStream(VideoRenderer.Callbacks callbacks, MediaStream stream) {
        log("attachMediaStream " + callbacks + " " + stream);
        VideoRenderer renderer = null;
        for (VideoRenderer r : renderers) {
            if (r.callbacks == callbacks) {
                renderer = r;
                break;
            }
        }

        if (renderer == null) {
            renderer = new VideoRenderer(callbacks);
            renderers.add(renderer);
        }

        if (!isStopped()) {
            AttachMediaStream(stream.nativeStream, renderer.nativeVideoRenderer);
        }
    }

    public void unattachMediaStream(VideoRenderer.Callbacks callbacks, MediaStream stream) {
        log("unattachMediaStream " + callbacks + " " + stream);
        VideoRenderer renderer = null;
        for (VideoRenderer r : renderers) {
            if (r.callbacks == callbacks) {
                renderer = r;
                break;
            }
        }

        if (renderer != null && !isStopped()) {
            if (stream != null) {
                UnAttachMediaStream(stream.nativeStream, renderer.nativeVideoRenderer);
            }
        }
    }


    public void createOfferForPeer(final String peer,
                                   final boolean master,
                                   final PendingPeerCallback listener) {
        log("createOfferForPeer " + peer + " " + master + " " + listener);
        if (isStopped()) {
            log("stopped");
            return;
        }

        mHandler.post(new Runnable() {
            @Override
            public void run() {
                try {
                    if (isStopped()) return;

                    createConnection(peer, master);

                    if (!isStopped() && !CreateOffer(peer)) {
                        throw new MediaException(String.format("Unable to create offer for Peer: %s", peer));
                    }

                    if (!isStopped()) {
                        WaitForSdpMessage(peer, Type.OFFER, listener);
                    }
                } catch (MediaException e) {
                    e.printStackTrace();
                }
            }
        });
    }


    public void createAnswerForPeer(final String peer,
                                    final long ssrc,
                                    final SdpDescription offer,
                                    final List<SdpCandidate> offerCandidates,
                                    final boolean isMaster,
                                    final PendingPeerCallback listener) {
        log("createAnswerForPeer " + peer + " " + ssrc + " " + offer.getSdp() + " " + offer.getSsrc() + " " + offer.getSdp() + " " + offerCandidates + " " + isMaster + " " + listener);
        Log.d("SDKMEDIA", "createAnswerForPeer " + peer + " " + ssrc + " " + offer.getSdp() + " " + offer.getSsrc() + " " + offer.getSdp() + " " + offerCandidates + " " + isMaster + " " + listener);
        log("Sending async...");

        if (isStopped()) {
            log("stopped");
            return;
        }

        mHandler.post(new Runnable() {
            @Override
            public void run() {
                try {
                    Timer timer = new Timer("SDKMEDIA", "createConnection>>");
                    createConnection(peer, isMaster);
                    timer.stop();

                    timer = new Timer("SDKMEDIA", "SetPeerDescription>> " + peer + " " + ssrc + " " + offer.getType() + " " + offer.getSdp() + " " + offer.getSsrc());
                    if (isStopped() || !SetPeerDescription(peer, ssrc, offer.getType(), offer.getSdp())) {
                        throw new MediaException(String.format("Unable set remote SDP for Peer: %s", peer));
                    }
                    timer.stop();

                    //SetPeerDescription(peer, ssrc, offer.type(), offer.sdp());

                    timer = new Timer("SDKMEDIA", "SetPeerCandidate>>");
                    for (SdpCandidate candidate : offerCandidates) {
                        final SdpCandidate c = candidate;
                        SetPeerCandidate(peer, c.getSdpMid(), c.getSdpMLineIndex(), c.getSdp());
                    }
                    timer.stop();

                    //create answer async
                    mHandler.post(new Runnable() {
                        @Override
                        public void run() {
                            if (!isStopped() && CreateAnswer(peer)) {
                                WaitForSdpMessage(peer, Type.ANSWER, listener);
                            } else {
                                try {
                                    throw new MediaException(String.format("Unable to create answer for Peer: %s", peer));

                                } catch (MediaException e) {
                                    e.printStackTrace();
                                }
                            }
                        }
                    });
                } catch (MediaException e) {
                    e.printStackTrace();
                }
            }
        });
    }


    public void addPeerFromAnswer(final String peer, final long ssrc, final SdpDescription answer, final List<SdpCandidate> candidates) throws MediaException {
        if (isStopped()) {
            log("stopped");
            return;
        }

        mHandler.post(new Runnable() {
            @Override
            public void run() {
                log("addPeerFromAnswer " + peer + " " + ssrc + " " + answer + " " + candidates);
                if (isStopped() || !SetPeerDescription(peer, ssrc, answer.getType(), answer.getSdp())) {
                    return;
                }

                mHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        if (isStopped()) {
                            log("stopped");
                            return;
                        }

                        for (SdpCandidate candidate : candidates) {
                            SetPeerCandidate(peer, candidate.getSdpMid(), candidate.getSdpMLineIndex(), candidate.getSdp());
                        }
                    }
                });
            }
        });
    }

    public void removePeer(final String peer) {
        if (isStopped() || !ClosePeerConnection(peer)) {
            Log.w(TAG, String.format("Unable to close peer connection for peer: %s", peer));
        }
    }

    public void changePeerPosition(String peer, double angle, double distance) {
        if (isStopped() || !SetPeerPosition(peer, angle, distance)) {
            Log.w(TAG, String.format("Unable to change position for peer: %s", peer));
        }
    }

    public void changePeerPosition(String peer, double angle, double distance, float gain) {
        if (isStopped() || !SetPeerPositionGain(peer, angle, distance, gain)) {
            Log.w(TAG, String.format("Unable to change position for peer: %s", peer));
        }
    }

    public void changePeerGain(String peer, float gain) {
        if (isStopped() || SetPeerGain(peer, gain)) {
            Log.w(TAG, String.format("Unable to change gain for peer: {0}", peer));
        }
    }

    public void muteRecording() {
        isMuted = true;
        SetMute(true);
    }

    public void unMuteRecording() {
        isMuted = false;
        SetMute(false);
    }

    public boolean isMuted() {
        return isMuted;
    }

    public int getLocalVuMeter() {
        return GetLocalVuMeter();
    }

    public int getPeerVuMeter(String peer) {
        return GetPeerVuMeter(peer);
    }

    public boolean setCodecQuality(int quality) {
        return SetCodecQuality(quality);
    }

    public boolean pstnNeeded() {
        return PstnNeeded();
    }

    public boolean resetAudioDevice() {
        return ResetAudioDevice();
    }

    //-------------------------------------------------------------------------------------------------
    // Callbacks
    //-------------------------------------------------------------------------------------------------


    public void onSessionCreated(String peer, String type, String sdp) {
        log("onSessionCreated " + peer + " " + type + " " + sdp);
        Type operationType;

        if (type.equals("offer")) {
            operationType = Type.OFFER;
        } else if (type.equals("answer")) {
            operationType = Type.ANSWER;
        } else {
            Log.w(TAG, String.format("Session description received from an unknown type: %s", type));
            return;
        }

        unlockPeerOperation(operationType, peer, new SdpDescription(type, sdp));
    }


    public void onIceCandidateDiscovered(String peer, String sdpMid, int sdpMLineIndex, String sdp) {
        log(String.format("onIceCandidateDiscovered ICE candidate discovered for peer: %s with id: %s and SDP: %s", peer, sdpMid, sdp));

        synchronized (candidateLock) {
            peerCandidates.add(peer, new SdpCandidate(sdpMid, sdpMLineIndex, sdp));
        }

        onIceComplete(peer);
    }


    public void onIceGatheringComplete(final String peer) {
        log(String.format("onIceGatheringComplete ICE candidate gathering complete for peer: %s", peer));

        onIceComplete(peer);
    }

    private void onIceComplete(final String peer) {
        if (isStopped()) {
            log("stopped");
            return;
        }

        mHandler.post(new Runnable() {
            @Override
            public void run() {
                List<SdpCandidate> candidates;

                synchronized (candidateLock) {
                    candidates = peerCandidates.candidates(peer);
                }

                if (candidates.isEmpty()) {
                    Log.w(TAG, String.format("No ICE candidate gathered for peer: %s", peer));
                } else {
                    for (PendingPeerOperation operation : peerOperations) {
                        if (operation.getPeer().equals(peer)) {
                            Type type = operation.get_type();
                            SdpCandidate candidate = candidates.get(0);

                            onSessionCreated(peer, type.toString(), candidate.getSdp());
                            return;
                        }
                    }
                }
            }
        });
    }

    public void onStreamAdded(String peer, MediaStream stream) {
        log("Java stream added " + peer + " " + stream);
        if (this.mediaStreamListener != null) {
            this.mediaStreamListener.onStreamAdded(peer, stream);
        }
    }


    public void onStreamUpdated(@NonNull String peer, @NonNull MediaStream stream) {
        log("Java stream updated " + peer + " " + stream);
        if (this.mediaStreamListener != null) {
            this.mediaStreamListener.onStreamUpdated(peer, stream);
        }
    }


    public void onStreamRemoved(@NonNull String peer, @NonNull MediaStream stream) {
        log("Java stream removed " + peer + " " + stream);

        if (this.mediaStreamListener != null) {
            this.mediaStreamListener.onStreamRemoved(peer);
        }
    }


    public void onScreenStreamAdded(@NonNull String peer, @NonNull MediaStream stream) {
        log("onScreenStreamAdded " + peer + " " + stream);
        log("Java screen stream added");
        stream.setIsScreenShare(true);

        if (this.mediaStreamListener != null) {
            this.mediaStreamListener.onScreenStreamAdded(peer, stream);
        }
    }

    public void onScreenStreamRemoved(@NonNull String peer, @NonNull MediaStream stream) {
        log("onScreenStreamRemoved " + peer + " " + stream);
        log("Java stream removed");

        if (this.mediaStreamListener != null) {
            this.mediaStreamListener.onScreenStreamRemoved(peer);
        }
    }

    //-------------------------------------------------------------------------------------------------
    // Helper methods
    //-------------------------------------------------------------------------------------------------

    private void createConnection(final @NonNull String peer, final boolean master) throws MediaException {
        log("createConnection " + peer + " " + master);
        if (isStopped() || !CreatePeerConnection(peer, master)) {
            throw new MediaException(String.format("Unable to create connection with Peer: %s", peer));
        }
    }

    @Nullable
    private PendingPeerOperation createPeerOperation(@NonNull Type type,
                                                     @NonNull String peer,
                                                     @NonNull PendingPeerCallback listener) {
        log("createPeerOperation " + type + " " + peer + " " + listener);
        PendingPeerOperation operation = new PendingPeerOperation(type, peer, listener);

        synchronized (operationLock) {
            if (peerOperations.contains(operation)) {
                Log.e(TAG, String.format("An operation of type: %s is already pending for peer: %s", type.name(), peer));
                return null;
            }
        }

        peerOperations.add(operation);

        return operation;
    }

    private void unlockPeerOperation(@NonNull Type type, @NonNull String peer, @NonNull SdpDescription value) {
        log("unlockPeerOperation " + type + " " + peer + " " + value);
        synchronized (operationLock) {
            boolean isFound = false;

            for (PendingPeerOperation operation : peerOperations) {
                if (operation.tryUnlock(type, peer, value)) {
                    isFound = true;
                    peerOperations.remove(operation);
                    break;
                }
            }

            if (!isFound) {
                Log.w(TAG, String.format("No pending operation found for peer: %s with type: %s", peer, type.name()));
            }
        }
    }

    private void WaitForSdpMessage(final @NonNull String peer, final @NonNull Type type, final @NonNull PendingPeerCallback listener) {
        log("Sending async...");

        if (isStopped()) {
            log("stopped");
            return;
        }

        mHandler.post(new Runnable() {
            @Override
            public void run() {
                log("waitForSdpMessage " + peer + " " + type + " " + listener);
                PendingPeerOperation descriptionOperation = createPeerOperation(type,
                        peer,
                        new PendingPeerCallback() {
                            @Override
                            public void onMessage(SdpMessage message) {

                                if (isStopped()) {
                                    log("stopped");
                                    return;
                                }

                                listener.onMessage(message);

                                log("WaitForSdpMessage: answer obtained after unlock ");
                            }
                        });
                if (descriptionOperation != null) {
                    descriptionOperation.waitOperation(sDescriptionTimeout);
                }
            }
        });
    }


    public void MuteConference(boolean shouldMute) {
        if (isMuted != shouldMute) {
            SetMute(isMuted = shouldMute);
        }
    }

    public static void Register(Context context) {
        if (!registered.get()) {
            register(context);
        }
        registered.set(true);
    }

    public static void UnRegister() {
        if (registered.get()) {
            unRegister();
        }
        registered.set(false);
    }

    //-------------------------------------------------------------------------------------------------
    // JNI Native methods
    //-------------------------------------------------------------------------------------------------

    public static native void register(Context context);

    public static native void unRegister();

    private native boolean Init(String peer, EglBase.Context eglContext, VideoCapturer capturer, AudioSettings settings);

    private native void Release();

    private native void StartVideo(EglBase.Context eglContext, VideoCapturer capturer);

    private native void StopVideo();

    private native void AttachMediaStream(long stream, long nativeRenderer);

    private native void UnAttachMediaStream(long stream, long nativeRenderer);

    private native boolean CreatePeerConnection(String peer, boolean master);

    private native boolean ClosePeerConnection(String peer);

    private native boolean CreateOffer(String peer);

    private native boolean CreateAnswer(String peer);

    private native boolean SetPeerDescription(String peer, long ssrc, String type, String sdp);

    private native boolean SetPeerCandidate(String peer, String sdpMid, int sdpMLineIndex, String sdp);

    private native boolean SetPeerPosition(String peer, double angle, double position);

    private native boolean SetPeerPositionGain(String peer, double angle, double position, float gain);

    private native boolean SetPeerGain(String peer, float gain);

    private native void SetMute(boolean mute);

    private native void SetAudioOptions(boolean ns, boolean agc, boolean ec, boolean typingDetection);

    private native int GetLocalVuMeter();

    private native int GetPeerVuMeter(String peer);

    private native boolean SetCodecQuality(int quality);

    private native boolean SetOpusCustomProperties(int type, int value);

    private native boolean DecreasePeersQuality();

    private native boolean PstnNeeded();

    public native int Calibrate(Context context, String sound);

    public native boolean ResetAudioDevice();

    private static void log(String value) {
        Log.d(TAG, "log: " + value);
    }

    private boolean isStopped() {
        return mStopped;
    }
}
