package com.voxeet.android.media;

import android.content.Context;
import android.hardware.Sensor;
import android.hardware.SensorEvent;
import android.hardware.SensorEventListener;
import android.hardware.SensorManager;
import android.net.wifi.WifiManager;
import android.net.wifi.WifiManager.WifiLock;
import android.os.Build;
import android.os.PowerManager;
import android.os.PowerManager.WakeLock;
import android.util.Log;

/**
 * Class that manage the difference WakeLock required for the conference.
 *
 * @author Thomas Gourgues
 */
public class MediaPowerManager implements SensorEventListener {
    private static final String TAG = MediaPowerManager.class.getSimpleName();

    private final int PROXIMITY_SCREEN_OFF_WAKE_LOCK = 32;

    private Context mContext;

    private PowerManager mPowerManager;

    private WifiManager mWifiManager;

    private SensorManager mSensorManager;

    private WakeLock mProximityWakeLock;

    private WakeLock mPartialWakeLock;

    private WifiLock mWifiLock;

    private Media mMedia;

    private Sensor mProximity;

    private boolean shouldRegisterForProximity = false;

    public MediaPowerManager(Context context, Media media) {
        mContext = context;

        mMedia = media;

        mPowerManager = (PowerManager) mContext.getSystemService(Context.POWER_SERVICE);

        mWifiManager = (WifiManager) mContext.getSystemService(Context.WIFI_SERVICE);

        mSensorManager = (SensorManager) mContext.getSystemService(Context.SENSOR_SERVICE);

        mProximity = mSensorManager.getDefaultSensor(Sensor.TYPE_PROXIMITY);

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP
                || (Build.VERSION.SDK_INT < Build.VERSION_CODES.LOLLIPOP && !Build.BRAND.contains("samsung")))
            shouldRegisterForProximity = true;

        try {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP)
                mProximityWakeLock = mPowerManager.newWakeLock(PowerManager.PROXIMITY_SCREEN_OFF_WAKE_LOCK, TAG);
            else
                mProximityWakeLock = mPowerManager.newWakeLock(PROXIMITY_SCREEN_OFF_WAKE_LOCK, TAG);

            mPartialWakeLock = mPowerManager.newWakeLock(PowerManager.PARTIAL_WAKE_LOCK, TAG);

            mWifiLock = mWifiManager.createWifiLock(WifiManager.WIFI_MODE_FULL_HIGH_PERF, TAG);
        } catch (Exception e) {
            Log.e(TAG, "Failed to create required WakeLock");
        }
    }

    public void acquire() {
        if (shouldRegisterForProximity)
            mSensorManager.registerListener(this, mProximity, SensorManager.SENSOR_DELAY_NORMAL);

        if (!mPartialWakeLock.isHeld())
            mPartialWakeLock.acquire();

        if (!mWifiLock.isHeld())
            mWifiLock.acquire();
    }

    public void release() {
        if (shouldRegisterForProximity)
            mSensorManager.unregisterListener(this);

        if (mPartialWakeLock.isHeld())
            mPartialWakeLock.release();

        if (mWifiLock.isHeld())
            mWifiLock.release();

        if (mProximityWakeLock.isHeld())
            mProximityWakeLock.release();
    }

    @Override
    public final void onAccuracyChanged(Sensor sensor, int accuracy) {
        // Do something here if sensor accuracy changes.
    }

    @Override
    public final void onSensorChanged(SensorEvent event) {
        float distance = event.values[0];

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            if (mMedia.outputRoute() == Media.AudioRoute.ROUTE_PHONE && mProximityWakeLock != null && !mProximityWakeLock.isHeld()) { // black off screen if default audio route
                if (distance <= 5) // distance changes from one brand to another (Samsung: 0; Motorola: 3)
                    mProximityWakeLock.acquire();
            } else if (mProximityWakeLock != null && mProximityWakeLock.isHeld()) { // enable screen
                mProximityWakeLock.release(0);
            }
        }
    }
}
