package com.voxeet.log.factory;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.log.endpoints.LogEndpoints;
import com.voxeet.log.models.Disconnect;
import com.voxeet.log.models.Identify;
import com.voxeet.log.models.Connect;

import java.util.UUID;

import okhttp3.ResponseBody;
import retrofit2.Call;
import retrofit2.Callback;
import retrofit2.Response;

public class LogFactory {
    public static final String ENDPOINT = "https://report.voxeet.com";
    @Nullable
    private LogEndpoints logger;

    private LogFactory() {

    }

    public static final LogFactory instance = new LogFactory();

    @NonNull
    private final String sessionId = UUID.randomUUID().toString();

    public void init(@NonNull LogEndpoints logger) {
        if (null == this.logger) {
            this.logger = logger;
        }
    }

    /**
     * Call and consume the log api to send an identify call from the other packages
     *
     * @param madeFromSdk indicate if the call was made from an automatic retrial, or false if from developers
     */
    public void identify(boolean madeFromSdk) {
        if (null == logger) return;
        Call<ResponseBody> body = logger.identify(new Identify(sessionId, madeFromSdk));
        consume(body);
    }

    /**
     * Call and consume the log api to send a login call made from the developer PoV.
     * This call does not check for multiple login called in parallel etc... It is a simple implementation.
     */
    public void connect() {
        if (null == logger) return;
        Call<ResponseBody> body = logger.connect(new Connect(sessionId));
        consume(body);
    }

    /**
     * Call and consume the log api to send a logout call made from the developer PoV.
     * This call does not check for any state and will indicate a call made by the developer, not if it was canceled, already logged out etc...
     */
    public void disconnect() {
        if (null == logger) return;
        Call<ResponseBody> body = logger.disconnect(new Disconnect(sessionId));
        consume(body);
    }

    private <T> void consume(Call<T> call) {
        call.enqueue(new Callback<T>() {
            @Override
            public void onResponse(Call<T> call, Response<T> response) {
                //call done
            }

            @Override
            public void onFailure(Call<T> call, Throwable t) {
                //call failed
            }
        });
    }
}
